/*
 * Copyright 1999-2001 Vizdom Software, Inc. All Rights Reserved.
 */

package com.vizdom.ber;

import java.io.EOFException;
import java.io.InputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.BitSet;
import com.vizdom.util.Debug;

/**
 * A BER BIT STRING.
 *
 * @author: John Lacey
 * @version: $Revision: 1.6 $
 */
public class BerBitString extends BerObject
{
    /** The BIT STRING identifier, [UNIVERSAL 3]. */
    static final BerIdentifier gIDENTIFIER = 
        new BerIdentifier(BerTypes.PRIMITIVE, BerTypes.BIT_STRING);

    /*
     * A bit set may be reasonable represented in three different ways. 
     * We accomodate all three representations. Users may choose between
     * a BitSet and a string of 0's and 1's. The mReadContents method
     * doesn't know which representation the user may want, so it leaves
     * the encoded bits in a byte array.
     */

    /** The underlying byte array of this BER BIT STRING. */
    private byte[] mByteArray;


    /** The number of trailing unused bits in the byte array. */
    private int mUnusedBits;


    /** The underlying BitSet of this BER BIT STRING. */
    private BitSet mBitSet;


    /** The underlying String value of this BER BIT STRING. */
    private String mString;


    /** The decoding constructor. */
    protected BerBitString()
    {
    }


    /** 
     * An encoding constructor for Strings.
     * 
     * @param aString a string
     */
    public BerBitString(String aString)
    {
        mString = aString;
    }


    /** 
     * An encoding constructor for BitSets.
     * 
     * @param aBitSet a bit set
     */
    public BerBitString(BitSet aBitSet)
    {
        mBitSet = aBitSet;
    }


    /**
     * Returns the BER identifier for BER BIT STRING, [UNIVERSAL 3].
     * 
     * @return the BER identifier for BER BIT STRING, [UNIVERSAL 3]
     */
    public BerIdentifier getIdentifier()
    {
        return gIDENTIFIER;
    }


    /**
     * Returns the size of the encoded contents, in bytes.
     * 
     * @return the size of the encoded contents, in bytes
     */
    protected final int mGetLength()
    {
        return mByteArray.length;
    }


    /** 
     * Writes the encoded contents to the output stream.
     * 
     * @param anOut an output stream
     * @exception IOException if an I/O error occurs
     */
    protected final void mWriteContents(OutputStream anOut) throws IOException
    {
        anOut.write(mBitSet);
    }


    /** 
     * Reads the encoded contents from the input stream.
     * 
     * @param anIn an input stream
     * @param aModule a BER module for reading constructed encodings
     * @param anIdentifier the BER identifier of the encoded object
     * @param aLength the length in bytes of the encoded contents
     * @exception IOException if an I/O error occurs. In particular,
     *     an <code>EOFException</code> may be thrown if the end of
     *     stream is reached before the contents have been fully read
     */
    protected void mReadContents(InputStream anIn, BerModule aModule,
        BerIdentifier anIdentifier, int aLength) throws IOException
    {
        if (Debug.ASSERT && getClass() == BerBitString.class)
            Debug.assertTrue(anIdentifier == gIDENTIFIER); // [sic; not equals]

        // Read the first byte. This tells us how many unused bits 
        // are at the end of the encoded bit string.
        int octet = anIn.read();
        if (octet == -1)
            throw new EOFException();

        // Compute the number of bits in the bit set.
        int bitCount = (aLength - 1) * 8 - octet;

        // Read exactly aLength bytes (which may be more than are
        // available all at once). Any EOF is unexpected.
        mByteArray = new byte[aLength];
        int count;
        for (int offset = 0; offset < aLength; offset += count)
        {
            count = anIn.read(mByteArray, offset, aLength - offset);
            if (count == -1)
                throw new EOFException(); // ??? BerException?
        }

        mString = CharacterEncoder.toString(mByteArray, 
            aModule.getCharacterEncoding());
    }


    /**
     * Returns a copy of the bit set held by this object.
     * 
     * @return a copy of the bit set held by this object
     */
    public BitSet toBitSet()
    {
        return (BitSet) mBitSet.clone();
    }


    /**
     * Returns a string representation of the object.
     * 
     * @return a string representation of the object
     */
    public String toString()
    {
        return mString;
    }
}

