use strict;
use warnings;

# NOTE
# Now, this script disables "scoped_lexical_variables" and "file_lexical_variables" tests.
# And this looks on "scoped_lexical_variables", "scoped_lexical_variables" and "local_lexical_variables" are the same.

my $this_program = __FILE__;
(my $test_file_name = $this_program) =~ s/[.]PL\Z/.t/ms;
if ($this_program eq $test_file_name) {
    die "Was not able to figure out the name of the file to generate. This program: $this_program.";
}

print "\n\nGenerating $test_file_name.\n";

open my $test_file, '>', $test_file_name or die "Could not open $test_file_name: $!";

print {$test_file} "# Do not edit!!! This test suite generated by $this_program.\n";
print {$test_file} <<'END_CODE';

use strict;
use warnings;
use Perl::Lint::Policy::NamingConventions::Capitalization;
use t::Policy::Util qw/fetch_violations/;
use Test::Base::Less;

my $class_name = 'NamingConventions::Capitalization';

filters {
    params => [qw/eval/],
};

for my $block (blocks) {
    my $violations = fetch_violations($class_name, $block->input, $block->params);
    is scalar @$violations, $block->failures, $block->dscr;
}

done_testing;

__DATA__

END_CODE

emit_simple_tests($test_file);
emit_primary_tests($test_file);
emit_footer($test_file);

close $test_file;
print "Done.\n\n";

#-----------------------------------------------------------------------------

sub emit_simple_tests {
    my ($test_file) = @_;

    print {$test_file} <<'END_SIMPLE_TESTS';
===
--- dscr: Basic Passes
--- failures: 0
--- params:
--- input
my  $foo;
our $bar;
my($foo, $bar) = ("BLEH", "BLEH");
my @foo;
my %bar;
sub foo {}

my  $foo123;
my  $foo123bar;
sub foo123 {}
sub foo123bar {}

package This::SomeThing;
package This;
package This::Thing;
package Acme::12345;
package YYZ;

===
--- dscr: Basic Failures
--- failures: 14
--- params:
--- input
my  $Foo;
our $Bar;
my  @Foo;
my  %Bar;
sub Foo {}

my  $foo_Bar;
sub foo_Bar {}

my  $FooBar;
sub FooBar {}

my  $foo123Bar;
sub foo123Bar {}

package pragma;
package Foo::baz;
package baz::FooBar;

===
--- dscr: Special case: main
--- failures: 0
--- params:
--- input
package main;

===
--- dscr: Combined passes and fails
--- failures: 2
--- params:
--- input
my($foo, $Bar);
our($Bar, $foo);

===
--- dscr: Variables from other packages should pass
--- failures: 0
--- params:
--- input
local $Other::Package::Foo;
$Other::Package::Foo;

===
--- dscr: Only cares about declarations
--- failures: 0
--- params:
--- input
Foo();
$Foo = 42;

===
--- dscr: Constants must be all caps, passes
--- failures: 0
--- params:
--- input
Readonly::Scalar my $CONSTANT = 23;
const my $CONSTANT = 23;
use constant FOO => 42;
use constant { BAR => 3, BAZ => 7 };
use constant 1.16 FOO => 42;
use constant 1.16 { BAR => 3, BAZ => 7 };

===
--- dscr: Constants must be all caps, failures
--- failures: 3
--- params:
--- input
Readonly::Scalar my $Foo = 23;
Readonly::Scalar my $foo = 23;
const my $fooBAR = 23;

===
--- dscr: PPI misparses part of ternary as a label (RT #41170)
--- failures: 0
--- params:
--- input
my $foo = $condition ? $objection->method : $alternative;
my $foo = $condition ? undef : 1;

END_SIMPLE_TESTS

    return;
}

sub emit_primary_tests {
    my ($test_file) = @_;

    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'all_lower_case', 'guaranteed_to_pass', ':single_case', 0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'ALL_UPPER_CASE', 'guaranteed_to_pass', ':single_case', 0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'mixedCase', 'guaranteed_to_pass', ':single_case', 1,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'a11_lower_case_with_digits', 'guaranteed_to_pass', ':single_case', 0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'A11_UPPER_CASE_WITH_DIGITS', 'guaranteed_to_pass', ':single_case', 0,
    );

    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'all_lower_case', 'guaranteed_to_pass', ':all_lower', 0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'ALL_UPPER_CASE', 'guaranteed_to_pass', ':all_lower', 1,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'mixedCase', 'guaranteed_to_pass', ':all_lower', 1,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'a11_lower_case_with_digits', 'guaranteed_to_pass', ':all_lower', 0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'A11_UPPER_CASE_WITH_DIGITS', 'guaranteed_to_pass', ':all_lower', 1,
    );

    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'all_lower_case', 'GUARANTEED_TO_PASS', ':all_upper', 1,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'ALL_UPPER_CASE', 'GUARANTEED_TO_PASS', ':all_upper', 0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'mixedCase', 'GUARANTEED_TO_PASS', ':all_upper', 1,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'a11_lower_case_with_digits', 'GUARANTEED_TO_PASS', ':all_upper', 1,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'A11_UPPER_CASE_WITH_DIGITS', 'GUARANTEED_TO_PASS', ':all_upper', 0,
    );

    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'all_lower_case', 'guaranteed_to_pass', ':starts_with_lower', 0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'ALL_UPPER_CASE', 'guaranteed_to_pass', ':starts_with_lower', 1,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file,
        '________all_lower_case_with_leading_underscores',
        'guaranteed_to_pass',
        ':starts_with_lower',
        0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'a11_lower_case_with_digits', 'guaranteed_to_pass', ':starts_with_lower', 0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'A11_UPPER_CASE_WITH_DIGITS', 'guaranteed_to_pass', ':starts_with_lower', 1,
    );

    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'all_lower_case', 'GUARANTEED_TO_PASS', ':starts_with_upper', 1,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'ALL_UPPER_CASE', 'GUARANTEED_TO_PASS', ':starts_with_upper', 0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file,
        '________ALL_UPPER_CASE_WITH_LEADING_UNDERSCORES',
        'GUARANTEED_TO_PASS',
        ':starts_with_upper',
        0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'a11_lower_case_with_digits', 'GUARANTEED_TO_PASS', ':starts_with_upper', 1,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'A11_UPPER_CASE_WITH_DIGITS', 'GUARANTEED_TO_PASS', ':starts_with_upper', 0,
    );

    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'all_lower_case', 'guaranteed_to_pass', ':no_restriction', 0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'ALL_UPPER_CASE', 'guaranteed_to_pass', ':no_restriction', 0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'mixedCase', 'guaranteed_to_pass', ':no_restriction', 0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file,
        '________ALL_UPPER_CASE_WITH_LEADING_UNDERSCORES',
        'guaranteed_to_pass',
        ':no_restriction',
        0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'a11_lower_case_with_digits', 'guaranteed_to_pass', ':no_restriction', 0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'A11_UPPER_CASE_WITH_DIGITS', 'guaranteed_to_pass', ':no_restriction', 0,
    );

    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'foobar', 'foo_guaranteed_to_pass_bar', 'foo.*bar', 0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'fooXYZZYbar', 'foo_guaranteed_to_pass_bar', 'foo.*bar', 0,
    );
    emit_all_tests_for_name_and_capitalization_scheme(
        $test_file, 'xyzzy', 'foo_guaranteed_to_pass_bar', 'foo.*bar', 1,
    );

    return;
}

sub emit_all_tests_for_name_and_capitalization_scheme {
    my ($test_file, $name, $guaranteed_to_pass, $capitalization_scheme, $failures)
        = @_;

    emit_package_test(
        $test_file,
        $name,
        $guaranteed_to_pass,
        $capitalization_scheme,
        $failures
    );
    emit_subroutine_test(
        $test_file, $name, $capitalization_scheme, $failures,
    );
    emit_local_lexical_variable_in_subroutine_test(
        $test_file, $name, $capitalization_scheme, $failures,
    );
    emit_local_lexical_variable_in_scheduled_subroutine_test(
        $test_file, $name, $capitalization_scheme, $failures,
    );

    # NOTE need it?
    # emit_scoped_lexical_variable_test(
    #     $test_file, $name, $capitalization_scheme, $failures,
    # );

    # NOTE need it?
    # emit_file_lexical_variable_test(
    #     $test_file, $name, $capitalization_scheme, $failures,
    # );

    emit_global_variable_test(
        $test_file, $name, $capitalization_scheme, $failures,
    );
    emit_foreach_loop_variable_tests(
        $test_file, $name, $capitalization_scheme, $failures,
    );
    emit_c_style_for_loop_variable_tests(
        $test_file, $name, $capitalization_scheme, $failures,
    );
    emit_local_lexical_variable_in_io_assignment_in_while_loop_test(
        $test_file, $name, $capitalization_scheme, $failures,
    );
    emit_label_test($test_file, $name, $capitalization_scheme, $failures);

    return;
}

sub emit_package_test {
    my ($test_file, $package_name, $good_package_component, $capitalization_scheme, $failures) = @_;

    print {$test_file} <<"END_PACKAGE_TEST";
===
--- dscr: Package named "$package_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {packages => '$capitalization_scheme'}}
--- input
package $package_name;

===
--- dscr: Exempted package named "bLaHlAhLaH" vs the "$capitalization_scheme" capitalization scheme.
--- failures: 0
--- params: {capitalization => {packages => '$capitalization_scheme', package_exemptions => 'bLa.*LaH'}}
--- input
package bLaHlAhLaH;

===
--- dscr: Package named "${good_package_component}::$package_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {packages => '$capitalization_scheme'}}
--- input
package ${good_package_component}::$package_name;

END_PACKAGE_TEST

    return;
}

sub emit_subroutine_test {
    my ($test_file, $subroutine_name, $capitalization_scheme, $failures) = @_;

    print {$test_file} <<"END_SUBROUTINE_TEST";
===
--- dscr: Subroutine named "$subroutine_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {subroutines => '$capitalization_scheme'}}
--- input
sub $subroutine_name {
    # Blah blah blah
}

===
--- dscr: Exempted subroutine named "bLaHlAhLaH" vs the "$capitalization_scheme" capitalization scheme.
--- failures: 0
--- params: {capitalization => {subroutines => '$capitalization_scheme', subroutine_exemptions => 'bLa.*LaH'}}
--- input
sub bLaHlAhLaH {
    # Blah blah blah
}

===
--- dscr: Subroutine named "Foo::Bar::$subroutine_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {subroutines => '$capitalization_scheme'}}
--- input
sub Foo::Bar::$subroutine_name;

END_SUBROUTINE_TEST

    return;
}

sub emit_local_lexical_variable_in_subroutine_test {
    my ($test_file, $variable_name, $capitalization_scheme, $failures) = @_;

    print {$test_file} <<"END_LOCAL_LEXICAL_VARIABLE_TEST";
===
--- dscr: Local lexical variable in subroutine named "$variable_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {subroutines => ':all_lower', local_lexical_variables => '$capitalization_scheme'}}
--- input
sub some_subroutine {
    my \$$variable_name;
}

===
--- dscr: Exempted local lexical variable in subroutine named "bLaHlAhLaH" vs the "$capitalization_scheme" capitalization scheme.
--- failures: 0
--- params: {capitalization => {subroutines => ':all_lower', local_lexical_variables => '$capitalization_scheme', local_lexical_variable_exemptions => 'bLa.*LaH'}}
--- input
sub some_subroutine {
    my \$bLaHlAhLaH;
}

END_LOCAL_LEXICAL_VARIABLE_TEST

    return;
}

sub emit_local_lexical_variable_in_scheduled_subroutine_test {
    my ($test_file, $variable_name, $capitalization_scheme, $failures) = @_;

    print {$test_file} <<"END_LOCAL_LEXICAL_VARIABLE_TEST";
===
--- dscr: Local lexical variable in subroutine named "$variable_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {local_lexical_variables => '$capitalization_scheme'}}
--- input
CHECK {
    my \$$variable_name;
}

END_LOCAL_LEXICAL_VARIABLE_TEST

    return;
}

sub emit_scoped_lexical_variable_test {
    my ($test_file, $variable_name, $capitalization_scheme, $failures) = @_;

    print {$test_file} <<"END_SCOPED_LEXICAL_VARIABLE_TEST";
===
--- dscr: Scoped lexical variable named "$variable_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {scoped_lexical_variables => '$capitalization_scheme'}}
--- input
{
    my \$$variable_name;
}

===
--- dscr: Exempted scoped lexical variable named "bLaHlAhLaH" vs the "$capitalization_scheme" capitalization scheme.
--- failures: 0
--- params: {capitalization => {scoped_lexical_variables => '$capitalization_scheme', scoped_lexical_variable_exemptions => 'bLa.*LaH'}}
--- input
{
    my \$bLaHlAhLaH;
}

END_SCOPED_LEXICAL_VARIABLE_TEST

    return;
}

sub emit_file_lexical_variable_test {
    my ($test_file, $variable_name, $capitalization_scheme, $failures) = @_;

    print {$test_file} <<"END_FILE_LEXICAL_VARIABLE_TEST";
===
--- dscr: File lexical variable named "$variable_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {file_lexical_variables => '$capitalization_scheme'}}
--- input
my \$$variable_name;

===
--- dscr: Exempted file lexical variable named "bLaHlAhLaH" vs the "$capitalization_scheme" capitalization scheme.
--- failures: 0
--- params: {capitalization => {file_lexical_variables => '$capitalization_scheme', file_lexical_variable_exemptions => 'bLa.*LaH'}}
--- input
my \$bLaHlAhLaH;

END_FILE_LEXICAL_VARIABLE_TEST

    return;
}

sub emit_global_variable_test {
    my ($test_file, $variable_name, $capitalization_scheme, $failures) = @_;

    foreach my $variable_type ( qw< our local > ) {
        print {$test_file} <<"END_FILE_LEXICAL_VARIABLE_TEST";
===
--- dscr: "$variable_type" variable named "$variable_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {global_variables => '$capitalization_scheme'}}
--- input
$variable_type \$$variable_name;

===
--- dscr: Exempted "$variable_type" variable named "bLaHlAhLaH" vs the "$capitalization_scheme" capitalization scheme.
--- failures: 0
--- params: {capitalization => {global_variables => '$capitalization_scheme', global_variable_exemptions => 'bLa.*LaH'}}
--- input
$variable_type \$bLaHlAhLaH;

END_FILE_LEXICAL_VARIABLE_TEST
    }

    return;
}

sub emit_foreach_loop_variable_tests {
    my ($test_file, $variable_name, $capitalization_scheme, $failures) = @_;

    print {$test_file} <<"END_FOREACH_LOOP_VARIABLE_TESTS";
===
--- dscr: Local lexical variable as foreach loop variable named "$variable_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {local_lexical_variables => '$capitalization_scheme'}}
--- input
foreach my \$$variable_name (\@_) {
    say \$$variable_name;
}

===
--- dscr: State variable as foreach loop variable named "$variable_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {local_lexical_variables => '$capitalization_scheme'}}
--- input
foreach state \$$variable_name (\@_) {
    say \$$variable_name;
}

# NOTE need it?
# ===
# --- dscr: Implied local lexical variable as foreach loop variable named "$variable_name" vs the "$capitalization_scheme" capitalization scheme.
# --- failures: $failures
# --- params: {capitalization => {local_lexical_variables => '$capitalization_scheme'}}
# --- input
# foreach \$$variable_name (\@_) {
#     say \$$variable_name;
# }

===
--- dscr: Global variable as foreach loop variable named "$variable_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {global_variables => '$capitalization_scheme'}}
--- input
foreach our \$$variable_name (\@_) {
    say \$$variable_name;
}

===
--- dscr: Localized builtin variable
--- failures: 0
--- params:
--- input
local \$\@;

===
--- dscr: Localized \$\\
--- failures: 0
--- params:
--- input
# \$\\ was missing from the built in global exceptions
local \$\\;

END_FOREACH_LOOP_VARIABLE_TESTS

    return;
}


sub emit_c_style_for_loop_variable_tests {
    my ($test_file, $variable_name, $capitalization_scheme, $failures) = @_;

    print {$test_file} <<"END_C_STYLE_FOR_LOOP_VARIABLE_TESTS";
===
--- dscr: Local lexical variable as C-style for loop variable named "$variable_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {local_lexical_variables => '$capitalization_scheme'}}
--- input
for (my \$$variable_name = -7; \$$variable_name <= 17; \$$variable_name += 3) {
    say \$$variable_name;
}

===
--- dscr: State variable as C-style for loop variable named "$variable_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {local_lexical_variables => '$capitalization_scheme'}}
--- input
# Declare the same variable twice in order to catch the case where the
# variable is the second one in the loop definition.
for (state \$$variable_name = -7; \$$variable_name <= 17; \$$variable_name += 3) {
    say \$$variable_name;
}

===
--- dscr: Global variable as C-style for loop variable named "$variable_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {global_variables => '$capitalization_scheme'}}
--- input
# Declare the same variable twice in order to catch the case where the
# variable is the second one in the loop definition.
for (our \$$variable_name = -7; \$$variable_name <= 17; \$$variable_name += 3) {
    say \$$variable_name;
}

===
--- dscr: Localized variable as C-style for loop variable named "$variable_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {global_variables => '$capitalization_scheme'}}
--- input
# Localize the same variable twice in order to catch the case where the
# variable is the second one in the loop definition.
for (local \$$variable_name = -7; \$$variable_name <= 17; \$$variable_name += 3) {
    say \$$variable_name;
}

===
--- dscr: Localized builtin variable as C-style for loop variable vs the "$capitalization_scheme" capitalization scheme.
--- failures: 0
--- params: {capitalization => {global_variables => '$capitalization_scheme'}}
--- input
for (local \$EVAL_ERROR = -23; \$EVAL_ERROR <= 17; \$EVAL_ERROR += 3) {
    say \$EVAL_ERROR;
}

===
--- dscr: Localized variable in another package as C-style for loop variable vs the "$capitalization_scheme" capitalization scheme.
--- failures: 0
--- params: {capitalization => {global_variables => '$capitalization_scheme'}}
--- input
for (local \$Foo::Baz = -23; \$Foo::Baz <= 17; \$Foo::Baz += 3) {
    say \$Foo::Baz
}

END_C_STYLE_FOR_LOOP_VARIABLE_TESTS

    return;
}

sub emit_local_lexical_variable_in_io_assignment_in_while_loop_test {
    my ($test_file, $variable_name, $capitalization_scheme, $failures) = @_;

    print {$test_file} <<"END_LOCAL_LEXICAL_VARIABLE_TEST";
===
--- dscr: Local lexical variable, in I/O assignment in while loop, named "$variable_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {subroutines => ':all_lower', local_lexical_variables => '$capitalization_scheme'}}
--- input
sub some_subroutine {
    while (my \$$variable_name = <>) {
        say \$$variable_name;
    }
}

END_LOCAL_LEXICAL_VARIABLE_TEST

    return;
}

sub emit_label_test {
    my ($test_file, $label_name, $capitalization_scheme, $failures) = @_;

    print {$test_file} <<"END_LABEL_TEST";
===
--- dscr: Label named "$label_name" vs the "$capitalization_scheme" capitalization scheme.
--- failures: $failures
--- params: {capitalization => {labels => '$capitalization_scheme'}}
--- input
$label_name:
while ( foo() ) {
    next $label_name;
}

===
--- dscr: Exempted label named "bLaHlAhLaH" vs the "$capitalization_scheme" capitalization scheme.
--- failures: 0
--- params: {capitalization => {labels => '$capitalization_scheme', label_exemptions => 'bLa.*LaH'}}
--- input
bLaHlAhLaH:
while ( foo() ) {
    next bLaHlAhLaH;
}
END_LABEL_TEST

    return;
}

sub emit_footer {
    my ($test_file) = @_;

    print {$test_file} <<'END_FOOTER';
# NOTE need it?
# ===
# --- dscr: Variable in continue block gets handled as a local lexical and not a scoped lexical.
# --- failures: 1
# --- params: {capitalization => {local_lexical_variables => ':all_upper', scoped_lexical_variables => ':all_lower'}}
# --- input
# while (blah) {
#     blah;
# }
# continue {
#     my $this_should_be_local_and_not_scoped;
# }

===
--- dscr: Builtin variables and variables in other packages are exempt.
--- failures: 0
--- params: {capitalization => {global_variables => ':all_lower'}}
--- input
local $EVAL_ERROR
local @ARGV;
local %INC;
local $Foo::Bar;

===
--- dscr: Test customization example in the Capitalization POD passing.
--- failures: 0
--- params: {capitalization => {global_variables => 'G_(?:(?!_)\w)+', global_variable_exemptions => '.*THINGY.*'}}
--- input
our $G_FooBar;
our $THINGY;
our @otherTHINGY;

===
--- dscr: Test customization example in the Capitalization POD failing.
--- failures: 4
--- params: {capitalization => {global_variables => 'G_(?:(?!_)\w)+', global_variable_exemptions => '.*THINGY.*'}}
--- input
our $FooBar;
our $G_;
our $G_foo_bar;
our @THING;

END_FOOTER

    return;
}
