/*
Copyright 2014 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package content

import (
	"fmt"
	"reflect"

	"k8s.io/apimachinery/pkg/api/validate/constraints"
)

// MinError returns a string explanation of a "must be greater than or equal"
// validation failure.
func MinError[T constraints.Integer](min T) string {
	return fmt.Sprintf("must be greater than or equal to %d", min)
}

// MaxLenError returns a string explanation of a "string too long" validation
// failure.
func MaxLenError(length int) string {
	return fmt.Sprintf("must be no more than %d bytes", length)
}

// EmptyError returns a string explanation of an "empty string" validation.
func EmptyError() string {
	return "must be non-empty"
}

// RegexError returns a string explanation of a regex validation failure.
func RegexError(msg string, re string, examples ...string) string {
	if len(examples) == 0 {
		return msg + " (regex used for validation is '" + re + "')"
	}
	msg += " (e.g. "
	for i := range examples {
		if i > 0 {
			msg += " or "
		}
		msg += "'" + examples[i] + "', "
	}
	msg += "regex used for validation is '" + re + "')"
	return msg
}

// NEQError returns a string explanation of a "must not be equal to" validation failure.
func NEQError[T any](disallowed T) string {
	format := "%v"
	if reflect.ValueOf(disallowed).Kind() == reflect.String {
		format = "%q"
	}
	return fmt.Sprintf("must not be equal to "+format, disallowed)
}
