﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/EndpointStatus.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace SageMaker {
namespace Model {

/**
 * <p>Provides summary information for an endpoint.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/sagemaker-2017-07-24/EndpointSummary">AWS
 * API Reference</a></p>
 */
class EndpointSummary {
 public:
  AWS_SAGEMAKER_API EndpointSummary() = default;
  AWS_SAGEMAKER_API EndpointSummary(Aws::Utils::Json::JsonView jsonValue);
  AWS_SAGEMAKER_API EndpointSummary& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_SAGEMAKER_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the endpoint.</p>
   */
  inline const Aws::String& GetEndpointName() const { return m_endpointName; }
  inline bool EndpointNameHasBeenSet() const { return m_endpointNameHasBeenSet; }
  template <typename EndpointNameT = Aws::String>
  void SetEndpointName(EndpointNameT&& value) {
    m_endpointNameHasBeenSet = true;
    m_endpointName = std::forward<EndpointNameT>(value);
  }
  template <typename EndpointNameT = Aws::String>
  EndpointSummary& WithEndpointName(EndpointNameT&& value) {
    SetEndpointName(std::forward<EndpointNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the endpoint.</p>
   */
  inline const Aws::String& GetEndpointArn() const { return m_endpointArn; }
  inline bool EndpointArnHasBeenSet() const { return m_endpointArnHasBeenSet; }
  template <typename EndpointArnT = Aws::String>
  void SetEndpointArn(EndpointArnT&& value) {
    m_endpointArnHasBeenSet = true;
    m_endpointArn = std::forward<EndpointArnT>(value);
  }
  template <typename EndpointArnT = Aws::String>
  EndpointSummary& WithEndpointArn(EndpointArnT&& value) {
    SetEndpointArn(std::forward<EndpointArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A timestamp that shows when the endpoint was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTime() const { return m_creationTime; }
  inline bool CreationTimeHasBeenSet() const { return m_creationTimeHasBeenSet; }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  void SetCreationTime(CreationTimeT&& value) {
    m_creationTimeHasBeenSet = true;
    m_creationTime = std::forward<CreationTimeT>(value);
  }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  EndpointSummary& WithCreationTime(CreationTimeT&& value) {
    SetCreationTime(std::forward<CreationTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A timestamp that shows when the endpoint was last modified.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedTime() const { return m_lastModifiedTime; }
  inline bool LastModifiedTimeHasBeenSet() const { return m_lastModifiedTimeHasBeenSet; }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  void SetLastModifiedTime(LastModifiedTimeT&& value) {
    m_lastModifiedTimeHasBeenSet = true;
    m_lastModifiedTime = std::forward<LastModifiedTimeT>(value);
  }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  EndpointSummary& WithLastModifiedTime(LastModifiedTimeT&& value) {
    SetLastModifiedTime(std::forward<LastModifiedTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the endpoint.</p> <ul> <li> <p> <code>OutOfService</code>:
   * Endpoint is not available to take incoming requests.</p> </li> <li> <p>
   * <code>Creating</code>: <a
   * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateEndpoint.html">CreateEndpoint</a>
   * is executing.</p> </li> <li> <p> <code>Updating</code>: <a
   * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateEndpoint.html">UpdateEndpoint</a>
   * or <a
   * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateEndpointWeightsAndCapacities.html">UpdateEndpointWeightsAndCapacities</a>
   * is executing.</p> </li> <li> <p> <code>SystemUpdating</code>: Endpoint is
   * undergoing maintenance and cannot be updated or deleted or re-scaled until it
   * has completed. This maintenance operation does not change any customer-specified
   * values such as VPC config, KMS encryption, model, instance type, or instance
   * count.</p> </li> <li> <p> <code>RollingBack</code>: Endpoint fails to scale up
   * or down or change its variant weight and is in the process of rolling back to
   * its previous configuration. Once the rollback completes, endpoint returns to an
   * <code>InService</code> status. This transitional status only applies to an
   * endpoint that has autoscaling enabled and is undergoing variant weight or
   * capacity changes as part of an <a
   * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateEndpointWeightsAndCapacities.html">UpdateEndpointWeightsAndCapacities</a>
   * call or when the <a
   * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateEndpointWeightsAndCapacities.html">UpdateEndpointWeightsAndCapacities</a>
   * operation is called explicitly.</p> </li> <li> <p> <code>InService</code>:
   * Endpoint is available to process incoming requests.</p> </li> <li> <p>
   * <code>Deleting</code>: <a
   * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteEndpoint.html">DeleteEndpoint</a>
   * is executing.</p> </li> <li> <p> <code>Failed</code>: Endpoint could not be
   * created, updated, or re-scaled. Use
   * <code>DescribeEndpointOutput$FailureReason</code> for information about the
   * failure. <a
   * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteEndpoint.html">DeleteEndpoint</a>
   * is the only operation that can be performed on a failed endpoint.</p> </li>
   * </ul> <p>To get a list of endpoints with a specified status, use the
   * <code>StatusEquals</code> filter with a call to <a
   * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListEndpoints.html">ListEndpoints</a>.</p>
   */
  inline EndpointStatus GetEndpointStatus() const { return m_endpointStatus; }
  inline bool EndpointStatusHasBeenSet() const { return m_endpointStatusHasBeenSet; }
  inline void SetEndpointStatus(EndpointStatus value) {
    m_endpointStatusHasBeenSet = true;
    m_endpointStatus = value;
  }
  inline EndpointSummary& WithEndpointStatus(EndpointStatus value) {
    SetEndpointStatus(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_endpointName;
  bool m_endpointNameHasBeenSet = false;

  Aws::String m_endpointArn;
  bool m_endpointArnHasBeenSet = false;

  Aws::Utils::DateTime m_creationTime{};
  bool m_creationTimeHasBeenSet = false;

  Aws::Utils::DateTime m_lastModifiedTime{};
  bool m_lastModifiedTimeHasBeenSet = false;

  EndpointStatus m_endpointStatus{EndpointStatus::NOT_SET};
  bool m_endpointStatusHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
