﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/connect/ConnectRequest.h>
#include <aws/connect/Connect_EXPORTS.h>
#include <aws/connect/model/UseCaseType.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Connect {
namespace Model {

/**
 */
class CreateUseCaseRequest : public ConnectRequest {
 public:
  AWS_CONNECT_API CreateUseCaseRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateUseCase"; }

  AWS_CONNECT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The identifier of the Amazon Connect instance. You can <a
   * href="https://docs.aws.amazon.com/connect/latest/adminguide/find-instance-arn.html">find
   * the instance ID</a> in the Amazon Resource Name (ARN) of the instance.</p>
   */
  inline const Aws::String& GetInstanceId() const { return m_instanceId; }
  inline bool InstanceIdHasBeenSet() const { return m_instanceIdHasBeenSet; }
  template <typename InstanceIdT = Aws::String>
  void SetInstanceId(InstanceIdT&& value) {
    m_instanceIdHasBeenSet = true;
    m_instanceId = std::forward<InstanceIdT>(value);
  }
  template <typename InstanceIdT = Aws::String>
  CreateUseCaseRequest& WithInstanceId(InstanceIdT&& value) {
    SetInstanceId(std::forward<InstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier for the integration association.</p>
   */
  inline const Aws::String& GetIntegrationAssociationId() const { return m_integrationAssociationId; }
  inline bool IntegrationAssociationIdHasBeenSet() const { return m_integrationAssociationIdHasBeenSet; }
  template <typename IntegrationAssociationIdT = Aws::String>
  void SetIntegrationAssociationId(IntegrationAssociationIdT&& value) {
    m_integrationAssociationIdHasBeenSet = true;
    m_integrationAssociationId = std::forward<IntegrationAssociationIdT>(value);
  }
  template <typename IntegrationAssociationIdT = Aws::String>
  CreateUseCaseRequest& WithIntegrationAssociationId(IntegrationAssociationIdT&& value) {
    SetIntegrationAssociationId(std::forward<IntegrationAssociationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of use case to associate to the integration association. Each
   * integration association can have only one of each use case type.</p>
   */
  inline UseCaseType GetUseCaseType() const { return m_useCaseType; }
  inline bool UseCaseTypeHasBeenSet() const { return m_useCaseTypeHasBeenSet; }
  inline void SetUseCaseType(UseCaseType value) {
    m_useCaseTypeHasBeenSet = true;
    m_useCaseType = value;
  }
  inline CreateUseCaseRequest& WithUseCaseType(UseCaseType value) {
    SetUseCaseType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags used to organize, track, or control access for this resource. For
   * example, { "Tags": {"key1":"value1", "key2":"value2"} }.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateUseCaseRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateUseCaseRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_instanceId;

  Aws::String m_integrationAssociationId;

  UseCaseType m_useCaseType{UseCaseType::NOT_SET};

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_instanceIdHasBeenSet = false;
  bool m_integrationAssociationIdHasBeenSet = false;
  bool m_useCaseTypeHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Connect
}  // namespace Aws
