"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.KmsKeyConfig = void 0;
const kms_keyring_1 = require("@aws-crypto/kms-keyring");
const kms_keyring_2 = require("@aws-crypto/kms-keyring");
const material_management_1 = require("@aws-crypto/material-management");
// an abstract class defining common behavior for operations that SRK and MRK compatibility
// configs should perform
class KmsKeyConfig {
    //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-configuration
    //# `KMS Key ARN` and `KMS MRKey ARN` MUST take an additional argument
    //# that is a KMS ARN.
    constructor(config) {
        (0, material_management_1.readOnlyProperty)(this, '_config', config);
        /* Precondition: config must be a string or object */
        const configType = typeof config;
        (0, material_management_1.needs)(!!config && (configType === 'object' || configType === 'string'), 'Config must be a `discovery` or an object.');
        if (configType === 'string') {
            /* Precondition: Only `discovery` is a valid string value */
            (0, material_management_1.needs)(config === 'discovery', 'Unexpected config shape');
        }
        else if ('identifier' in config ||
            'mrkIdentifier' in config) {
            const arn = 'identifier' in config
                ? config.identifier
                : config.mrkIdentifier;
            /* Precondition: ARN must be a string */
            (0, material_management_1.needs)(typeof arn === 'string', 'ARN must be a string');
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-configuration
            //# To be clear, an KMS ARN for a Multi-Region Key MAY be provided to the `KMS Key ARN` configuration,
            //# and a KMS ARN for non Multi-Region Key MAY be provided to the `KMS MRKey ARN` configuration.
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-configuration
            //# This ARN MUST NOT be an Alias.
            //# This ARN MUST be a valid
            //# [AWS KMS Key ARN](./aws-kms/aws-kms-key-arn.md#a-valid-aws-kms-arn).
            const parsedArn = (0, kms_keyring_1.parseAwsKmsKeyArn)(arn);
            (0, material_management_1.needs)(parsedArn && parsedArn.ResourceType === 'key', `${arn} must be a well-formed AWS KMS non-alias resource arn`);
            (0, material_management_1.readOnlyProperty)(this, '_parsedArn', parsedArn);
            (0, material_management_1.readOnlyProperty)(this, '_arn', arn);
        }
        else if ('region' in config) {
            (0, material_management_1.readOnlyProperty)(this, '_mrkRegion', config.region);
        }
        else {
            (0, material_management_1.needs)(false, 'Unexpected config shape');
        }
        Object.freeze(this);
    }
    getRegion() {
        if (this._config === 'discovery') {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
            //# If a DDB client needs to be constructed and the AWS KMS Configuration is Discovery,
            //# a new DynamoDb client MUST be created with the default configuration.
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
            //# If AWS KMS client needs to be constructed and the AWS KMS Configuration is Discovery,
            //# a new AWS KMS client MUST be created with the default configuration.
            return undefined;
        }
        else if ('identifier' in this._config ||
            'mrkIdentifier' in this._config) {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
            //# If a DDB client needs to be constructed and the AWS KMS Configuration is KMS Key ARN or KMS MRKey ARN,
            //# a new DynamoDb client MUST be created with the region of the supplied KMS ARN.
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
            //# If AWS KMS client needs to be constructed and the AWS KMS Configuration is KMS Key ARN or KMS MRKey ARN,
            //# a new AWS KMS client MUST be created with the region of the supplied KMS ARN.
            return this._parsedArn.Region;
        }
        else if ('region' in this._config) {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
            //# If a DDB client needs to be constructed and the AWS KMS Configuration is MRDiscovery,
            //# a new DynamoDb client MUST be created with the region configured in the MRDiscovery.
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
            //# If AWS KMS client needs to be constructed and the AWS KMS Configuration is MRDiscovery,
            //# a new AWS KMS client MUST be created with the region configured in the MRDiscovery.
            return this._mrkRegion;
        }
        else {
            (0, material_management_1.needs)(false, 'Unexpected configuration state');
        }
    }
    isCompatibleWithArn(otherArn) {
        if (this._config === 'discovery' || 'region' in this._config) {
            // This may result in the function being called twice.
            // However this is the most correct behavior
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-branch-key-decryption
            //# If the Keystore's [AWS KMS Configuration](#aws-kms-configuration) is `Discovery` or `MRDiscovery`,
            //# the `kms-arn` field of DDB response item MUST NOT be an Alias
            //# or the operation MUST fail.
            const parsedArn = (0, kms_keyring_1.parseAwsKmsKeyArn)(otherArn);
            (0, material_management_1.needs)(parsedArn && parsedArn.ResourceType === 'key', `${otherArn} must be a well-formed AWS KMS non-alias resource arn`);
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-key-arn-compatibility
            //# If the [AWS KMS Configuration](#aws-kms-configuration) is Discovery or MRDiscovery,
            //# no comparison is ever made between ARNs.
            return true;
        }
        else if ('identifier' in this._config) {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-key-arn-compatibility
            //# For two ARNs to be compatible:
            //#
            //# If the [AWS KMS Configuration](#aws-kms-configuration) designates single region ARN compatibility,
            //# then two ARNs are compatible if they are exactly equal.
            return this._arn === otherArn;
        }
        else if ('mrkIdentifier' in this._config) {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-key-arn-compatibility
            //# If the [AWS KMS Configuration](#aws-kms-configuration) designates MRK ARN compatibility,
            //# then two ARNs are compatible if they are equal in all parts other than the region.
            //# That is, they are compatible if [AWS KMS MRK Match for Decrypt](aws-kms/aws-kms-mrk-match-for-decrypt.md#implementation) returns true.
            return (0, kms_keyring_2.mrkAwareAwsKmsKeyIdCompare)(this._arn, otherArn);
        }
        else {
            (0, material_management_1.needs)(false, 'Unexpected configuration state');
        }
    }
    getCompatibleArnArn(otherArn) {
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-branch-key-decryption
        //# If the Keystore's [AWS KMS Configuration](#aws-kms-configuration) is `KMS Key ARN` or `KMS MRKey ARN`,
        //# the `kms-arn` field of the DDB response item MUST be
        //# [compatible with](#aws-key-arn-compatibility) the configured KMS Key in
        //# the [AWS KMS Configuration](#aws-kms-configuration) for this keystore,
        //# or the operation MUST fail.
        //# If the Keystore's [AWS KMS Configuration](#aws-kms-configuration) is `Discovery` or `MRDiscovery`,
        //# the `kms-arn` field of DDB response item MUST NOT be an Alias
        //# or the operation MUST fail.
        (0, material_management_1.needs)(this.isCompatibleWithArn(otherArn), 'KMS ARN from DDB response item MUST be compatible with the configured KMS Key in the AWS KMS Configuration for this keystore');
        if (this._config == 'discovery') {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-branch-key-decryption
            //# - `KeyId`, if the KMS Configuration is Discovery, MUST be the `kms-arn` attribute value of the AWS DDB response item.
            return otherArn;
        }
        else if ('region' in this._config) {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-branch-key-decryption
            //# If the KMS Configuration is MRDiscovery, `KeyId` MUST be the `kms-arn` attribute value of the AWS DDB response item, with the region replaced by the configured region.
            const parsedArn = (0, kms_keyring_1.parseAwsKmsKeyArn)(otherArn);
            (0, material_management_1.needs)(parsedArn, 'KMS ARN from the keystore is not an ARN:' + otherArn);
            return (0, kms_keyring_1.isMultiRegionAwsKmsArn)(parsedArn)
                ? (0, kms_keyring_2.constructArnInOtherRegion)(parsedArn, this._mrkRegion)
                : otherArn;
        }
        else if ('identifier' in this._config ||
            'mrkIdentifier' in this._config) {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-branch-key-decryption
            //# Otherwise, it MUST BE the Keystore's configured KMS Key.
            // In this case, the equality condition has already been satisfied.
            // In the SRK case this is strict equality,
            // in the MKR case this is functional (everything but region)
            return this._arn;
        }
        else {
            // This is for completeness.
            // It should should be impossible to get here.
            (0, material_management_1.needs)(false, 'Unexpected configuration state');
        }
    }
}
exports.KmsKeyConfig = KmsKeyConfig;
//# sourceMappingURL=data:application/json;base64,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