/* SPDX-FileCopyrightText: 2025 - Sébastien Wilmet
 * SPDX-License-Identifier: LGPL-3.0-or-later
 */

#include "gedit-tab-utils.h"
#include <glib/gi18n.h>

gchar *
_gedit_tab_utils_get_tooltip (GeditTab *tab)
{
	GeditDocument *doc;
	gchar *full_name;
	gchar *full_name_markup;
	GeditTabState tab_state;
	gchar *tip = NULL;

	g_return_val_if_fail (GEDIT_IS_TAB (tab), NULL);

	doc = gedit_tab_get_document (tab);
	full_name = tepl_file_get_full_name (tepl_buffer_get_file (TEPL_BUFFER (doc)));
	full_name_markup = g_markup_printf_escaped ("<i>%s</i>", full_name);

	tab_state = gedit_tab_get_state (tab);

	if (tab_state == GEDIT_TAB_STATE_LOADING_ERROR)
	{
		tip = g_strdup_printf (_("Error opening file %s"), full_name_markup);
	}
	else if (tab_state == GEDIT_TAB_STATE_REVERTING_ERROR)
	{
		tip = g_strdup_printf (_("Error reverting file %s"), full_name_markup);
	}
	else if (tab_state == GEDIT_TAB_STATE_SAVING_ERROR)
	{
		tip = g_strdup_printf (_("Error saving file %s"), full_name_markup);
	}
	else
	{
		gchar *content_type;
		gchar *content_type_description = NULL;
		gchar *mime_type;
		gchar *content_full_description;
		GtkSourceFile *file;
		const GtkSourceEncoding *encoding;
		gchar *charset;

		content_type = gedit_document_get_content_type (doc);
		if (content_type != NULL)
		{
			content_type_description = g_content_type_get_description (content_type);
		}

		mime_type = gedit_document_get_mime_type (doc);

		if (content_type_description == NULL)
		{
			content_full_description = g_strdup (mime_type);
		}
		else
		{
			content_full_description = g_strdup_printf ("%s (%s)",
								    content_type_description,
								    mime_type);
		}

		file = gedit_document_get_file (doc);
		encoding = gtk_source_file_get_encoding (file);

		if (encoding != NULL)
		{
			charset = gtk_source_encoding_to_string (encoding);
		}
		else
		{
			GtkSourceEncoding *utf8_encoding;

			utf8_encoding = gtk_source_encoding_new_utf8 ();
			charset = gtk_source_encoding_to_string (utf8_encoding);
			gtk_source_encoding_free (utf8_encoding);
		}

		tip = g_markup_printf_escaped ("<b>%s</b> %s\n\n"
					       "<b>%s</b> %s\n"
					       "<b>%s</b> %s",
					       _("Name:"), full_name,
					       _("MIME Type:"), content_full_description,
					       _("Encoding:"), charset);

		g_free (content_type);
		g_free (content_type_description);
		g_free (mime_type);
		g_free (content_full_description);
		g_free (charset);
	}

	g_free (full_name);
	g_free (full_name_markup);
	return tip;
}

const gchar *
_gedit_tab_utils_get_icon_name (GeditTab *tab)
{
	GeditTabState tab_state;

	g_return_val_if_fail (GEDIT_IS_TAB (tab), NULL);

	tab_state = gedit_tab_get_state (tab);

	switch (tab_state)
	{
		case GEDIT_TAB_STATE_PRINTING:
			return "printer-printing-symbolic";

		case GEDIT_TAB_STATE_SHOWING_PRINT_PREVIEW:
			return "printer-symbolic";

		case GEDIT_TAB_STATE_LOADING_ERROR:
		case GEDIT_TAB_STATE_REVERTING_ERROR:
		case GEDIT_TAB_STATE_SAVING_ERROR:
		case GEDIT_TAB_STATE_GENERIC_ERROR:
			return "dialog-error-symbolic";

		case GEDIT_TAB_STATE_EXTERNALLY_MODIFIED_NOTIFICATION:
			return "dialog-warning-symbolic";

		default:
			break;
	}

	return NULL;
}
