/* Copyright (C) 2020 Google, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include "tu_tracepoints.h"

#include "vk_enum_to_str.h"
#include "tu_cmd_buffer.h"
#include "tu_device.h"
#include "vulkan/vulkan_core.h"

#define __NEEDS_TRACE_PRIV
#include "util/u_debug.h"
#include "util/perf/u_trace_priv.h"

static const struct debug_control config_control[] = {
   { "cmd_buffer", TU_GPU_TRACEPOINT_CMD_BUFFER, },
   { "secondary_cmd_buffer", TU_GPU_TRACEPOINT_SECONDARY_CMD_BUFFER, },
   { "render_pass", TU_GPU_TRACEPOINT_RENDER_PASS, },
   { "draw", TU_GPU_TRACEPOINT_DRAW, },
   { "binning_ib", TU_GPU_TRACEPOINT_BINNING_IB, },
   { "draw_ib_sysmem", TU_GPU_TRACEPOINT_DRAW_IB_SYSMEM, },
   { "draw_ib_gmem", TU_GPU_TRACEPOINT_DRAW_IB_GMEM, },
   { "generic_clear", TU_GPU_TRACEPOINT_GENERIC_CLEAR, },
   { "gmem_clear", TU_GPU_TRACEPOINT_GMEM_CLEAR, },
   { "sysmem_clear", TU_GPU_TRACEPOINT_SYSMEM_CLEAR, },
   { "sysmem_clear_all", TU_GPU_TRACEPOINT_SYSMEM_CLEAR_ALL, },
   { "gmem_load", TU_GPU_TRACEPOINT_GMEM_LOAD, },
   { "gmem_store", TU_GPU_TRACEPOINT_GMEM_STORE, },
   { "sysmem_resolve", TU_GPU_TRACEPOINT_SYSMEM_RESOLVE, },
   { "blit", TU_GPU_TRACEPOINT_BLIT, },
   { "compute", TU_GPU_TRACEPOINT_COMPUTE, },
   { "compute_indirect", TU_GPU_TRACEPOINT_COMPUTE_INDIRECT, },
   { "cmd_buffer_annotation", TU_GPU_TRACEPOINT_CMD_BUFFER_ANNOTATION, },
   { "cmd_buffer_annotation_rp", TU_GPU_TRACEPOINT_CMD_BUFFER_ANNOTATION_RP, },
   { NULL, 0, },
};
uint64_t tu_gpu_tracepoint = 0;

static void
tu_gpu_tracepoint_variable_once(void)
{
   uint64_t default_value = 0
     | TU_GPU_TRACEPOINT_CMD_BUFFER
     | TU_GPU_TRACEPOINT_RENDER_PASS
     | TU_GPU_TRACEPOINT_BINNING_IB
     | TU_GPU_TRACEPOINT_DRAW_IB_SYSMEM
     | TU_GPU_TRACEPOINT_DRAW_IB_GMEM
     | TU_GPU_TRACEPOINT_GENERIC_CLEAR
     | TU_GPU_TRACEPOINT_GMEM_CLEAR
     | TU_GPU_TRACEPOINT_SYSMEM_CLEAR
     | TU_GPU_TRACEPOINT_SYSMEM_CLEAR_ALL
     | TU_GPU_TRACEPOINT_GMEM_LOAD
     | TU_GPU_TRACEPOINT_GMEM_STORE
     | TU_GPU_TRACEPOINT_SYSMEM_RESOLVE
     | TU_GPU_TRACEPOINT_BLIT
     | TU_GPU_TRACEPOINT_COMPUTE
     | TU_GPU_TRACEPOINT_COMPUTE_INDIRECT
     | TU_GPU_TRACEPOINT_CMD_BUFFER_ANNOTATION
     | TU_GPU_TRACEPOINT_CMD_BUFFER_ANNOTATION_RP
     ;

   tu_gpu_tracepoint =
      parse_enable_string(os_get_option("TU_GPU_TRACEPOINT"),
                          default_value,
                          config_control);
}

void
tu_gpu_tracepoint_config_variable(void)
{
   static once_flag process_tu_gpu_tracepoint_variable_flag = ONCE_FLAG_INIT;

   call_once(&process_tu_gpu_tracepoint_variable_flag,
             tu_gpu_tracepoint_variable_once);
}

/*
 * start_cmd_buffer
 */
static void __print_start_cmd_buffer(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_cmd_buffer *__entry =
      (const struct trace_start_cmd_buffer *)arg;
   fprintf(out, ""
      "TUdebugFlags=%s, "
      "IR3debugFlags=%s, "
      "command_buffer_handle=%" PRIu64 ", "
      "appName=%s, "
      "engineName=%s, "
      "oneTimeSubmit=%u, "
      "simultaneousUse=%u, "
         "\n"
   ,__entry->TUdebugFlags
   ,__entry->IR3debugFlags
   ,(uint64_t)__entry->command_buffer_handle
   ,__entry->appName
   ,__entry->engineName
   ,__entry->oneTimeSubmit
   ,__entry->simultaneousUse
   );
}

static void __print_json_start_cmd_buffer(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_cmd_buffer *__entry =
      (const struct trace_start_cmd_buffer *)arg;
   fprintf(out, ""
      "\"TUdebugFlags\": \"%s\""
         ", "
      "\"IR3debugFlags\": \"%s\""
         ", "
      "\"command_buffer_handle\": \"%" PRIu64 "\""
         ", "
      "\"appName\": \"%s\""
         ", "
      "\"engineName\": \"%s\""
         ", "
      "\"oneTimeSubmit\": \"%u\""
         ", "
      "\"simultaneousUse\": \"%u\""
   ,__entry->TUdebugFlags
   ,__entry->IR3debugFlags
   ,(uint64_t)__entry->command_buffer_handle
   ,__entry->appName
   ,__entry->engineName
   ,__entry->oneTimeSubmit
   ,__entry->simultaneousUse
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_cmd_buffer(struct u_trace_context *utctx, void *cs, struct trace_start_cmd_buffer *entry) {
   tu_cs_trace_start(utctx, cs, "start_cmd_buffer("
      "TUdebugFlags=%s"
      ",IR3debugFlags=%s"
      ",command_buffer_handle=%" PRIu64 ""
      ",appName=%s"
      ",engineName=%s"
      ",oneTimeSubmit=%u"
      ",simultaneousUse=%u"
      ")"
      ,entry->TUdebugFlags
      ,entry->IR3debugFlags
      ,(uint64_t)entry->command_buffer_handle
      ,entry->appName
      ,entry->engineName
      ,entry->oneTimeSubmit
      ,entry->simultaneousUse
   );
}

static const struct u_tracepoint __tp_start_cmd_buffer = {
    "start_cmd_buffer",
    ALIGN_POT(sizeof(struct trace_start_cmd_buffer), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    0,
    __print_start_cmd_buffer,
    __print_json_start_cmd_buffer,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_cmd_buffer,
#endif
};
void __trace_start_cmd_buffer(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
   , const char *TUdebugFlags
   , const char *IR3debugFlags
) {
   struct trace_start_cmd_buffer entry;
   UNUSED struct trace_start_cmd_buffer *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_cmd_buffer *)u_trace_appendv(ut, cs, &__tp_start_cmd_buffer,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   __entry->appName = cmd->device->instance->vk.app_info.app_name ? cmd->device->instance->vk.app_info.app_name : "Unknown";
   __entry->engineName = cmd->device->instance->vk.app_info.engine_name ? cmd->device->instance->vk.app_info.engine_name : "Unknown";
   __entry->oneTimeSubmit = (cmd->usage_flags & VK_COMMAND_BUFFER_USAGE_ONE_TIME_SUBMIT_BIT);
   __entry->simultaneousUse = (cmd->usage_flags & VK_COMMAND_BUFFER_USAGE_SIMULTANEOUS_USE_BIT);
   strncpy(__entry->TUdebugFlags, TUdebugFlags, 96);
   strncpy(__entry->IR3debugFlags, IR3debugFlags, 96);
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_cmd_buffer(ut->utctx, cs, __entry);
}

/*
 * end_cmd_buffer
 */
static void __print_end_cmd_buffer(FILE *out, const void *arg, const void *indirect) {
   const struct trace_end_cmd_buffer *__entry =
      (const struct trace_end_cmd_buffer *)arg;
   fprintf(out, ""
      "renderpasses=%u, "
      "dispatches=%u, "
         "\n"
   ,__entry->renderpasses
   ,__entry->dispatches
   );
}

static void __print_json_end_cmd_buffer(FILE *out, const void *arg, const void *indirect) {
   const struct trace_end_cmd_buffer *__entry =
      (const struct trace_end_cmd_buffer *)arg;
   fprintf(out, ""
      "\"renderpasses\": \"%u\""
         ", "
      "\"dispatches\": \"%u\""
   ,__entry->renderpasses
   ,__entry->dispatches
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_cmd_buffer(struct u_trace_context *utctx, void *cs, struct trace_end_cmd_buffer *entry) {
   tu_cs_trace_end(utctx, cs, "end_cmd_buffer("
      "renderpasses=%u"
      ",dispatches=%u"
      ")"
      ,entry->renderpasses
      ,entry->dispatches
   );
}

static const struct u_tracepoint __tp_end_cmd_buffer = {
    "end_cmd_buffer",
    ALIGN_POT(sizeof(struct trace_end_cmd_buffer), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    1,
    __print_end_cmd_buffer,
    __print_json_end_cmd_buffer,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_cmd_buffer,
#endif
};
void __trace_end_cmd_buffer(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , const struct tu_cmd_buffer * cmd
) {
   struct trace_end_cmd_buffer entry;
   UNUSED struct trace_end_cmd_buffer *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_cmd_buffer *)u_trace_appendv(ut, cs, &__tp_end_cmd_buffer,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->renderpasses = cmd->state.total_renderpasses;
   __entry->dispatches = cmd->state.total_dispatches;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_cmd_buffer(ut->utctx, cs, __entry);
}

/*
 * start_secondary_cmd_buffer
 */
static void __print_start_secondary_cmd_buffer(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_secondary_cmd_buffer *__entry =
      (const struct trace_start_secondary_cmd_buffer *)arg;
   fprintf(out, ""
      "command_buffer_handle=%" PRIu64 ", "
      "render_pass_continue=%u, "
         "\n"
   ,(uint64_t)__entry->command_buffer_handle
   ,__entry->render_pass_continue
   );
}

static void __print_json_start_secondary_cmd_buffer(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_secondary_cmd_buffer *__entry =
      (const struct trace_start_secondary_cmd_buffer *)arg;
   fprintf(out, ""
      "\"command_buffer_handle\": \"%" PRIu64 "\""
         ", "
      "\"render_pass_continue\": \"%u\""
   ,(uint64_t)__entry->command_buffer_handle
   ,__entry->render_pass_continue
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_secondary_cmd_buffer(struct u_trace_context *utctx, void *cs, struct trace_start_secondary_cmd_buffer *entry) {
   tu_cs_trace_start(utctx, cs, "start_secondary_cmd_buffer("
      "command_buffer_handle=%" PRIu64 ""
      ",render_pass_continue=%u"
      ")"
      ,(uint64_t)entry->command_buffer_handle
      ,entry->render_pass_continue
   );
}

static const struct u_tracepoint __tp_start_secondary_cmd_buffer = {
    "start_secondary_cmd_buffer",
    ALIGN_POT(sizeof(struct trace_start_secondary_cmd_buffer), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    2,
    __print_start_secondary_cmd_buffer,
    __print_json_start_secondary_cmd_buffer,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_secondary_cmd_buffer,
#endif
};
void __trace_start_secondary_cmd_buffer(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
) {
   struct trace_start_secondary_cmd_buffer entry;
   UNUSED struct trace_start_secondary_cmd_buffer *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_secondary_cmd_buffer *)u_trace_appendv(ut, cs, &__tp_start_secondary_cmd_buffer,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   __entry->render_pass_continue = !!(cmd->usage_flags & VK_COMMAND_BUFFER_USAGE_RENDER_PASS_CONTINUE_BIT);
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_secondary_cmd_buffer(ut->utctx, cs, __entry);
}

/*
 * end_secondary_cmd_buffer
 */
#define __print_end_secondary_cmd_buffer NULL
#define __print_json_end_secondary_cmd_buffer NULL

__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_secondary_cmd_buffer(struct u_trace_context *utctx, void *cs, struct trace_end_secondary_cmd_buffer *entry) {
   tu_cs_trace_end(utctx, cs, "end_secondary_cmd_buffer("
      ")"
   );
}

static const struct u_tracepoint __tp_end_secondary_cmd_buffer = {
    "end_secondary_cmd_buffer",
    ALIGN_POT(sizeof(struct trace_end_secondary_cmd_buffer), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    3,
    __print_end_secondary_cmd_buffer,
    __print_json_end_secondary_cmd_buffer,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_secondary_cmd_buffer,
#endif
};
void __trace_end_secondary_cmd_buffer(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
) {
   struct trace_end_secondary_cmd_buffer entry;
   UNUSED struct trace_end_secondary_cmd_buffer *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_secondary_cmd_buffer *)u_trace_appendv(ut, cs, &__tp_end_secondary_cmd_buffer,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_secondary_cmd_buffer(ut->utctx, cs, __entry);
}

/*
 * start_render_pass
 */
static void __print_start_render_pass(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_render_pass *__entry =
      (const struct trace_start_render_pass *)arg;
   fprintf(out, ""
      "maxSamples=%u, "
      "clearCPP=%u, "
      "loadCPP=%u, "
      "storeCPP=%u, "
      "hasDepth=%s, "
      "ubwc=%s, "
      "command_buffer_handle=%" PRIu64 ", "
      "width=%u, "
      "height=%u, "
      "attachment_count=%u, "
      "numberOfBins=%u, "
      "binWidth=%u, "
      "binHeight=%u, "
         "\n"
   ,__entry->maxSamples
   ,__entry->clearCPP
   ,__entry->loadCPP
   ,__entry->storeCPP
   ,(__entry->hasDepth ? "true" : "false")
   ,__entry->ubwc
   ,(uint64_t)__entry->command_buffer_handle
   ,__entry->width
   ,__entry->height
   ,__entry->attachment_count
   ,__entry->numberOfBins
   ,__entry->binWidth
   ,__entry->binHeight
   );
}

static void __print_json_start_render_pass(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_render_pass *__entry =
      (const struct trace_start_render_pass *)arg;
   fprintf(out, ""
      "\"maxSamples\": \"%u\""
         ", "
      "\"clearCPP\": \"%u\""
         ", "
      "\"loadCPP\": \"%u\""
         ", "
      "\"storeCPP\": \"%u\""
         ", "
      "\"hasDepth\": \"%s\""
         ", "
      "\"ubwc\": \"%s\""
         ", "
      "\"command_buffer_handle\": \"%" PRIu64 "\""
         ", "
      "\"width\": \"%u\""
         ", "
      "\"height\": \"%u\""
         ", "
      "\"attachment_count\": \"%u\""
         ", "
      "\"numberOfBins\": \"%u\""
         ", "
      "\"binWidth\": \"%u\""
         ", "
      "\"binHeight\": \"%u\""
   ,__entry->maxSamples
   ,__entry->clearCPP
   ,__entry->loadCPP
   ,__entry->storeCPP
   ,(__entry->hasDepth ? "true" : "false")
   ,__entry->ubwc
   ,(uint64_t)__entry->command_buffer_handle
   ,__entry->width
   ,__entry->height
   ,__entry->attachment_count
   ,__entry->numberOfBins
   ,__entry->binWidth
   ,__entry->binHeight
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_render_pass(struct u_trace_context *utctx, void *cs, struct trace_start_render_pass *entry) {
   tu_cs_trace_start(utctx, cs, "start_render_pass("
      "maxSamples=%u"
      ",clearCPP=%u"
      ",loadCPP=%u"
      ",storeCPP=%u"
      ",hasDepth=%s"
      ",ubwc=%s"
      ",command_buffer_handle=%" PRIu64 ""
      ",width=%u"
      ",height=%u"
      ",attachment_count=%u"
      ",numberOfBins=%u"
      ",binWidth=%u"
      ",binHeight=%u"
      ")"
      ,entry->maxSamples
      ,entry->clearCPP
      ,entry->loadCPP
      ,entry->storeCPP
      ,(entry->hasDepth ? "true" : "false")
      ,entry->ubwc
      ,(uint64_t)entry->command_buffer_handle
      ,entry->width
      ,entry->height
      ,entry->attachment_count
      ,entry->numberOfBins
      ,entry->binWidth
      ,entry->binHeight
   );
}

static const struct u_tracepoint __tp_start_render_pass = {
    "start_render_pass",
    ALIGN_POT(sizeof(struct trace_start_render_pass), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    4,
    __print_start_render_pass,
    __print_json_start_render_pass,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_render_pass,
#endif
};
void __trace_start_render_pass(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
   , const struct tu_framebuffer * fb
   , const struct tu_tiling_config * tiling
   , uint8_t maxSamples
   , uint8_t clearCPP
   , uint8_t loadCPP
   , uint8_t storeCPP
   , bool hasDepth
   , const char *ubwc
) {
   struct trace_start_render_pass entry;
   UNUSED struct trace_start_render_pass *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_render_pass *)u_trace_appendv(ut, cs, &__tp_start_render_pass,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   __entry->width = fb->width;
   __entry->height = fb->height;
   __entry->attachment_count = fb->attachment_count;
   __entry->numberOfBins = tiling->vsc.tile_count.width * tiling->vsc.tile_count.height;
   __entry->binWidth = tiling->tile0.width;
   __entry->binHeight = tiling->tile0.height;
   __entry->maxSamples = maxSamples;
   __entry->clearCPP = clearCPP;
   __entry->loadCPP = loadCPP;
   __entry->storeCPP = storeCPP;
   __entry->hasDepth = hasDepth;
   strncpy(__entry->ubwc, ubwc, 11);
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_render_pass(ut->utctx, cs, __entry);
}

/*
 * end_render_pass
 */
static void __print_end_render_pass(FILE *out, const void *arg, const void *indirect) {
   const struct trace_end_render_pass *__entry =
      (const struct trace_end_render_pass *)arg;
   const uint32_t *__lrzStatus = (const uint32_t *) ((char *)indirect + 0);
   fprintf(out, ""
      "tiledRender=%s, "
      "tilingDisableReason=%s, "
      "drawCount=%u, "
      "avgPerSampleBandwidth=%u, "
      "lrz=%s, "
      "lrzDisableReason=%s, "
      "lrzDisabledAtDraw=%d, "
      "lrzWriteDisabledAtDraw=%d, "
      "lrzStatus=%s, "
         "\n"
   ,(__entry->tiledRender ? "true" : "false")
   ,__entry->tilingDisableReason
   ,__entry->drawCount
   ,__entry->avgPerSampleBandwidth
   ,(__entry->lrz ? "true" : "false")
   ,__entry->lrzDisableReason
   ,__entry->lrzDisabledAtDraw
   ,__entry->lrzWriteDisabledAtDraw
   ,(fd_lrz_gpu_dir_to_str((enum fd_lrz_gpu_dir)(*__lrzStatus & 0xff)))
   );
}

static void __print_json_end_render_pass(FILE *out, const void *arg, const void *indirect) {
   const struct trace_end_render_pass *__entry =
      (const struct trace_end_render_pass *)arg;
   const uint32_t *__lrzStatus = (const uint32_t *) ((char *)indirect + 0);
   fprintf(out, ""
      "\"tiledRender\": \"%s\""
         ", "
      "\"tilingDisableReason\": \"%s\""
         ", "
      "\"drawCount\": \"%u\""
         ", "
      "\"avgPerSampleBandwidth\": \"%u\""
         ", "
      "\"lrz\": \"%s\""
         ", "
      "\"lrzDisableReason\": \"%s\""
         ", "
      "\"lrzDisabledAtDraw\": \"%d\""
         ", "
      "\"lrzWriteDisabledAtDraw\": \"%d\""
         ", "
      "\"lrzStatus\": \"%s\""
   ,(__entry->tiledRender ? "true" : "false")
   ,__entry->tilingDisableReason
   ,__entry->drawCount
   ,__entry->avgPerSampleBandwidth
   ,(__entry->lrz ? "true" : "false")
   ,__entry->lrzDisableReason
   ,__entry->lrzDisabledAtDraw
   ,__entry->lrzWriteDisabledAtDraw
   ,(fd_lrz_gpu_dir_to_str((enum fd_lrz_gpu_dir)(*__lrzStatus & 0xff)))
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_render_pass(struct u_trace_context *utctx, void *cs, struct trace_end_render_pass *entry) {
   tu_cs_trace_end(utctx, cs, "end_render_pass("
      "tiledRender=%s"
      ",tilingDisableReason=%s"
      ",drawCount=%u"
      ",avgPerSampleBandwidth=%u"
      ",lrz=%s"
      ",lrzDisableReason=%s"
      ",lrzDisabledAtDraw=%d"
      ",lrzWriteDisabledAtDraw=%d"
      ")"
      ,(entry->tiledRender ? "true" : "false")
      ,entry->tilingDisableReason
      ,entry->drawCount
      ,entry->avgPerSampleBandwidth
      ,(entry->lrz ? "true" : "false")
      ,entry->lrzDisableReason
      ,entry->lrzDisabledAtDraw
      ,entry->lrzWriteDisabledAtDraw
   );
}

static const struct u_tracepoint __tp_end_render_pass = {
    "end_render_pass",
    ALIGN_POT(sizeof(struct trace_end_render_pass), 8),   /* keep size 64b aligned */
    0
    + sizeof(uint32_t)
    ,
    0,
    5,
    __print_end_render_pass,
    __print_json_end_render_pass,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_render_pass,
#endif
};
void __trace_end_render_pass(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , bool tiledRender
   , const char * tilingDisableReason
   , uint32_t drawCount
   , uint32_t avgPerSampleBandwidth
   , bool lrz
   , const char * lrzDisableReason
   , int32_t lrzDisabledAtDraw
   , int32_t lrzWriteDisabledAtDraw
   , struct u_trace_address lrzStatus
) {
   struct trace_end_render_pass entry;
   struct u_trace_address indirects[] = {
      lrzStatus,
   };
   uint8_t indirect_sizes[] = {
      sizeof(uint32_t),
   };
   UNUSED struct trace_end_render_pass *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_render_pass *)u_trace_appendv(ut, cs, &__tp_end_render_pass,
                                                    0
                                                    ,
                                                    ARRAY_SIZE(indirects), indirects, indirect_sizes
                                                    ) :
      &entry;
   __entry->tiledRender = tiledRender;
   __entry->tilingDisableReason = tilingDisableReason;
   __entry->drawCount = drawCount;
   __entry->avgPerSampleBandwidth = avgPerSampleBandwidth;
   __entry->lrz = lrz;
   __entry->lrzDisableReason = lrzDisableReason;
   __entry->lrzDisabledAtDraw = lrzDisabledAtDraw;
   __entry->lrzWriteDisabledAtDraw = lrzWriteDisabledAtDraw;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_render_pass(ut->utctx, cs, __entry);
}

/*
 * start_draw
 */
static void __print_start_draw(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_draw *__entry =
      (const struct trace_start_draw *)arg;
   fprintf(out, ""
      "count=%u, "
      "vs_sha1=%s, "
      "tcs_sha1=%s, "
      "tes_sha1=%s, "
      "gs_sha1=%s, "
      "fs_sha1=%s, "
      "command_buffer_handle=%" PRIu64 ", "
         "\n"
   ,__entry->count
   ,__entry->vs_sha1
   ,__entry->tcs_sha1
   ,__entry->tes_sha1
   ,__entry->gs_sha1
   ,__entry->fs_sha1
   ,(uint64_t)__entry->command_buffer_handle
   );
}

static void __print_json_start_draw(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_draw *__entry =
      (const struct trace_start_draw *)arg;
   fprintf(out, ""
      "\"count\": \"%u\""
         ", "
      "\"vs_sha1\": \"%s\""
         ", "
      "\"tcs_sha1\": \"%s\""
         ", "
      "\"tes_sha1\": \"%s\""
         ", "
      "\"gs_sha1\": \"%s\""
         ", "
      "\"fs_sha1\": \"%s\""
         ", "
      "\"command_buffer_handle\": \"%" PRIu64 "\""
   ,__entry->count
   ,__entry->vs_sha1
   ,__entry->tcs_sha1
   ,__entry->tes_sha1
   ,__entry->gs_sha1
   ,__entry->fs_sha1
   ,(uint64_t)__entry->command_buffer_handle
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_draw(struct u_trace_context *utctx, void *cs, struct trace_start_draw *entry) {
   tu_cs_trace_start(utctx, cs, "start_draw("
      "count=%u"
      ",vs_sha1=%s"
      ",tcs_sha1=%s"
      ",tes_sha1=%s"
      ",gs_sha1=%s"
      ",fs_sha1=%s"
      ",command_buffer_handle=%" PRIu64 ""
      ")"
      ,entry->count
      ,entry->vs_sha1
      ,entry->tcs_sha1
      ,entry->tes_sha1
      ,entry->gs_sha1
      ,entry->fs_sha1
      ,(uint64_t)entry->command_buffer_handle
   );
}

static const struct u_tracepoint __tp_start_draw = {
    "start_draw",
    ALIGN_POT(sizeof(struct trace_start_draw), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    6,
    __print_start_draw,
    __print_json_start_draw,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_draw,
#endif
};
void __trace_start_draw(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
   , uint32_t count
   , tu_sha1_str vs_sha1
   , tu_sha1_str tcs_sha1
   , tu_sha1_str tes_sha1
   , tu_sha1_str gs_sha1
   , tu_sha1_str fs_sha1
) {
   struct trace_start_draw entry;
   UNUSED struct trace_start_draw *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_draw *)u_trace_appendv(ut, cs, &__tp_start_draw,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   __entry->count = count;
     strcpy(__entry->vs_sha1, vs_sha1);
     strcpy(__entry->tcs_sha1, tcs_sha1);
     strcpy(__entry->tes_sha1, tes_sha1);
     strcpy(__entry->gs_sha1, gs_sha1);
     strcpy(__entry->fs_sha1, fs_sha1);
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_draw(ut->utctx, cs, __entry);
}

/*
 * end_draw
 */
#define __print_end_draw NULL
#define __print_json_end_draw NULL

__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_draw(struct u_trace_context *utctx, void *cs, struct trace_end_draw *entry) {
   tu_cs_trace_end(utctx, cs, "end_draw("
      ")"
   );
}

static const struct u_tracepoint __tp_end_draw = {
    "end_draw",
    ALIGN_POT(sizeof(struct trace_end_draw), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    7,
    __print_end_draw,
    __print_json_end_draw,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_draw,
#endif
};
void __trace_end_draw(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
) {
   struct trace_end_draw entry;
   UNUSED struct trace_end_draw *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_draw *)u_trace_appendv(ut, cs, &__tp_end_draw,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_draw(ut->utctx, cs, __entry);
}

/*
 * start_binning_ib
 */
static void __print_start_binning_ib(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_binning_ib *__entry =
      (const struct trace_start_binning_ib *)arg;
   fprintf(out, ""
      "command_buffer_handle=%" PRIu64 ", "
         "\n"
   ,(uint64_t)__entry->command_buffer_handle
   );
}

static void __print_json_start_binning_ib(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_binning_ib *__entry =
      (const struct trace_start_binning_ib *)arg;
   fprintf(out, ""
      "\"command_buffer_handle\": \"%" PRIu64 "\""
   ,(uint64_t)__entry->command_buffer_handle
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_binning_ib(struct u_trace_context *utctx, void *cs, struct trace_start_binning_ib *entry) {
   tu_cs_trace_start(utctx, cs, "start_binning_ib("
      "command_buffer_handle=%" PRIu64 ""
      ")"
      ,(uint64_t)entry->command_buffer_handle
   );
}

static const struct u_tracepoint __tp_start_binning_ib = {
    "start_binning_ib",
    ALIGN_POT(sizeof(struct trace_start_binning_ib), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    8,
    __print_start_binning_ib,
    __print_json_start_binning_ib,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_binning_ib,
#endif
};
void __trace_start_binning_ib(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
) {
   struct trace_start_binning_ib entry;
   UNUSED struct trace_start_binning_ib *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_binning_ib *)u_trace_appendv(ut, cs, &__tp_start_binning_ib,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_binning_ib(ut->utctx, cs, __entry);
}

/*
 * end_binning_ib
 */
#define __print_end_binning_ib NULL
#define __print_json_end_binning_ib NULL

__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_binning_ib(struct u_trace_context *utctx, void *cs, struct trace_end_binning_ib *entry) {
   tu_cs_trace_end(utctx, cs, "end_binning_ib("
      ")"
   );
}

static const struct u_tracepoint __tp_end_binning_ib = {
    "end_binning_ib",
    ALIGN_POT(sizeof(struct trace_end_binning_ib), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    9,
    __print_end_binning_ib,
    __print_json_end_binning_ib,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_binning_ib,
#endif
};
void __trace_end_binning_ib(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
) {
   struct trace_end_binning_ib entry;
   UNUSED struct trace_end_binning_ib *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_binning_ib *)u_trace_appendv(ut, cs, &__tp_end_binning_ib,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_binning_ib(ut->utctx, cs, __entry);
}

/*
 * start_draw_ib_sysmem
 */
static void __print_start_draw_ib_sysmem(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_draw_ib_sysmem *__entry =
      (const struct trace_start_draw_ib_sysmem *)arg;
   fprintf(out, ""
      "command_buffer_handle=%" PRIu64 ", "
         "\n"
   ,(uint64_t)__entry->command_buffer_handle
   );
}

static void __print_json_start_draw_ib_sysmem(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_draw_ib_sysmem *__entry =
      (const struct trace_start_draw_ib_sysmem *)arg;
   fprintf(out, ""
      "\"command_buffer_handle\": \"%" PRIu64 "\""
   ,(uint64_t)__entry->command_buffer_handle
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_draw_ib_sysmem(struct u_trace_context *utctx, void *cs, struct trace_start_draw_ib_sysmem *entry) {
   tu_cs_trace_start(utctx, cs, "start_draw_ib_sysmem("
      "command_buffer_handle=%" PRIu64 ""
      ")"
      ,(uint64_t)entry->command_buffer_handle
   );
}

static const struct u_tracepoint __tp_start_draw_ib_sysmem = {
    "start_draw_ib_sysmem",
    ALIGN_POT(sizeof(struct trace_start_draw_ib_sysmem), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    10,
    __print_start_draw_ib_sysmem,
    __print_json_start_draw_ib_sysmem,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_draw_ib_sysmem,
#endif
};
void __trace_start_draw_ib_sysmem(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
) {
   struct trace_start_draw_ib_sysmem entry;
   UNUSED struct trace_start_draw_ib_sysmem *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_draw_ib_sysmem *)u_trace_appendv(ut, cs, &__tp_start_draw_ib_sysmem,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_draw_ib_sysmem(ut->utctx, cs, __entry);
}

/*
 * end_draw_ib_sysmem
 */
#define __print_end_draw_ib_sysmem NULL
#define __print_json_end_draw_ib_sysmem NULL

__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_draw_ib_sysmem(struct u_trace_context *utctx, void *cs, struct trace_end_draw_ib_sysmem *entry) {
   tu_cs_trace_end(utctx, cs, "end_draw_ib_sysmem("
      ")"
   );
}

static const struct u_tracepoint __tp_end_draw_ib_sysmem = {
    "end_draw_ib_sysmem",
    ALIGN_POT(sizeof(struct trace_end_draw_ib_sysmem), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    11,
    __print_end_draw_ib_sysmem,
    __print_json_end_draw_ib_sysmem,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_draw_ib_sysmem,
#endif
};
void __trace_end_draw_ib_sysmem(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
) {
   struct trace_end_draw_ib_sysmem entry;
   UNUSED struct trace_end_draw_ib_sysmem *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_draw_ib_sysmem *)u_trace_appendv(ut, cs, &__tp_end_draw_ib_sysmem,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_draw_ib_sysmem(ut->utctx, cs, __entry);
}

/*
 * start_draw_ib_gmem
 */
static void __print_start_draw_ib_gmem(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_draw_ib_gmem *__entry =
      (const struct trace_start_draw_ib_gmem *)arg;
   fprintf(out, ""
      "command_buffer_handle=%" PRIu64 ", "
         "\n"
   ,(uint64_t)__entry->command_buffer_handle
   );
}

static void __print_json_start_draw_ib_gmem(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_draw_ib_gmem *__entry =
      (const struct trace_start_draw_ib_gmem *)arg;
   fprintf(out, ""
      "\"command_buffer_handle\": \"%" PRIu64 "\""
   ,(uint64_t)__entry->command_buffer_handle
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_draw_ib_gmem(struct u_trace_context *utctx, void *cs, struct trace_start_draw_ib_gmem *entry) {
   tu_cs_trace_start(utctx, cs, "start_draw_ib_gmem("
      "command_buffer_handle=%" PRIu64 ""
      ")"
      ,(uint64_t)entry->command_buffer_handle
   );
}

static const struct u_tracepoint __tp_start_draw_ib_gmem = {
    "start_draw_ib_gmem",
    ALIGN_POT(sizeof(struct trace_start_draw_ib_gmem), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    12,
    __print_start_draw_ib_gmem,
    __print_json_start_draw_ib_gmem,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_draw_ib_gmem,
#endif
};
void __trace_start_draw_ib_gmem(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
) {
   struct trace_start_draw_ib_gmem entry;
   UNUSED struct trace_start_draw_ib_gmem *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_draw_ib_gmem *)u_trace_appendv(ut, cs, &__tp_start_draw_ib_gmem,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_draw_ib_gmem(ut->utctx, cs, __entry);
}

/*
 * end_draw_ib_gmem
 */
#define __print_end_draw_ib_gmem NULL
#define __print_json_end_draw_ib_gmem NULL

__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_draw_ib_gmem(struct u_trace_context *utctx, void *cs, struct trace_end_draw_ib_gmem *entry) {
   tu_cs_trace_end(utctx, cs, "end_draw_ib_gmem("
      ")"
   );
}

static const struct u_tracepoint __tp_end_draw_ib_gmem = {
    "end_draw_ib_gmem",
    ALIGN_POT(sizeof(struct trace_end_draw_ib_gmem), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    13,
    __print_end_draw_ib_gmem,
    __print_json_end_draw_ib_gmem,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_draw_ib_gmem,
#endif
};
void __trace_end_draw_ib_gmem(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
) {
   struct trace_end_draw_ib_gmem entry;
   UNUSED struct trace_end_draw_ib_gmem *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_draw_ib_gmem *)u_trace_appendv(ut, cs, &__tp_end_draw_ib_gmem,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_draw_ib_gmem(ut->utctx, cs, __entry);
}

/*
 * start_generic_clear
 */
static void __print_start_generic_clear(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_generic_clear *__entry =
      (const struct trace_start_generic_clear *)arg;
   fprintf(out, ""
      "format=%s, "
      "ubwc=%s, "
      "samples=%u, "
      "command_buffer_handle=%" PRIu64 ", "
         "\n"
   ,vk_format_description(__entry->format)->short_name
   ,(__entry->ubwc ? "true" : "false")
   ,__entry->samples
   ,(uint64_t)__entry->command_buffer_handle
   );
}

static void __print_json_start_generic_clear(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_generic_clear *__entry =
      (const struct trace_start_generic_clear *)arg;
   fprintf(out, ""
      "\"format\": \"%s\""
         ", "
      "\"ubwc\": \"%s\""
         ", "
      "\"samples\": \"%u\""
         ", "
      "\"command_buffer_handle\": \"%" PRIu64 "\""
   ,vk_format_description(__entry->format)->short_name
   ,(__entry->ubwc ? "true" : "false")
   ,__entry->samples
   ,(uint64_t)__entry->command_buffer_handle
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_generic_clear(struct u_trace_context *utctx, void *cs, struct trace_start_generic_clear *entry) {
   tu_cs_trace_start(utctx, cs, "start_generic_clear("
      "format=%s"
      ",ubwc=%s"
      ",samples=%u"
      ",command_buffer_handle=%" PRIu64 ""
      ")"
      ,vk_format_description(entry->format)->short_name
      ,(entry->ubwc ? "true" : "false")
      ,entry->samples
      ,(uint64_t)entry->command_buffer_handle
   );
}

static const struct u_tracepoint __tp_start_generic_clear = {
    "start_generic_clear",
    ALIGN_POT(sizeof(struct trace_start_generic_clear), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    14,
    __print_start_generic_clear,
    __print_json_start_generic_clear,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_generic_clear,
#endif
};
void __trace_start_generic_clear(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
   , enum VkFormat format
   , bool ubwc
   , uint8_t samples
) {
   struct trace_start_generic_clear entry;
   UNUSED struct trace_start_generic_clear *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_generic_clear *)u_trace_appendv(ut, cs, &__tp_start_generic_clear,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   __entry->format = format;
   __entry->ubwc = ubwc;
   __entry->samples = samples;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_generic_clear(ut->utctx, cs, __entry);
}

/*
 * end_generic_clear
 */
#define __print_end_generic_clear NULL
#define __print_json_end_generic_clear NULL

__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_generic_clear(struct u_trace_context *utctx, void *cs, struct trace_end_generic_clear *entry) {
   tu_cs_trace_end(utctx, cs, "end_generic_clear("
      ")"
   );
}

static const struct u_tracepoint __tp_end_generic_clear = {
    "end_generic_clear",
    ALIGN_POT(sizeof(struct trace_end_generic_clear), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    15,
    __print_end_generic_clear,
    __print_json_end_generic_clear,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_generic_clear,
#endif
};
void __trace_end_generic_clear(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
) {
   struct trace_end_generic_clear entry;
   UNUSED struct trace_end_generic_clear *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_generic_clear *)u_trace_appendv(ut, cs, &__tp_end_generic_clear,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_generic_clear(ut->utctx, cs, __entry);
}

/*
 * start_gmem_clear
 */
static void __print_start_gmem_clear(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_gmem_clear *__entry =
      (const struct trace_start_gmem_clear *)arg;
   fprintf(out, ""
      "format=%s, "
      "samples=%u, "
      "command_buffer_handle=%" PRIu64 ", "
         "\n"
   ,vk_format_description(__entry->format)->short_name
   ,__entry->samples
   ,(uint64_t)__entry->command_buffer_handle
   );
}

static void __print_json_start_gmem_clear(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_gmem_clear *__entry =
      (const struct trace_start_gmem_clear *)arg;
   fprintf(out, ""
      "\"format\": \"%s\""
         ", "
      "\"samples\": \"%u\""
         ", "
      "\"command_buffer_handle\": \"%" PRIu64 "\""
   ,vk_format_description(__entry->format)->short_name
   ,__entry->samples
   ,(uint64_t)__entry->command_buffer_handle
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_gmem_clear(struct u_trace_context *utctx, void *cs, struct trace_start_gmem_clear *entry) {
   tu_cs_trace_start(utctx, cs, "start_gmem_clear("
      "format=%s"
      ",samples=%u"
      ",command_buffer_handle=%" PRIu64 ""
      ")"
      ,vk_format_description(entry->format)->short_name
      ,entry->samples
      ,(uint64_t)entry->command_buffer_handle
   );
}

static const struct u_tracepoint __tp_start_gmem_clear = {
    "start_gmem_clear",
    ALIGN_POT(sizeof(struct trace_start_gmem_clear), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    16,
    __print_start_gmem_clear,
    __print_json_start_gmem_clear,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_gmem_clear,
#endif
};
void __trace_start_gmem_clear(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
   , enum VkFormat format
   , uint8_t samples
) {
   struct trace_start_gmem_clear entry;
   UNUSED struct trace_start_gmem_clear *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_gmem_clear *)u_trace_appendv(ut, cs, &__tp_start_gmem_clear,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   __entry->format = format;
   __entry->samples = samples;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_gmem_clear(ut->utctx, cs, __entry);
}

/*
 * end_gmem_clear
 */
#define __print_end_gmem_clear NULL
#define __print_json_end_gmem_clear NULL

__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_gmem_clear(struct u_trace_context *utctx, void *cs, struct trace_end_gmem_clear *entry) {
   tu_cs_trace_end(utctx, cs, "end_gmem_clear("
      ")"
   );
}

static const struct u_tracepoint __tp_end_gmem_clear = {
    "end_gmem_clear",
    ALIGN_POT(sizeof(struct trace_end_gmem_clear), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    17,
    __print_end_gmem_clear,
    __print_json_end_gmem_clear,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_gmem_clear,
#endif
};
void __trace_end_gmem_clear(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
) {
   struct trace_end_gmem_clear entry;
   UNUSED struct trace_end_gmem_clear *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_gmem_clear *)u_trace_appendv(ut, cs, &__tp_end_gmem_clear,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_gmem_clear(ut->utctx, cs, __entry);
}

/*
 * start_sysmem_clear
 */
static void __print_start_sysmem_clear(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_sysmem_clear *__entry =
      (const struct trace_start_sysmem_clear *)arg;
   fprintf(out, ""
      "format=%s, "
      "uses_3d_ops=%u, "
      "samples=%u, "
      "command_buffer_handle=%" PRIu64 ", "
         "\n"
   ,vk_format_description(__entry->format)->short_name
   ,__entry->uses_3d_ops
   ,__entry->samples
   ,(uint64_t)__entry->command_buffer_handle
   );
}

static void __print_json_start_sysmem_clear(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_sysmem_clear *__entry =
      (const struct trace_start_sysmem_clear *)arg;
   fprintf(out, ""
      "\"format\": \"%s\""
         ", "
      "\"uses_3d_ops\": \"%u\""
         ", "
      "\"samples\": \"%u\""
         ", "
      "\"command_buffer_handle\": \"%" PRIu64 "\""
   ,vk_format_description(__entry->format)->short_name
   ,__entry->uses_3d_ops
   ,__entry->samples
   ,(uint64_t)__entry->command_buffer_handle
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_sysmem_clear(struct u_trace_context *utctx, void *cs, struct trace_start_sysmem_clear *entry) {
   tu_cs_trace_start(utctx, cs, "start_sysmem_clear("
      "format=%s"
      ",uses_3d_ops=%u"
      ",samples=%u"
      ",command_buffer_handle=%" PRIu64 ""
      ")"
      ,vk_format_description(entry->format)->short_name
      ,entry->uses_3d_ops
      ,entry->samples
      ,(uint64_t)entry->command_buffer_handle
   );
}

static const struct u_tracepoint __tp_start_sysmem_clear = {
    "start_sysmem_clear",
    ALIGN_POT(sizeof(struct trace_start_sysmem_clear), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    18,
    __print_start_sysmem_clear,
    __print_json_start_sysmem_clear,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_sysmem_clear,
#endif
};
void __trace_start_sysmem_clear(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
   , enum VkFormat format
   , uint8_t uses_3d_ops
   , uint8_t samples
) {
   struct trace_start_sysmem_clear entry;
   UNUSED struct trace_start_sysmem_clear *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_sysmem_clear *)u_trace_appendv(ut, cs, &__tp_start_sysmem_clear,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   __entry->format = format;
   __entry->uses_3d_ops = uses_3d_ops;
   __entry->samples = samples;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_sysmem_clear(ut->utctx, cs, __entry);
}

/*
 * end_sysmem_clear
 */
#define __print_end_sysmem_clear NULL
#define __print_json_end_sysmem_clear NULL

__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_sysmem_clear(struct u_trace_context *utctx, void *cs, struct trace_end_sysmem_clear *entry) {
   tu_cs_trace_end(utctx, cs, "end_sysmem_clear("
      ")"
   );
}

static const struct u_tracepoint __tp_end_sysmem_clear = {
    "end_sysmem_clear",
    ALIGN_POT(sizeof(struct trace_end_sysmem_clear), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    19,
    __print_end_sysmem_clear,
    __print_json_end_sysmem_clear,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_sysmem_clear,
#endif
};
void __trace_end_sysmem_clear(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
) {
   struct trace_end_sysmem_clear entry;
   UNUSED struct trace_end_sysmem_clear *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_sysmem_clear *)u_trace_appendv(ut, cs, &__tp_end_sysmem_clear,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_sysmem_clear(ut->utctx, cs, __entry);
}

/*
 * start_sysmem_clear_all
 */
static void __print_start_sysmem_clear_all(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_sysmem_clear_all *__entry =
      (const struct trace_start_sysmem_clear_all *)arg;
   fprintf(out, ""
      "mrt_count=%u, "
      "rect_count=%u, "
      "command_buffer_handle=%" PRIu64 ", "
         "\n"
   ,__entry->mrt_count
   ,__entry->rect_count
   ,(uint64_t)__entry->command_buffer_handle
   );
}

static void __print_json_start_sysmem_clear_all(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_sysmem_clear_all *__entry =
      (const struct trace_start_sysmem_clear_all *)arg;
   fprintf(out, ""
      "\"mrt_count\": \"%u\""
         ", "
      "\"rect_count\": \"%u\""
         ", "
      "\"command_buffer_handle\": \"%" PRIu64 "\""
   ,__entry->mrt_count
   ,__entry->rect_count
   ,(uint64_t)__entry->command_buffer_handle
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_sysmem_clear_all(struct u_trace_context *utctx, void *cs, struct trace_start_sysmem_clear_all *entry) {
   tu_cs_trace_start(utctx, cs, "start_sysmem_clear_all("
      "mrt_count=%u"
      ",rect_count=%u"
      ",command_buffer_handle=%" PRIu64 ""
      ")"
      ,entry->mrt_count
      ,entry->rect_count
      ,(uint64_t)entry->command_buffer_handle
   );
}

static const struct u_tracepoint __tp_start_sysmem_clear_all = {
    "start_sysmem_clear_all",
    ALIGN_POT(sizeof(struct trace_start_sysmem_clear_all), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    20,
    __print_start_sysmem_clear_all,
    __print_json_start_sysmem_clear_all,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_sysmem_clear_all,
#endif
};
void __trace_start_sysmem_clear_all(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
   , uint8_t mrt_count
   , uint8_t rect_count
) {
   struct trace_start_sysmem_clear_all entry;
   UNUSED struct trace_start_sysmem_clear_all *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_sysmem_clear_all *)u_trace_appendv(ut, cs, &__tp_start_sysmem_clear_all,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   __entry->mrt_count = mrt_count;
   __entry->rect_count = rect_count;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_sysmem_clear_all(ut->utctx, cs, __entry);
}

/*
 * end_sysmem_clear_all
 */
#define __print_end_sysmem_clear_all NULL
#define __print_json_end_sysmem_clear_all NULL

__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_sysmem_clear_all(struct u_trace_context *utctx, void *cs, struct trace_end_sysmem_clear_all *entry) {
   tu_cs_trace_end(utctx, cs, "end_sysmem_clear_all("
      ")"
   );
}

static const struct u_tracepoint __tp_end_sysmem_clear_all = {
    "end_sysmem_clear_all",
    ALIGN_POT(sizeof(struct trace_end_sysmem_clear_all), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    21,
    __print_end_sysmem_clear_all,
    __print_json_end_sysmem_clear_all,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_sysmem_clear_all,
#endif
};
void __trace_end_sysmem_clear_all(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
) {
   struct trace_end_sysmem_clear_all entry;
   UNUSED struct trace_end_sysmem_clear_all *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_sysmem_clear_all *)u_trace_appendv(ut, cs, &__tp_end_sysmem_clear_all,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_sysmem_clear_all(ut->utctx, cs, __entry);
}

/*
 * start_gmem_load
 */
static void __print_start_gmem_load(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_gmem_load *__entry =
      (const struct trace_start_gmem_load *)arg;
   fprintf(out, ""
      "format=%s, "
      "force_load=%u, "
      "command_buffer_handle=%" PRIu64 ", "
         "\n"
   ,vk_format_description(__entry->format)->short_name
   ,__entry->force_load
   ,(uint64_t)__entry->command_buffer_handle
   );
}

static void __print_json_start_gmem_load(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_gmem_load *__entry =
      (const struct trace_start_gmem_load *)arg;
   fprintf(out, ""
      "\"format\": \"%s\""
         ", "
      "\"force_load\": \"%u\""
         ", "
      "\"command_buffer_handle\": \"%" PRIu64 "\""
   ,vk_format_description(__entry->format)->short_name
   ,__entry->force_load
   ,(uint64_t)__entry->command_buffer_handle
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_gmem_load(struct u_trace_context *utctx, void *cs, struct trace_start_gmem_load *entry) {
   tu_cs_trace_start(utctx, cs, "start_gmem_load("
      "format=%s"
      ",force_load=%u"
      ",command_buffer_handle=%" PRIu64 ""
      ")"
      ,vk_format_description(entry->format)->short_name
      ,entry->force_load
      ,(uint64_t)entry->command_buffer_handle
   );
}

static const struct u_tracepoint __tp_start_gmem_load = {
    "start_gmem_load",
    ALIGN_POT(sizeof(struct trace_start_gmem_load), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    22,
    __print_start_gmem_load,
    __print_json_start_gmem_load,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_gmem_load,
#endif
};
void __trace_start_gmem_load(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
   , enum VkFormat format
   , uint8_t force_load
) {
   struct trace_start_gmem_load entry;
   UNUSED struct trace_start_gmem_load *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_gmem_load *)u_trace_appendv(ut, cs, &__tp_start_gmem_load,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   __entry->format = format;
   __entry->force_load = force_load;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_gmem_load(ut->utctx, cs, __entry);
}

/*
 * end_gmem_load
 */
#define __print_end_gmem_load NULL
#define __print_json_end_gmem_load NULL

__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_gmem_load(struct u_trace_context *utctx, void *cs, struct trace_end_gmem_load *entry) {
   tu_cs_trace_end(utctx, cs, "end_gmem_load("
      ")"
   );
}

static const struct u_tracepoint __tp_end_gmem_load = {
    "end_gmem_load",
    ALIGN_POT(sizeof(struct trace_end_gmem_load), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    23,
    __print_end_gmem_load,
    __print_json_end_gmem_load,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_gmem_load,
#endif
};
void __trace_end_gmem_load(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
) {
   struct trace_end_gmem_load entry;
   UNUSED struct trace_end_gmem_load *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_gmem_load *)u_trace_appendv(ut, cs, &__tp_end_gmem_load,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_gmem_load(ut->utctx, cs, __entry);
}

/*
 * start_gmem_store
 */
static void __print_start_gmem_store(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_gmem_store *__entry =
      (const struct trace_start_gmem_store *)arg;
   fprintf(out, ""
      "format=%s, "
      "fast_path=%u, "
      "unaligned=%u, "
      "command_buffer_handle=%" PRIu64 ", "
         "\n"
   ,vk_format_description(__entry->format)->short_name
   ,__entry->fast_path
   ,__entry->unaligned
   ,(uint64_t)__entry->command_buffer_handle
   );
}

static void __print_json_start_gmem_store(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_gmem_store *__entry =
      (const struct trace_start_gmem_store *)arg;
   fprintf(out, ""
      "\"format\": \"%s\""
         ", "
      "\"fast_path\": \"%u\""
         ", "
      "\"unaligned\": \"%u\""
         ", "
      "\"command_buffer_handle\": \"%" PRIu64 "\""
   ,vk_format_description(__entry->format)->short_name
   ,__entry->fast_path
   ,__entry->unaligned
   ,(uint64_t)__entry->command_buffer_handle
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_gmem_store(struct u_trace_context *utctx, void *cs, struct trace_start_gmem_store *entry) {
   tu_cs_trace_start(utctx, cs, "start_gmem_store("
      "format=%s"
      ",fast_path=%u"
      ",unaligned=%u"
      ",command_buffer_handle=%" PRIu64 ""
      ")"
      ,vk_format_description(entry->format)->short_name
      ,entry->fast_path
      ,entry->unaligned
      ,(uint64_t)entry->command_buffer_handle
   );
}

static const struct u_tracepoint __tp_start_gmem_store = {
    "start_gmem_store",
    ALIGN_POT(sizeof(struct trace_start_gmem_store), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    24,
    __print_start_gmem_store,
    __print_json_start_gmem_store,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_gmem_store,
#endif
};
void __trace_start_gmem_store(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
   , enum VkFormat format
   , uint8_t fast_path
   , uint8_t unaligned
) {
   struct trace_start_gmem_store entry;
   UNUSED struct trace_start_gmem_store *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_gmem_store *)u_trace_appendv(ut, cs, &__tp_start_gmem_store,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   __entry->format = format;
   __entry->fast_path = fast_path;
   __entry->unaligned = unaligned;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_gmem_store(ut->utctx, cs, __entry);
}

/*
 * end_gmem_store
 */
#define __print_end_gmem_store NULL
#define __print_json_end_gmem_store NULL

__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_gmem_store(struct u_trace_context *utctx, void *cs, struct trace_end_gmem_store *entry) {
   tu_cs_trace_end(utctx, cs, "end_gmem_store("
      ")"
   );
}

static const struct u_tracepoint __tp_end_gmem_store = {
    "end_gmem_store",
    ALIGN_POT(sizeof(struct trace_end_gmem_store), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    25,
    __print_end_gmem_store,
    __print_json_end_gmem_store,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_gmem_store,
#endif
};
void __trace_end_gmem_store(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
) {
   struct trace_end_gmem_store entry;
   UNUSED struct trace_end_gmem_store *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_gmem_store *)u_trace_appendv(ut, cs, &__tp_end_gmem_store,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_gmem_store(ut->utctx, cs, __entry);
}

/*
 * start_sysmem_resolve
 */
static void __print_start_sysmem_resolve(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_sysmem_resolve *__entry =
      (const struct trace_start_sysmem_resolve *)arg;
   fprintf(out, ""
      "format=%s, "
      "command_buffer_handle=%" PRIu64 ", "
         "\n"
   ,vk_format_description(__entry->format)->short_name
   ,(uint64_t)__entry->command_buffer_handle
   );
}

static void __print_json_start_sysmem_resolve(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_sysmem_resolve *__entry =
      (const struct trace_start_sysmem_resolve *)arg;
   fprintf(out, ""
      "\"format\": \"%s\""
         ", "
      "\"command_buffer_handle\": \"%" PRIu64 "\""
   ,vk_format_description(__entry->format)->short_name
   ,(uint64_t)__entry->command_buffer_handle
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_sysmem_resolve(struct u_trace_context *utctx, void *cs, struct trace_start_sysmem_resolve *entry) {
   tu_cs_trace_start(utctx, cs, "start_sysmem_resolve("
      "format=%s"
      ",command_buffer_handle=%" PRIu64 ""
      ")"
      ,vk_format_description(entry->format)->short_name
      ,(uint64_t)entry->command_buffer_handle
   );
}

static const struct u_tracepoint __tp_start_sysmem_resolve = {
    "start_sysmem_resolve",
    ALIGN_POT(sizeof(struct trace_start_sysmem_resolve), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    26,
    __print_start_sysmem_resolve,
    __print_json_start_sysmem_resolve,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_sysmem_resolve,
#endif
};
void __trace_start_sysmem_resolve(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
   , enum VkFormat format
) {
   struct trace_start_sysmem_resolve entry;
   UNUSED struct trace_start_sysmem_resolve *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_sysmem_resolve *)u_trace_appendv(ut, cs, &__tp_start_sysmem_resolve,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   __entry->format = format;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_sysmem_resolve(ut->utctx, cs, __entry);
}

/*
 * end_sysmem_resolve
 */
#define __print_end_sysmem_resolve NULL
#define __print_json_end_sysmem_resolve NULL

__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_sysmem_resolve(struct u_trace_context *utctx, void *cs, struct trace_end_sysmem_resolve *entry) {
   tu_cs_trace_end(utctx, cs, "end_sysmem_resolve("
      ")"
   );
}

static const struct u_tracepoint __tp_end_sysmem_resolve = {
    "end_sysmem_resolve",
    ALIGN_POT(sizeof(struct trace_end_sysmem_resolve), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    27,
    __print_end_sysmem_resolve,
    __print_json_end_sysmem_resolve,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_sysmem_resolve,
#endif
};
void __trace_end_sysmem_resolve(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
) {
   struct trace_end_sysmem_resolve entry;
   UNUSED struct trace_end_sysmem_resolve *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_sysmem_resolve *)u_trace_appendv(ut, cs, &__tp_end_sysmem_resolve,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_sysmem_resolve(ut->utctx, cs, __entry);
}

/*
 * start_blit
 */
static void __print_start_blit(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_blit *__entry =
      (const struct trace_start_blit *)arg;
   fprintf(out, ""
      "uses_3d_blit=%u, "
      "src_format=%s, "
      "dst_format=%s, "
      "layers=%u, "
      "command_buffer_handle=%" PRIu64 ", "
         "\n"
   ,__entry->uses_3d_blit
   ,vk_format_description(__entry->src_format)->short_name
   ,vk_format_description(__entry->dst_format)->short_name
   ,__entry->layers
   ,(uint64_t)__entry->command_buffer_handle
   );
}

static void __print_json_start_blit(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_blit *__entry =
      (const struct trace_start_blit *)arg;
   fprintf(out, ""
      "\"uses_3d_blit\": \"%u\""
         ", "
      "\"src_format\": \"%s\""
         ", "
      "\"dst_format\": \"%s\""
         ", "
      "\"layers\": \"%u\""
         ", "
      "\"command_buffer_handle\": \"%" PRIu64 "\""
   ,__entry->uses_3d_blit
   ,vk_format_description(__entry->src_format)->short_name
   ,vk_format_description(__entry->dst_format)->short_name
   ,__entry->layers
   ,(uint64_t)__entry->command_buffer_handle
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_blit(struct u_trace_context *utctx, void *cs, struct trace_start_blit *entry) {
   tu_cs_trace_start(utctx, cs, "start_blit("
      "uses_3d_blit=%u"
      ",src_format=%s"
      ",dst_format=%s"
      ",layers=%u"
      ",command_buffer_handle=%" PRIu64 ""
      ")"
      ,entry->uses_3d_blit
      ,vk_format_description(entry->src_format)->short_name
      ,vk_format_description(entry->dst_format)->short_name
      ,entry->layers
      ,(uint64_t)entry->command_buffer_handle
   );
}

static const struct u_tracepoint __tp_start_blit = {
    "start_blit",
    ALIGN_POT(sizeof(struct trace_start_blit), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    28,
    __print_start_blit,
    __print_json_start_blit,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_blit,
#endif
};
void __trace_start_blit(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
   , uint8_t uses_3d_blit
   , enum VkFormat src_format
   , enum VkFormat dst_format
   , uint8_t layers
) {
   struct trace_start_blit entry;
   UNUSED struct trace_start_blit *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_blit *)u_trace_appendv(ut, cs, &__tp_start_blit,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   __entry->uses_3d_blit = uses_3d_blit;
   __entry->src_format = src_format;
   __entry->dst_format = dst_format;
   __entry->layers = layers;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_blit(ut->utctx, cs, __entry);
}

/*
 * end_blit
 */
#define __print_end_blit NULL
#define __print_json_end_blit NULL

__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_blit(struct u_trace_context *utctx, void *cs, struct trace_end_blit *entry) {
   tu_cs_trace_end(utctx, cs, "end_blit("
      ")"
   );
}

static const struct u_tracepoint __tp_end_blit = {
    "end_blit",
    ALIGN_POT(sizeof(struct trace_end_blit), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    29,
    __print_end_blit,
    __print_json_end_blit,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_blit,
#endif
};
void __trace_end_blit(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
) {
   struct trace_end_blit entry;
   UNUSED struct trace_end_blit *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_blit *)u_trace_appendv(ut, cs, &__tp_end_blit,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_blit(ut->utctx, cs, __entry);
}

/*
 * start_compute
 */
static void __print_start_compute(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_compute *__entry =
      (const struct trace_start_compute *)arg;
   fprintf(out, ""
      "indirect=%u, "
      "unaligned=%u, "
      "local_size_x=%u, "
      "local_size_y=%u, "
      "local_size_z=%u, "
      "num_groups_x=%u, "
      "num_groups_y=%u, "
      "num_groups_z=%u, "
      "cs_sha1=%s, "
      "command_buffer_handle=%" PRIu64 ", "
         "\n"
   ,__entry->indirect
   ,__entry->unaligned
   ,__entry->local_size_x
   ,__entry->local_size_y
   ,__entry->local_size_z
   ,__entry->num_groups_x
   ,__entry->num_groups_y
   ,__entry->num_groups_z
   ,__entry->cs_sha1
   ,(uint64_t)__entry->command_buffer_handle
   );
}

static void __print_json_start_compute(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_compute *__entry =
      (const struct trace_start_compute *)arg;
   fprintf(out, ""
      "\"indirect\": \"%u\""
         ", "
      "\"unaligned\": \"%u\""
         ", "
      "\"local_size_x\": \"%u\""
         ", "
      "\"local_size_y\": \"%u\""
         ", "
      "\"local_size_z\": \"%u\""
         ", "
      "\"num_groups_x\": \"%u\""
         ", "
      "\"num_groups_y\": \"%u\""
         ", "
      "\"num_groups_z\": \"%u\""
         ", "
      "\"cs_sha1\": \"%s\""
         ", "
      "\"command_buffer_handle\": \"%" PRIu64 "\""
   ,__entry->indirect
   ,__entry->unaligned
   ,__entry->local_size_x
   ,__entry->local_size_y
   ,__entry->local_size_z
   ,__entry->num_groups_x
   ,__entry->num_groups_y
   ,__entry->num_groups_z
   ,__entry->cs_sha1
   ,(uint64_t)__entry->command_buffer_handle
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_compute(struct u_trace_context *utctx, void *cs, struct trace_start_compute *entry) {
   tu_cs_trace_start(utctx, cs, "start_compute("
      "indirect=%u"
      ",unaligned=%u"
      ",local_size_x=%u"
      ",local_size_y=%u"
      ",local_size_z=%u"
      ",num_groups_x=%u"
      ",num_groups_y=%u"
      ",num_groups_z=%u"
      ",cs_sha1=%s"
      ",command_buffer_handle=%" PRIu64 ""
      ")"
      ,entry->indirect
      ,entry->unaligned
      ,entry->local_size_x
      ,entry->local_size_y
      ,entry->local_size_z
      ,entry->num_groups_x
      ,entry->num_groups_y
      ,entry->num_groups_z
      ,entry->cs_sha1
      ,(uint64_t)entry->command_buffer_handle
   );
}

static const struct u_tracepoint __tp_start_compute = {
    "start_compute",
    ALIGN_POT(sizeof(struct trace_start_compute), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    30,
    __print_start_compute,
    __print_json_start_compute,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_compute,
#endif
};
void __trace_start_compute(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
   , uint8_t indirect
   , uint8_t unaligned
   , uint16_t local_size_x
   , uint16_t local_size_y
   , uint16_t local_size_z
   , uint16_t num_groups_x
   , uint16_t num_groups_y
   , uint16_t num_groups_z
   , tu_sha1_str cs_sha1
) {
   struct trace_start_compute entry;
   UNUSED struct trace_start_compute *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_compute *)u_trace_appendv(ut, cs, &__tp_start_compute,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   __entry->indirect = indirect;
   __entry->unaligned = unaligned;
   __entry->local_size_x = local_size_x;
   __entry->local_size_y = local_size_y;
   __entry->local_size_z = local_size_z;
   __entry->num_groups_x = num_groups_x;
   __entry->num_groups_y = num_groups_y;
   __entry->num_groups_z = num_groups_z;
     strcpy(__entry->cs_sha1, cs_sha1);
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_compute(ut->utctx, cs, __entry);
}

/*
 * end_compute
 */
#define __print_end_compute NULL
#define __print_json_end_compute NULL

__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_compute(struct u_trace_context *utctx, void *cs, struct trace_end_compute *entry) {
   tu_cs_trace_end(utctx, cs, "end_compute("
      ")"
   );
}

static const struct u_tracepoint __tp_end_compute = {
    "end_compute",
    ALIGN_POT(sizeof(struct trace_end_compute), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    31,
    __print_end_compute,
    __print_json_end_compute,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_compute,
#endif
};
void __trace_end_compute(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
) {
   struct trace_end_compute entry;
   UNUSED struct trace_end_compute *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_compute *)u_trace_appendv(ut, cs, &__tp_end_compute,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_compute(ut->utctx, cs, __entry);
}

/*
 * start_compute_indirect
 */
static void __print_start_compute_indirect(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_compute_indirect *__entry =
      (const struct trace_start_compute_indirect *)arg;
   fprintf(out, ""
      "unaligned=%u, "
      "cs_sha1=%s, "
      "command_buffer_handle=%" PRIu64 ", "
         "\n"
   ,__entry->unaligned
   ,__entry->cs_sha1
   ,(uint64_t)__entry->command_buffer_handle
   );
}

static void __print_json_start_compute_indirect(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_compute_indirect *__entry =
      (const struct trace_start_compute_indirect *)arg;
   fprintf(out, ""
      "\"unaligned\": \"%u\""
         ", "
      "\"cs_sha1\": \"%s\""
         ", "
      "\"command_buffer_handle\": \"%" PRIu64 "\""
   ,__entry->unaligned
   ,__entry->cs_sha1
   ,(uint64_t)__entry->command_buffer_handle
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_compute_indirect(struct u_trace_context *utctx, void *cs, struct trace_start_compute_indirect *entry) {
   tu_cs_trace_start(utctx, cs, "start_compute_indirect("
      "unaligned=%u"
      ",cs_sha1=%s"
      ",command_buffer_handle=%" PRIu64 ""
      ")"
      ,entry->unaligned
      ,entry->cs_sha1
      ,(uint64_t)entry->command_buffer_handle
   );
}

static const struct u_tracepoint __tp_start_compute_indirect = {
    "start_compute_indirect",
    ALIGN_POT(sizeof(struct trace_start_compute_indirect), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    32,
    __print_start_compute_indirect,
    __print_json_start_compute_indirect,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_compute_indirect,
#endif
};
void __trace_start_compute_indirect(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
   , uint8_t unaligned
   , tu_sha1_str cs_sha1
) {
   struct trace_start_compute_indirect entry;
   UNUSED struct trace_start_compute_indirect *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_compute_indirect *)u_trace_appendv(ut, cs, &__tp_start_compute_indirect,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   __entry->unaligned = unaligned;
     strcpy(__entry->cs_sha1, cs_sha1);
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_compute_indirect(ut->utctx, cs, __entry);
}

/*
 * end_compute_indirect
 */
static void __print_end_compute_indirect(FILE *out, const void *arg, const void *indirect) {
   const VkDispatchIndirectCommand *__size = (const VkDispatchIndirectCommand *) ((char *)indirect + 0);
   fprintf(out, ""
      "size=%ux%ux%u, "
         "\n"
   ,__size->x, __size->y, __size->z
   );
}

static void __print_json_end_compute_indirect(FILE *out, const void *arg, const void *indirect) {
   const VkDispatchIndirectCommand *__size = (const VkDispatchIndirectCommand *) ((char *)indirect + 0);
   fprintf(out, ""
      "\"size\": \"%ux%ux%u\""
   ,__size->x, __size->y, __size->z
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_compute_indirect(struct u_trace_context *utctx, void *cs, struct trace_end_compute_indirect *entry) {
   tu_cs_trace_end(utctx, cs, "end_compute_indirect("
      ")"
   );
}

static const struct u_tracepoint __tp_end_compute_indirect = {
    "end_compute_indirect",
    ALIGN_POT(sizeof(struct trace_end_compute_indirect), 8),   /* keep size 64b aligned */
    0
    + sizeof(VkDispatchIndirectCommand)
    ,
    0,
    33,
    __print_end_compute_indirect,
    __print_json_end_compute_indirect,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_compute_indirect,
#endif
};
void __trace_end_compute_indirect(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct u_trace_address size
) {
   struct trace_end_compute_indirect entry;
   struct u_trace_address indirects[] = {
      size,
   };
   uint8_t indirect_sizes[] = {
      sizeof(VkDispatchIndirectCommand),
   };
   UNUSED struct trace_end_compute_indirect *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_compute_indirect *)u_trace_appendv(ut, cs, &__tp_end_compute_indirect,
                                                    0
                                                    ,
                                                    ARRAY_SIZE(indirects), indirects, indirect_sizes
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_compute_indirect(ut->utctx, cs, __entry);
}

/*
 * start_cmd_buffer_annotation
 */
static void __print_start_cmd_buffer_annotation(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_cmd_buffer_annotation *__entry =
      (const struct trace_start_cmd_buffer_annotation *)arg;
   fprintf(out, ""
      "str=%s, "
      "command_buffer_handle=%" PRIu64 ", "
         "\n"
   ,__entry->str
   ,(uint64_t)__entry->command_buffer_handle
   );
}

static void __print_json_start_cmd_buffer_annotation(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_cmd_buffer_annotation *__entry =
      (const struct trace_start_cmd_buffer_annotation *)arg;
   fprintf(out, ""
      "\"str\": \"%s\""
         ", "
      "\"command_buffer_handle\": \"%" PRIu64 "\""
   ,__entry->str
   ,(uint64_t)__entry->command_buffer_handle
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_cmd_buffer_annotation(struct u_trace_context *utctx, void *cs, struct trace_start_cmd_buffer_annotation *entry) {
   tu_cs_trace_start(utctx, cs, "start_cmd_buffer_annotation("
      "str=%s"
      ",command_buffer_handle=%" PRIu64 ""
      ")"
      ,entry->str
      ,(uint64_t)entry->command_buffer_handle
   );
}

static const struct u_tracepoint __tp_start_cmd_buffer_annotation = {
    "start_cmd_buffer_annotation",
    ALIGN_POT(sizeof(struct trace_start_cmd_buffer_annotation), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    34,
    __print_start_cmd_buffer_annotation,
    __print_json_start_cmd_buffer_annotation,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_cmd_buffer_annotation,
#endif
};
void __trace_start_cmd_buffer_annotation(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
   , unsigned len
   , const char *str
) {
   struct trace_start_cmd_buffer_annotation entry;
   UNUSED struct trace_start_cmd_buffer_annotation *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_cmd_buffer_annotation *)u_trace_appendv(ut, cs, &__tp_start_cmd_buffer_annotation,
                                                    0
                                                    + len + 1
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   __entry->dummy = 0;
   strncpy(__entry->str, str, len + 1);
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_cmd_buffer_annotation(ut->utctx, cs, __entry);
}

/*
 * end_cmd_buffer_annotation
 */
#define __print_end_cmd_buffer_annotation NULL
#define __print_json_end_cmd_buffer_annotation NULL

__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_cmd_buffer_annotation(struct u_trace_context *utctx, void *cs, struct trace_end_cmd_buffer_annotation *entry) {
   tu_cs_trace_end(utctx, cs, "end_cmd_buffer_annotation("
      ")"
   );
}

static const struct u_tracepoint __tp_end_cmd_buffer_annotation = {
    "end_cmd_buffer_annotation",
    ALIGN_POT(sizeof(struct trace_end_cmd_buffer_annotation), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    35,
    __print_end_cmd_buffer_annotation,
    __print_json_end_cmd_buffer_annotation,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_cmd_buffer_annotation,
#endif
};
void __trace_end_cmd_buffer_annotation(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
) {
   struct trace_end_cmd_buffer_annotation entry;
   UNUSED struct trace_end_cmd_buffer_annotation *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_cmd_buffer_annotation *)u_trace_appendv(ut, cs, &__tp_end_cmd_buffer_annotation,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_cmd_buffer_annotation(ut->utctx, cs, __entry);
}

/*
 * start_cmd_buffer_annotation_rp
 */
static void __print_start_cmd_buffer_annotation_rp(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_cmd_buffer_annotation_rp *__entry =
      (const struct trace_start_cmd_buffer_annotation_rp *)arg;
   fprintf(out, ""
      "str=%s, "
      "command_buffer_handle=%" PRIu64 ", "
         "\n"
   ,__entry->str
   ,(uint64_t)__entry->command_buffer_handle
   );
}

static void __print_json_start_cmd_buffer_annotation_rp(FILE *out, const void *arg, const void *indirect) {
   const struct trace_start_cmd_buffer_annotation_rp *__entry =
      (const struct trace_start_cmd_buffer_annotation_rp *)arg;
   fprintf(out, ""
      "\"str\": \"%s\""
         ", "
      "\"command_buffer_handle\": \"%" PRIu64 "\""
   ,__entry->str
   ,(uint64_t)__entry->command_buffer_handle
   );
}


__attribute__((format(printf, 3, 4))) void tu_cs_trace_start(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_start_cmd_buffer_annotation_rp(struct u_trace_context *utctx, void *cs, struct trace_start_cmd_buffer_annotation_rp *entry) {
   tu_cs_trace_start(utctx, cs, "start_cmd_buffer_annotation_rp("
      "str=%s"
      ",command_buffer_handle=%" PRIu64 ""
      ")"
      ,entry->str
      ,(uint64_t)entry->command_buffer_handle
   );
}

static const struct u_tracepoint __tp_start_cmd_buffer_annotation_rp = {
    "start_cmd_buffer_annotation_rp",
    ALIGN_POT(sizeof(struct trace_start_cmd_buffer_annotation_rp), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    36,
    __print_start_cmd_buffer_annotation_rp,
    __print_json_start_cmd_buffer_annotation_rp,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_start_cmd_buffer_annotation_rp,
#endif
};
void __trace_start_cmd_buffer_annotation_rp(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
   , struct tu_cmd_buffer * cmd
   , unsigned len
   , const char *str
) {
   struct trace_start_cmd_buffer_annotation_rp entry;
   UNUSED struct trace_start_cmd_buffer_annotation_rp *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_start_cmd_buffer_annotation_rp *)u_trace_appendv(ut, cs, &__tp_start_cmd_buffer_annotation_rp,
                                                    0
                                                    + len + 1
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   __entry->command_buffer_handle = vk_command_buffer_to_handle(&cmd->vk);
   __entry->dummy = 0;
   strncpy(__entry->str, str, len + 1);
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_start_cmd_buffer_annotation_rp(ut->utctx, cs, __entry);
}

/*
 * end_cmd_buffer_annotation_rp
 */
#define __print_end_cmd_buffer_annotation_rp NULL
#define __print_json_end_cmd_buffer_annotation_rp NULL

__attribute__((format(printf, 3, 4))) void tu_cs_trace_end(struct u_trace_context *utctx, void *, const char *, ...);

static void __emit_label_end_cmd_buffer_annotation_rp(struct u_trace_context *utctx, void *cs, struct trace_end_cmd_buffer_annotation_rp *entry) {
   tu_cs_trace_end(utctx, cs, "end_cmd_buffer_annotation_rp("
      ")"
   );
}

static const struct u_tracepoint __tp_end_cmd_buffer_annotation_rp = {
    "end_cmd_buffer_annotation_rp",
    ALIGN_POT(sizeof(struct trace_end_cmd_buffer_annotation_rp), 8),   /* keep size 64b aligned */
    0
    ,
    0,
    37,
    __print_end_cmd_buffer_annotation_rp,
    __print_json_end_cmd_buffer_annotation_rp,
#ifdef HAVE_PERFETTO
    (void (*)(void *pctx, uint64_t, uint16_t, const void *, const void *, const void *))tu_perfetto_end_cmd_buffer_annotation_rp,
#endif
};
void __trace_end_cmd_buffer_annotation_rp(
     struct u_trace *ut
   , enum u_trace_type enabled_traces
   , void *cs
) {
   struct trace_end_cmd_buffer_annotation_rp entry;
   UNUSED struct trace_end_cmd_buffer_annotation_rp *__entry =
      enabled_traces & U_TRACE_TYPE_REQUIRE_QUEUING ?
      (struct trace_end_cmd_buffer_annotation_rp *)u_trace_appendv(ut, cs, &__tp_end_cmd_buffer_annotation_rp,
                                                    0
                                                    ,
                                                    0, NULL, NULL
                                                    ) :
      &entry;
   if (enabled_traces & U_TRACE_TYPE_MARKERS)
      __emit_label_end_cmd_buffer_annotation_rp(ut->utctx, cs, __entry);
}

