﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/application-insights/ApplicationInsightsRequest.h>
#include <aws/application-insights/ApplicationInsights_EXPORTS.h>
#include <aws/application-insights/model/ConfigurationEventStatus.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace ApplicationInsights {
namespace Model {

/**
 */
class ListConfigurationHistoryRequest : public ApplicationInsightsRequest {
 public:
  AWS_APPLICATIONINSIGHTS_API ListConfigurationHistoryRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListConfigurationHistory"; }

  AWS_APPLICATIONINSIGHTS_API Aws::String SerializePayload() const override;

  AWS_APPLICATIONINSIGHTS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Resource group to which the application belongs. </p>
   */
  inline const Aws::String& GetResourceGroupName() const { return m_resourceGroupName; }
  inline bool ResourceGroupNameHasBeenSet() const { return m_resourceGroupNameHasBeenSet; }
  template <typename ResourceGroupNameT = Aws::String>
  void SetResourceGroupName(ResourceGroupNameT&& value) {
    m_resourceGroupNameHasBeenSet = true;
    m_resourceGroupName = std::forward<ResourceGroupNameT>(value);
  }
  template <typename ResourceGroupNameT = Aws::String>
  ListConfigurationHistoryRequest& WithResourceGroupName(ResourceGroupNameT&& value) {
    SetResourceGroupName(std::forward<ResourceGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The start time of the event. </p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  ListConfigurationHistoryRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The end time of the event.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  ListConfigurationHistoryRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the configuration update event. Possible values include INFO,
   * WARN, and ERROR.</p>
   */
  inline ConfigurationEventStatus GetEventStatus() const { return m_eventStatus; }
  inline bool EventStatusHasBeenSet() const { return m_eventStatusHasBeenSet; }
  inline void SetEventStatus(ConfigurationEventStatus value) {
    m_eventStatusHasBeenSet = true;
    m_eventStatus = value;
  }
  inline ListConfigurationHistoryRequest& WithEventStatus(ConfigurationEventStatus value) {
    SetEventStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The maximum number of results returned by
   * <code>ListConfigurationHistory</code> in paginated output. When this parameter
   * is used, <code>ListConfigurationHistory</code> returns only
   * <code>MaxResults</code> in a single page along with a <code>NextToken</code>
   * response element. The remaining results of the initial request can be seen by
   * sending another <code>ListConfigurationHistory</code> request with the returned
   * <code>NextToken</code> value. If this parameter is not used, then
   * <code>ListConfigurationHistory</code> returns all results. </p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListConfigurationHistoryRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>NextToken</code> value returned from a previous paginated
   * <code>ListConfigurationHistory</code> request where <code>MaxResults</code> was
   * used and the results exceeded the value of that parameter. Pagination continues
   * from the end of the previous results that returned the <code>NextToken</code>
   * value. This value is <code>null</code> when there are no more results to
   * return.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListConfigurationHistoryRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services account ID for the resource group owner.</p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  ListConfigurationHistoryRequest& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_resourceGroupName;
  bool m_resourceGroupNameHasBeenSet = false;

  Aws::Utils::DateTime m_startTime{};
  bool m_startTimeHasBeenSet = false;

  Aws::Utils::DateTime m_endTime{};
  bool m_endTimeHasBeenSet = false;

  ConfigurationEventStatus m_eventStatus{ConfigurationEventStatus::NOT_SET};
  bool m_eventStatusHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  Aws::String m_accountId;
  bool m_accountIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace ApplicationInsights
}  // namespace Aws
