from __future__ import annotations


__copyright__ = "Copyright (C) 2013 Andreas Kloeckner"

__license__ = """
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""


from typing import TYPE_CHECKING, TextIO, cast

import numpy as np

from pytools.py_codegen import Indentation, PythonCodeGenerator

import pyopencl as cl


if TYPE_CHECKING:
    from numpy.typing import DTypeLike

    from pyopencl.typing import KernelArg, WaitList


def capture_kernel_call(
            kernel: cl.Kernel,
            output_file: str | TextIO,
            queue: cl.CommandQueue,
            g_size: tuple[int, ...],
            l_size: tuple[int, ...] | None,
            *args: KernelArg,
            wait_for: WaitList = None,  # pyright: ignore[reportUnusedParameter]
            g_times_l: bool = False,
            allow_empty_ndrange: bool = False,
            global_offset: tuple[int, ...] | None = None,
        ) -> None:
    try:
        source = cast("str | None", kernel._source)  # pyright: ignore[reportAttributeAccessIssue]
    except AttributeError as err:
        raise RuntimeError("cannot capture call, kernel source not available") from err

    if source is None:
        raise RuntimeError("cannot capture call, kernel source not available")

    cg = PythonCodeGenerator()

    cg("# generated by pyopencl.capture_call")
    cg("")
    cg("import numpy as np")
    cg("import pyopencl as cl")
    cg("from base64 import b64decode")
    cg("from zlib import decompress")
    cg("mf = cl.mem_flags")
    cg("")

    cg('CODE = r"""//CL//')
    for line in source.split("\n"):
        cg(line)
    cg('"""')

    # {{{ invocation

    arg_data: list[tuple[str, memoryview | bytearray]] = []

    cg("")
    cg("")
    cg("def main():")
    with Indentation(cg):
        cg("ctx = cl.create_some_context()")
        cg("queue = cl.CommandQueue(ctx)")
        cg("")

        kernel_args: list[str] = []

        for i, arg in enumerate(args):
            if isinstance(arg, cl.Buffer):
                buf = bytearray(arg.size)
                cl.enqueue_copy(queue, buf, arg)
                arg_data.append(("arg%d_data" % i, buf))
                cg("arg%d = cl.Buffer(ctx, "
                        "mf.READ_WRITE | cl.mem_flags.COPY_HOST_PTR,"
                        % i)
                cg("    hostbuf=decompress(b64decode(arg%d_data)))"
                        % i)
                kernel_args.append("arg%d" % i)
            elif isinstance(arg, (int, float)):
                kernel_args.append(repr(arg))
            elif isinstance(arg, np.integer):
                kernel_args.append("np.{}({})".format(
                    arg.dtype.type.__name__, repr(int(arg))))
            elif isinstance(arg, np.floating):
                kernel_args.append("np.{}({})".format(
                    arg.dtype.type.__name__, repr(float(arg))))
            elif isinstance(arg, np.complexfloating):
                kernel_args.append("np.{}({})".format(
                    arg.dtype.type.__name__, repr(complex(arg))))
            else:
                try:
                    arg_buf = memoryview(arg)
                except Exception as err:
                    raise RuntimeError("cannot capture: "
                            "unsupported arg nr %d (0-based)" % i) from err

                arg_data.append(("arg%d_data" % i, arg_buf))
                kernel_args.append("decompress(b64decode(arg%d_data))" % i)

        cg("")

        if g_times_l:
            assert l_size is not None
            dim = max(len(g_size), len(l_size))
            l_size = l_size + (1,) * (dim-len(l_size))
            g_size = g_size + (1,) * (dim-len(g_size))
            g_size = tuple(
                    gs*ls for gs, ls in zip(g_size, l_size, strict=True))

        if global_offset is not None:
            kernel_args.append("global_offset=%s" % repr(global_offset))
        if allow_empty_ndrange:
            kernel_args.append("allow_empty_ndrange=%s" % repr(allow_empty_ndrange))

        cg("prg = cl.Program(ctx, CODE).build()")
        cg("knl = prg.%s" % kernel.function_name)
        if hasattr(kernel, "_scalar_arg_dtypes"):
            def strify_dtype(d: DTypeLike):
                if d is None:
                    return "None"

                d = np.dtype(d)
                s = repr(d)
                if s.startswith("dtype"):
                    s = "np."+s

                return s

            cg("knl.set_scalar_arg_dtypes((%s,))"
                    % ", ".join(
                        strify_dtype(dt) for dt in kernel._scalar_arg_dtypes))

        cg("knl(queue, {}, {},".format(repr(g_size), repr(l_size)))
        cg("    %s)" % ", ".join(kernel_args))
        cg("")
        cg("queue.finish()")

    # }}}

    # {{{ data

    from base64 import b64encode
    from zlib import compress
    cg("")
    line_len = 70

    for name, val in arg_data:
        cg("%s = (" % name)
        with Indentation(cg):
            val = b64encode(compress(memoryview(val))).decode()
            i = 0
            while i < len(val):
                cg(repr(val[i:i+line_len]))
                i += line_len

            cg(")")

    # }}}

    # {{{ file trailer

    cg("")
    cg('if __name__ == "__main__":')
    with Indentation(cg):
        cg("main()")
    cg("")

    cg("# vim: filetype=pyopencl")

    # }}}

    if isinstance(output_file, str):
        with open(output_file, "w") as outf:
            outf.write(cg.get())
    else:
        output_file.write(cg.get())
