/* $Id: letters.c,v 1.4 2002/03/02 22:01:40 sverrehu Exp $ */
/*------------------------------------------------------------------------
 |  FILE            letters.c
 |  MODULE OF       banner
 |
 |  DESCRIPTION     
 |
 |  WRITTEN BY      Sverre H. Huseby
 +----------------------------------------------------------------------*/

#include <string.h>
#include <ctype.h>

#include <shhmsg.h>

#include "banner.h"

/*-----------------------------------------------------------------------+
|  PRIVATE DATA                                                          |
+-----------------------------------------------------------------------*/

extern int font1Height, font1NumLetters, font1InterCharWidth;
extern LetterData font1Letter[];
extern int font2Height, font2NumLetters, font2InterCharWidth;
extern LetterData font2Letter[];

static int height, numLetters, interCharWidth;
static LetterData *letter;

/*-----------------------------------------------------------------------+
|  PRIVATE FUNCTIONS                                                     |
+-----------------------------------------------------------------------*/

/*------------------------------------------------------------------------
 |  NAME          findLetter
 |
 |  FUNCTION      Look up a letter in the letter array.
 |
 |  INPUT         ascii   the ascii-code of the letter in question.
 |
 |  RETURNS       Pointer to the letter structure, or NULL if not found.
 */
LetterData *findLetter(unsigned ascii)
{
    int q;

    if (ascii == '\t')
	ascii = ' ';
    for (q = 0; q < numLetters; q++)
	if (letter[q].ascii == ascii)
	    return &letter[q];
    return NULL;
}

/*-----------------------------------------------------------------------+
|  PUBLIC FUNCTIONS                                                      |
+-----------------------------------------------------------------------*/

/*------------------------------------------------------------------------
 |  NAME          setFont
 |
 |  FUNCTION      Set the font to use.
 |
 |  SYNOPSIS      #include "banner.h"
 |                void setFont(int n);
 |
 |  INPUT         n       the font number. 1 is the default font.
 */
void setFont(int n)
{
    if (n == 1) {
	height = font1Height;
	numLetters = font1NumLetters;
	interCharWidth = font1InterCharWidth;
	letter = font1Letter;
    } else if (n == 2) {
	height = font2Height;
	numLetters = font2NumLetters;
	interCharWidth = font2InterCharWidth;
	letter = font2Letter;
    } else
	msgFatal("illegal font number: %d\n", n);
}

/*------------------------------------------------------------------------
 |  NAME          getLetterHeight
 |
 |  FUNCTION      Get number of lines for each letter. All are equal.
 |
 |  SYNOPSIS      #include "banner.h"
 |                int getLetterHeight(void);
 |
 |  RETURNS       Number of lines for each letter.
 */
int getLetterHeight(void)
{
    return height;
}

/*------------------------------------------------------------------------
 |  NAME          getInterCharWidth
 |
 |  FUNCTION      Get number of pixels between characters.
 |
 |  SYNOPSIS      #include "banner.h"
 |                int getInterCharWidth(void);
 |
 |  RETURNS       Number of pixels.
 */
int getInterCharWidth(void)
{
    return interCharWidth;
}

/*------------------------------------------------------------------------
 |  NAME          getLetterWidth
 |
 |  FUNCTION      Get number of horizontal points for each letter.
 |
 |  SYNOPSIS      #include "banner.h"
 |                int getLetterWidth(int ascii);
 |
 |  INPUT         ascii   the ascii-code of the letter in question.
 |
 |  RETURNS       Widt of letter in characters. 0 if letter not available.
 */
int getLetterWidth(int ascii)
{
    LetterData *l;

    if ((l = findLetter(ascii)) == NULL)
	return 0;
    return strlen(l->line[0]);
}

/*------------------------------------------------------------------------
 |  NAME          getTextWidth
 |
 |  FUNCTION      Get number of horizontal points for a text.
 |
 |  SYNOPSIS      #include "banner.h"
 |                int getTextWidth(const char *s);
 |
 |  INPUT         s       the text to sum letter widths of.
 |
 |  RETURNS       Width of text, including intercharacter spacing.
 */
int getTextWidth(const char *s)
{
    int ret = 0, w;

    while (*s) {
	w = getLetterWidth(*s);
	if (w && ret)
	    ret += getInterCharWidth();
	ret += w;
	++s;
    }
    return ret;
}

/*------------------------------------------------------------------------
 |  NAME          getLetterLine
 |
 |  FUNCTION      Get a "scan line" for a letter.
 |
 |  SYNOPSIS      #include "banner.h"
 |                char *getLetterLine(int ascii, int line);
 |
 |  INPUT         ascii   the ascii-code of the letter in question.
 |                line    the line number to get: [0, getLetterHeight() - 1]
 |
 |  RETURNS       Pointer to the scan-line, or empty string if letter not
 |                available.
 */
char *getLetterLine(int ascii, int line)
{
    LetterData *l;

    if ((l = findLetter(ascii)) == NULL)
	return "";
    return l->line[line];
}
