/*
 *  $Id: displfield.c 22419 2019-08-19 15:31:31Z yeti-dn $
 *  Copyright (C) 2019 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwythreads.h>
#include <libprocess/arithmetic.h>
#include <libprocess/correct.h>
#include <libprocess/filters.h>
#include <libprocess/stats.h>
#include <libgwydgets/gwyradiobuttons.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>
#include <app/gwymoduleutils.h>
#include "libgwyddion/gwyomp.h"
#include "preview.h"

#define DISPL_FIELD_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

typedef enum {
    DISPL_FIELD_METHOD_1D_GAUSSIAN = 0,
    DISPL_FIELD_METHOD_2D_GAUSSIAN = 1,
    DISPL_FIELD_METHOD_1D_TEAR     = 2,
    DISPL_FIELD_METHOD_1D_IMAGE    = 3,
    DISPL_FIELD_METHOD_2D_IMAGES   = 4,
    DISPL_FIELD_NMODES
} DisplFieldMethod;

typedef struct {
    DisplFieldMethod method;
    GwyExteriorType exterior;
    GwyInterpolationType interp;
    gdouble sigma;
    gdouble tau;
    gdouble density;
    gboolean update;
    gboolean randomize;
    gint seed;
    GwyAppDataId xdisplfield;
    GwyAppDataId ydisplfield;
} DisplFieldArgs;

typedef struct {
    DisplFieldArgs *args;
    GwyContainer *mydata;
    GwyDataField *field;
    GtkWidget *dialogue;
    GtkWidget *view;
    GtkWidget *method;
    GtkWidget *exterior;
    GtkWidget *interp;
    GtkObject *sigma;
    GtkWidget *sigma_value;
    GtkWidget *sigma_units;
    GtkObject *tau;
    GtkWidget *tau_value;
    GtkWidget *tau_units;
    GtkObject *density;
    GtkWidget *xdisplfield;
    GtkWidget *ydisplfield;
    GtkWidget *update;
    GtkWidget *update_now;
    GtkObject *seed;
    GtkWidget *seed_box;
    GtkWidget *randomize;
    GwySIValueFormat *vf;
} DisplFieldControls;

typedef struct {
    gint xres;
    gint yres;
    const gdouble *xdata;
    const gdouble *ydata;
} DisplacementData;

static gboolean      module_register      (void);
static void          displ_field          (GwyContainer *data,
                                           GwyRunType run);
static gboolean      displ_field_dialogue (DisplFieldArgs *args,
                                           GwyContainer *data,
                                           gint id);
static GtkWidget*    random_seed_new      (DisplFieldControls *controls);
static GtkWidget*    randomize_new        (DisplFieldControls *controls);
static GtkWidget*    instant_updates_new  (DisplFieldControls *controls);
static void          random_seed_changed  (GtkAdjustment *adj,
                                           DisplFieldControls *controls);
static void          randomize_changed    (GtkToggleButton *toggle,
                                           DisplFieldControls *controls);
static void          method_selected      (GtkComboBox *combo,
                                           DisplFieldControls *controls);
static void          sigma_changed        (GtkAdjustment *adj,
                                           DisplFieldControls *controls);
static void          tau_changed          (GtkAdjustment *adj,
                                           DisplFieldControls *controls);
static void          density_changed      (GtkAdjustment *adj,
                                           DisplFieldControls *controls);
static void          xdisplfield_changed  (GwyDataChooser *chooser,
                                           DisplFieldControls *controls);
static void          ydisplfield_changed  (GwyDataChooser *chooser,
                                           DisplFieldControls *controls);
static void          exterior_changed     (GtkComboBox *combo,
                                           DisplFieldControls *controls);
static void          interp_changed       (GtkComboBox *combo,
                                           DisplFieldControls *controls);
static void          update_changed       (GtkToggleButton *toggle,
                                           DisplFieldControls *controls);
static void          update_sensitivity   (DisplFieldControls *controls);
static gboolean      displ_field_filter   (GwyContainer *data,
                                           gint id,
                                           gpointer user_data);
static void          preview              (DisplFieldControls *controls);
static GwyDataField* displ_field_do       (GwyDataField *field,
                                           const DisplFieldArgs *args);
static GwyDataField* make_displacement_map(guint xres,
                                           guint yres,
                                           gdouble sigma,
                                           gdouble tau,
                                           GRand *rng);
static GwyDataField* make_tear_map        (guint xres,
                                           guint yres,
                                           gdouble sigma,
                                           gdouble density,
                                           gdouble length,
                                           GRand *rng);
static gboolean      method_has_tau_sigma (DisplFieldMethod method);
static gboolean      method_has_density   (DisplFieldMethod method);
static gboolean      method_needs_xdef    (DisplFieldMethod method);
static gboolean      method_needs_ydef    (DisplFieldMethod method);
static void          load_args            (GwyContainer *container,
                                           DisplFieldArgs *args);
static void          save_args            (GwyContainer *container,
                                           DisplFieldArgs *args);

static GwyAppDataId xdisplfield_id = GWY_APP_DATA_ID_NONE;
static GwyAppDataId ydisplfield_id = GWY_APP_DATA_ID_NONE;

static const DisplFieldArgs displfield_defaults = {
    DISPL_FIELD_METHOD_2D_GAUSSIAN,
    GWY_EXTERIOR_BORDER_EXTEND, GWY_INTERPOLATION_BILINEAR,
    10.0, 50.0, 0.02,
    TRUE, TRUE, 42,
    GWY_APP_DATA_ID_NONE, GWY_APP_DATA_ID_NONE,
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Distorts image or individual scan lines in plane using "
       "a displacement field."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2019",
};

GWY_MODULE_QUERY2(module_info, displfield)

static gboolean
module_register(void)
{
    gwy_process_func_register("displfield",
                              (GwyProcessFunc)&displ_field,
                              N_("/_Distortion/Displacement _Field..."),
                              NULL,
                              DISPL_FIELD_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              _("Deform image or scan lines in plane"));

    return TRUE;
}

static void
displ_field(GwyContainer *data, GwyRunType run)
{
    DisplFieldArgs args;
    GwyContainer *settings;
    GwyDataField *field, *result;
    gint id, newid;

    g_return_if_fail(run & DISPL_FIELD_RUN_MODES);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &field,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     0);

    settings = gwy_app_settings_get();
    load_args(settings, &args);

    if (args.method == DISPL_FIELD_METHOD_2D_IMAGES
        && !displ_field_filter(gwy_app_data_browser_get(args.ydisplfield.datano),
                               args.ydisplfield.id, field))
        args.method = DISPL_FIELD_METHOD_1D_IMAGE;

    if (args.method == DISPL_FIELD_METHOD_1D_IMAGE
        && !displ_field_filter(gwy_app_data_browser_get(args.xdisplfield.datano),
                               args.xdisplfield.id, field))
        args.method = DISPL_FIELD_METHOD_1D_GAUSSIAN;

    if (run == GWY_RUN_IMMEDIATE
        || (run == GWY_RUN_INTERACTIVE
            && displ_field_dialogue(&args, data, id))) {
        if (args.randomize)
            args.seed = g_random_int() & 0x7fffffff;

        result = displ_field_do(field, &args);

        newid = gwy_app_data_browser_add_data_field(result, data, TRUE);
        g_object_unref(result);
        gwy_app_sync_data_items(data, data, id, newid, FALSE,
                                GWY_DATA_ITEM_PALETTE,
                                GWY_DATA_ITEM_RANGE_TYPE,
                                GWY_DATA_ITEM_REAL_SQUARE,
                                0);
        gwy_app_set_data_field_title(data, newid, _("Distorted"));
        gwy_app_channel_log_add_proc(data, id, newid);
    }

    save_args(settings, &args);
}

static gboolean
displ_field_dialogue(DisplFieldArgs *args, GwyContainer *data, gint id)
{
    static const GwyEnum methods[] = {
        { N_("Gaussian (scan lines)"),      DISPL_FIELD_METHOD_1D_GAUSSIAN, },
        { N_("Gaussian (two-dimensional)"), DISPL_FIELD_METHOD_2D_GAUSSIAN, },
        { N_("Tear scan lines"),            DISPL_FIELD_METHOD_1D_TEAR,     },
        { N_("Image (scan lines)"),         DISPL_FIELD_METHOD_1D_IMAGE,    },
        { N_("Images (two-dimensional)"),   DISPL_FIELD_METHOD_2D_IMAGES,   },
    };

    DisplFieldControls controls;
    GtkWidget *dialogue, *table, *label, *vbox, *hbox;
    GwyDataField *field;
    GwyDataChooser *chooser;
    gint row, response;

    field = gwy_container_get_object(data, gwy_app_get_data_key_for_id(id));
    controls.args = args;
    controls.field = field;
    controls.vf
        = gwy_data_field_get_value_format_xy(field,
                                             GWY_SI_UNIT_FORMAT_VFMARKUP, NULL);

    dialogue = gtk_dialog_new_with_buttons(_("Displacement Field"), NULL, 0,
                                           NULL);
    gtk_dialog_add_action_widget(GTK_DIALOG(dialogue),
                                 gwy_stock_like_button_new(_("_Update"),
                                                           GTK_STOCK_EXECUTE),
                                 RESPONSE_PREVIEW);
    gtk_dialog_add_button(GTK_DIALOG(dialogue),
                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
    gtk_dialog_add_button(GTK_DIALOG(dialogue),
                          GTK_STOCK_OK, GTK_RESPONSE_OK);
    gtk_dialog_set_default_response(GTK_DIALOG(dialogue), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialogue), GWY_HELP_DEFAULT);
    controls.dialogue = dialogue;

    hbox = gtk_hbox_new(FALSE, 2);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialogue)->vbox), hbox,
                       FALSE, FALSE, 4);

    vbox = gtk_vbox_new(FALSE, 2);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 4);

    controls.mydata = gwy_container_new();
    field = gwy_data_field_new_alike(field, TRUE);
    gwy_container_set_object_by_name(controls.mydata, "/0/data", field);
    g_object_unref(field);
    gwy_app_sync_data_items(data, controls.mydata, id, 0, FALSE,
                            GWY_DATA_ITEM_PALETTE,
                            GWY_DATA_ITEM_MASK_COLOR,
                            GWY_DATA_ITEM_RANGE,
                            GWY_DATA_ITEM_REAL_SQUARE,
                            0);
    controls.view = create_preview(controls.mydata, 0, PREVIEW_SIZE, FALSE);
    gtk_box_pack_start(GTK_BOX(vbox), controls.view, FALSE, FALSE, 0);

    gtk_box_pack_start(GTK_BOX(vbox), instant_updates_new(&controls),
                       FALSE, FALSE, 0);
    g_signal_connect_swapped(controls.update_now, "clicked",
                             G_CALLBACK(preview), &controls);

    gtk_box_pack_start(GTK_BOX(vbox), random_seed_new(&controls),
                       FALSE, FALSE, 0);

    controls.randomize = randomize_new(&controls);
    gtk_box_pack_start(GTK_BOX(vbox), controls.randomize, FALSE, FALSE, 0);

    table = gtk_table_new(8, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, TRUE, 0);
    row = 0;

    controls.method = gwy_enum_combo_box_new(methods, G_N_ELEMENTS(methods),
                                             G_CALLBACK(method_selected),
                                             &controls,
                                             args->method, TRUE);
    gwy_table_attach_adjbar(table, row++, _("_Method:"), NULL,
                            GTK_OBJECT(controls.method),
                            GWY_HSCALE_WIDGET_NO_EXPAND);

    controls.xdisplfield = gwy_data_chooser_new_channels();
    chooser = GWY_DATA_CHOOSER(controls.xdisplfield);
    gwy_data_chooser_set_active_id(chooser, &args->xdisplfield);
    gwy_data_chooser_set_filter(chooser, displ_field_filter, controls.field,
                                NULL);
    gwy_data_chooser_set_active_id(chooser, &args->xdisplfield);
    gwy_data_chooser_get_active_id(chooser, &args->xdisplfield);
    g_signal_connect(chooser, "changed",
                     G_CALLBACK(xdisplfield_changed), &controls);
    gwy_table_attach_adjbar(table, row++, _("_X displacement:"), NULL,
                            GTK_OBJECT(chooser), GWY_HSCALE_WIDGET_NO_EXPAND);

    controls.ydisplfield = gwy_data_chooser_new_channels();
    chooser = GWY_DATA_CHOOSER(controls.ydisplfield);
    gwy_data_chooser_set_active_id(chooser, &args->ydisplfield);
    gwy_data_chooser_set_filter(chooser, displ_field_filter, controls.field,
                                NULL);
    gwy_data_chooser_set_active_id(chooser, &args->ydisplfield);
    gwy_data_chooser_get_active_id(chooser, &args->ydisplfield);
    g_signal_connect(chooser, "changed",
                     G_CALLBACK(ydisplfield_changed), &controls);
    gwy_table_attach_adjbar(table, row++, _("_Y displacement:"), NULL,
                            GTK_OBJECT(chooser), GWY_HSCALE_WIDGET_NO_EXPAND);

    controls.sigma = gtk_adjustment_new(args->sigma, 0.0, 100.0, 0.01, 1.0, 0);
    gwy_table_attach_adjbar(table, row++, _("_Amplitude:"), _("px"),
                            controls.sigma, GWY_HSCALE_SQRT);
    g_signal_connect(controls.sigma, "value-changed",
                     G_CALLBACK(sigma_changed), &controls);

    label = controls.sigma_value = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     1, 2, row, row+1, GTK_FILL, 0, 0, 0);
    label = controls.sigma_units = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_label_set_markup(GTK_LABEL(label), controls.vf->units);
    gtk_table_attach(GTK_TABLE(table), label,
                     2, 3, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.tau = gtk_adjustment_new(args->tau, 0.1, 1000.0, 0.01, 1.0, 0);
    gwy_table_attach_adjbar(table, row++, _("_Lateral scale:"), _("px"),
                            controls.tau, GWY_HSCALE_LOG);
    g_signal_connect(controls.tau, "value-changed",
                     G_CALLBACK(tau_changed), &controls);

    label = controls.tau_value = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     1, 2, row, row+1, GTK_FILL, 0, 0, 0);
    label = controls.tau_units = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_label_set_markup(GTK_LABEL(label), controls.vf->units);
    gtk_table_attach(GTK_TABLE(table), label,
                     2, 3, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.density = gtk_adjustment_new(args->density,
                                          0.0001, 0.25, 0.0001, 0.01, 0);
    gwy_table_attach_adjbar(table, row++, _("Densi_ty:"), NULL,
                            controls.density, GWY_HSCALE_SQRT);
    g_signal_connect(controls.density, "value-changed",
                     G_CALLBACK(density_changed), &controls);

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    label = gwy_label_new_header(_("Options"));
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.exterior
        = gwy_enum_combo_box_newl(G_CALLBACK(exterior_changed), &controls,
                                  args->exterior,
                                  gwy_sgettext("exterior|Border"),
                                  GWY_EXTERIOR_BORDER_EXTEND,
                                  gwy_sgettext("exterior|Mirror"),
                                  GWY_EXTERIOR_MIRROR_EXTEND,
                                  gwy_sgettext("exterior|Periodic"),
                                  GWY_EXTERIOR_PERIODIC,
                                  gwy_sgettext("exterior|Laplace"),
                                  GWY_EXTERIOR_LAPLACE,
                                  NULL);
    gwy_table_attach_adjbar(table, row++, _("_Exterior type:"), NULL,
                            GTK_OBJECT(controls.exterior),
                            GWY_HSCALE_WIDGET_NO_EXPAND);

    controls.interp
        = gwy_enum_combo_box_new(gwy_interpolation_type_get_enum(), -1,
                                 G_CALLBACK(interp_changed), &controls,
                                 args->interp, TRUE);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row++,
                            _("_Interpolation type:"), NULL,
                            GTK_OBJECT(controls.interp),
                            GWY_HSCALE_WIDGET_NO_EXPAND);

    update_sensitivity(&controls);
    gtk_widget_show_all(dialogue);
    if (args->update)
        preview(&controls);

    do {
        response = gtk_dialog_run(GTK_DIALOG(dialogue));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialogue);
            case GTK_RESPONSE_NONE:
            gwy_si_unit_value_format_free(controls.vf);
            g_object_unref(controls.mydata);
            return FALSE;
            break;

            case GTK_RESPONSE_OK:
            break;

            case RESPONSE_PREVIEW:
            preview(&controls);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialogue);
    g_object_unref(controls.mydata);
    gwy_si_unit_value_format_free(controls.vf);

    return TRUE;
}

static void
randomize_seed(GtkAdjustment *adj)
{
    /* Use the GLib's global PRNG for seeding */
    gtk_adjustment_set_value(adj, g_random_int() & 0x7fffffff);
}

static GtkWidget*
random_seed_new(DisplFieldControls *controls)
{
    GtkWidget *hbox, *button, *label, *spin;

    controls->seed_box = hbox = gtk_hbox_new(FALSE, 6);

    controls->seed = gtk_adjustment_new(controls->args->seed,
                                        1, 0x7fffffff, 1, 10, 0);
    g_signal_connect(controls->seed, "value-changed",
                     G_CALLBACK(random_seed_changed), controls);

    label = gtk_label_new_with_mnemonic(_("R_andom seed:"));
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
    spin = gtk_spin_button_new(GTK_ADJUSTMENT(controls->seed), 0, 0);
    gtk_label_set_mnemonic_widget(GTK_LABEL(label), spin);
    gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 0);

    button = gtk_button_new_with_mnemonic(gwy_sgettext("seed|_New"));
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    g_signal_connect_swapped(button, "clicked",
                             G_CALLBACK(randomize_seed), controls->seed);

    return hbox;
}

static GtkWidget*
randomize_new(DisplFieldControls *controls)
{
    GtkWidget *button = gtk_check_button_new_with_mnemonic(_("Randomi_ze"));

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),
                                 controls->args->randomize);
    g_signal_connect(button, "toggled",
                     G_CALLBACK(randomize_changed), controls);
    return button;
}

static GtkWidget*
instant_updates_new(DisplFieldControls *controls)
{
    GtkWidget *hbox, *button, *check;

    hbox = gtk_hbox_new(FALSE, 6);

    button = controls->update_now = gtk_button_new_with_mnemonic(_("_Update"));
    gtk_widget_set_sensitive(button, !controls->args->update);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);

    check = controls->update
        = gtk_check_button_new_with_mnemonic(_("I_nstant updates"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check),
                                 controls->args->update);
    gtk_box_pack_start(GTK_BOX(hbox), check, FALSE, FALSE, 0);
    g_signal_connect(check, "toggled", G_CALLBACK(update_changed), controls);

    return hbox;
}

static void
random_seed_changed(GtkAdjustment *adj, DisplFieldControls *controls)
{
    DisplFieldArgs *args = controls->args;

    args->seed = gwy_adjustment_get_int(adj);
    if (args->update)
        preview(controls);
}

static void
randomize_changed(GtkToggleButton *toggle, DisplFieldControls *controls)
{
    DisplFieldArgs *args = controls->args;

    args->randomize = gtk_toggle_button_get_active(toggle);
}

static void
method_selected(GtkComboBox *combo, DisplFieldControls *controls)
{
    DisplFieldArgs *args = controls->args;

    args->method = gwy_enum_combo_box_get_active(combo);
    update_sensitivity(controls);
    if (args->update)
        preview(controls);
}

static void
sigma_changed(GtkAdjustment *adj, DisplFieldControls *controls)
{
    DisplFieldArgs *args = controls->args;

    args->sigma = gtk_adjustment_get_value(adj);
    if (method_has_tau_sigma(args->method) && args->update)
        preview(controls);
}

static void
tau_changed(GtkAdjustment *adj, DisplFieldControls *controls)
{
    DisplFieldArgs *args = controls->args;

    args->tau = gtk_adjustment_get_value(adj);
    if (method_has_tau_sigma(args->method) && args->update)
        preview(controls);
}

static void
density_changed(GtkAdjustment *adj, DisplFieldControls *controls)
{
    DisplFieldArgs *args = controls->args;

    args->density = gtk_adjustment_get_value(adj);
    if (method_has_density(args->method) && args->update)
        preview(controls);
}

static void
xdisplfield_changed(GwyDataChooser *chooser, DisplFieldControls *controls)
{
    DisplFieldArgs *args = controls->args;

    gwy_data_chooser_get_active_id(chooser, &args->xdisplfield);
    if (method_needs_xdef(args->method) && args->update)
        preview(controls);
}

static void
ydisplfield_changed(GwyDataChooser *chooser, DisplFieldControls *controls)
{
    DisplFieldArgs *args = controls->args;

    gwy_data_chooser_get_active_id(chooser, &args->ydisplfield);
    if (method_needs_ydef(args->method) && args->update)
        preview(controls);
}

static void
interp_changed(GtkComboBox *combo, DisplFieldControls *controls)
{
    DisplFieldArgs *args = controls->args;

    args->interp = gwy_enum_combo_box_get_active(combo);
    if (args->update)
        preview(controls);
}

static void
exterior_changed(GtkComboBox *combo, DisplFieldControls *controls)
{
    DisplFieldArgs *args = controls->args;

    args->exterior = gwy_enum_combo_box_get_active(combo);
    if (args->update)
        preview(controls);
}

static void
update_changed(GtkToggleButton *toggle, DisplFieldControls *controls)
{
    DisplFieldArgs *args = controls->args;

    args->update = gtk_toggle_button_get_active(toggle);
    update_sensitivity(controls);
    if (args->update)
        preview(controls);
}

static void
update_sensitivity(DisplFieldControls *controls)
{
    DisplFieldArgs *args = controls->args;
    gboolean need_xdef, need_ydef, has_ts, has_density, has_rng;

    need_xdef = method_needs_xdef(args->method);
    need_ydef = method_needs_ydef(args->method);
    has_ts = method_has_tau_sigma(args->method);
    has_density = method_has_density(args->method);
    has_rng = !(need_xdef || need_ydef);
    /* TODO
     * Also must handle the case when image method is selected but no suitable
     * deformation fields are available.  Or prevent such case. */
    gwy_table_hscale_set_sensitive(GTK_OBJECT(controls->xdisplfield),
                                   need_xdef);
    gwy_table_hscale_set_sensitive(GTK_OBJECT(controls->ydisplfield),
                                   need_ydef);
    gwy_table_hscale_set_sensitive(controls->sigma, has_ts);
    gwy_table_hscale_set_sensitive(controls->tau, has_ts);
    gwy_table_hscale_set_sensitive(controls->density, has_density);
    gtk_widget_set_sensitive(controls->sigma_value, has_ts);
    gtk_widget_set_sensitive(controls->sigma_units, has_ts);
    gtk_widget_set_sensitive(controls->tau_value, has_ts);
    gtk_widget_set_sensitive(controls->tau_units, has_ts);
    gtk_dialog_set_response_sensitive(GTK_DIALOG(controls->dialogue),
                                      RESPONSE_PREVIEW, !args->update);
    gtk_widget_set_sensitive(controls->seed_box, has_rng);
    gtk_widget_set_sensitive(controls->randomize, has_rng);
}

static gboolean
displ_field_filter(GwyContainer *data,
                   gint id,
                   gpointer user_data)
{
    GwyDataField *dfield, *field = (GwyDataField*)user_data;
    GwySIUnit *fxyunit, *dzunit;
    GQuark quark;

    quark = gwy_app_get_data_key_for_id(id);
    if (!gwy_container_gis_object(data, quark, &dfield))
        return FALSE;

    if (dfield == field)
        return FALSE;
    if (gwy_data_field_check_compatibility(field, dfield,
                                           GWY_DATA_COMPATIBILITY_RES))
        return FALSE;

    /* Values of the distortion field are lateral offsets in the image. */
    fxyunit = gwy_data_field_get_si_unit_xy(field);
    dzunit = gwy_data_field_get_si_unit_z(dfield);

    return gwy_si_unit_equal(fxyunit, dzunit);
}

static void
preview(DisplFieldControls *controls)
{
    DisplFieldArgs *args = controls->args;
    GwyDataField *result;

    result = displ_field_do(controls->field, args);
    gwy_container_set_object_by_name(controls->mydata, "/0/data", result);
    g_object_unref(result);
}

static void
distort_func_2d(gdouble x, gdouble y,
                gdouble *px, gdouble *py,
                gpointer user_data)
{
    const DisplacementData *dd = (const DisplacementData*)user_data;
    gint i, j, k;

    j = (gint)floor(x);
    j = CLAMP(j, 0, dd->xres-1);
    i = (gint)floor(y);
    i = CLAMP(i, 0, dd->yres-1);
    k = i*dd->xres + j;
    *px = x - dd->xdata[k];
    *py = y - dd->ydata[k];
}

static GwyDataField*
create_displ_field(const GwyAppDataId *dataid)
{
    GwyDataField *displfield;
    GwyContainer *data;
    GQuark quark;

    if (dataid->datano <= 0 || dataid->id < 0)
        return NULL;

    data = gwy_app_data_browser_get(dataid->datano);
    quark = gwy_app_get_data_key_for_id(dataid->id);
    displfield = gwy_container_get_object(data, quark);

    return gwy_data_field_duplicate(displfield);
}

static GwyDataField*
displ_field_do(GwyDataField *field, const DisplFieldArgs *args)
{
    GwyDataField *result = gwy_data_field_new_alike(field, TRUE);
    GwyDataField *xdisplfield = NULL, *ydisplfield = NULL;
    DisplacementData dd;
    GRand *rng;
    gint xres, yres;

    gwy_clear(&dd, 1);
    xres = dd.xres = gwy_data_field_get_xres(field);
    yres = dd.yres = gwy_data_field_get_yres(field);
    rng = g_rand_new();
    g_rand_set_seed(rng, args->seed);

    if (args->method == DISPL_FIELD_METHOD_2D_GAUSSIAN) {
        xdisplfield = make_displacement_map(xres, yres,
                                            args->sigma, args->tau,
                                            rng);
        ydisplfield = make_displacement_map(xres, yres,
                                            args->sigma, args->tau,
                                            rng);
    }
    else if (args->method == DISPL_FIELD_METHOD_1D_GAUSSIAN) {
        xdisplfield = make_displacement_map(xres, yres,
                                            args->sigma, args->tau,
                                            rng);
    }
    else if (args->method == DISPL_FIELD_METHOD_1D_TEAR) {
        xdisplfield = make_tear_map(xres, yres,
                                    args->sigma, args->density, args->tau,
                                    rng);
    }
    else if (args->method == DISPL_FIELD_METHOD_1D_IMAGE) {
        if ((xdisplfield = create_displ_field(&args->xdisplfield))) {
            gwy_data_field_multiply(xdisplfield,
                                    1.0/gwy_data_field_get_dx(field));
            ydisplfield = gwy_data_field_new_alike(xdisplfield, TRUE);
        }
    }
    else if (args->method == DISPL_FIELD_METHOD_2D_IMAGES) {
        if ((xdisplfield = create_displ_field(&args->xdisplfield))
            && (ydisplfield = create_displ_field(&args->ydisplfield))) {
            gwy_data_field_multiply(xdisplfield,
                                    1.0/gwy_data_field_get_dx(field));
            gwy_data_field_multiply(ydisplfield,
                                    1.0/gwy_data_field_get_dy(field));
        }
        else {
            GWY_OBJECT_UNREF(xdisplfield);
            GWY_OBJECT_UNREF(ydisplfield);
        }
    }

    if (xdisplfield) {
        /* XXX XXX XXX: Very inefficient. */
        if (!ydisplfield)
            ydisplfield = gwy_data_field_new_alike(xdisplfield, TRUE);

        dd.xdata = gwy_data_field_get_data_const(xdisplfield);
        dd.ydata = gwy_data_field_get_data_const(ydisplfield);
        gwy_data_field_distort(field, result, &distort_func_2d, &dd,
                               args->interp, args->exterior, 0.0);
    }

    GWY_OBJECT_UNREF(xdisplfield);
    GWY_OBJECT_UNREF(ydisplfield);
    g_rand_free(rng);

    return result;
}

/* Iterating through square in a spiral fashion from the origin to preserve the
 * centre conrer if it's randomly generated.  Field @k holds the current index
 * in the two-dimensional array.  */
typedef struct {
    gint n;
    gint i, j, k;
    gint istep, jstep;
    gint s, segmentend, ntotalstep;
} GrowingIter;

static inline void
growing_iter_init(GrowingIter *giter, guint n)
{
    giter->n = n;
    giter->j = giter->i = 0;
    giter->istep = 0;
    giter->jstep = -1;
    giter->ntotalstep = n*n;
    giter->segmentend = MIN(1, n*n);
    giter->s = 0;
    giter->k = (n/2 - giter->i)*n + (giter->j + n/2);
}

static inline gboolean
growing_iter_next(GrowingIter *giter)
{
    giter->i += giter->istep;
    giter->j += giter->jstep;
    giter->k = (giter->n/2 - giter->i)*giter->n + (giter->j + giter->n/2);
    giter->s++;
    if (giter->s == giter->segmentend) {
        if (giter->s == giter->ntotalstep)
            return FALSE;

        if (giter->i == giter->j + 1) {
            giter->istep = 1;
            giter->jstep = 0;
            giter->segmentend = 1 - 2*giter->i;
        }
        else if (giter->i == giter->j) {
            giter->istep = -1;
            giter->jstep = 0;
            giter->segmentend = 2*giter->i;
        }
        else if (giter->j > 0) {
            giter->istep = 0;
            giter->jstep = -1;
            giter->segmentend = 2*giter->j + 1;
        }
        else {
            giter->istep = 0;
            giter->jstep = 1;
            giter->segmentend = 2*giter->i;
        }
        giter->segmentend += giter->s;
        giter->segmentend = MIN(giter->segmentend, giter->ntotalstep);
    }
    return TRUE;
}

/* Fill a data field with uncorrelated random numbers in a growing fashion to
 * preserve the character of the noise even if the dimensions change */
/* XXX: Duplicate with lat_synth, pat_synth and others */
static void
fill_displacement_map(GwyDataField *dfield, GRand *rng)
{
    GrowingIter giter;
    guint xres, yres;
    gdouble *data;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    g_return_if_fail(xres == yres);
    data = gwy_data_field_get_data(dfield);
    growing_iter_init(&giter, xres);

    do {
        data[giter.k] = g_rand_double(rng) - 0.5;
    } while (growing_iter_next(&giter));
}

static GwyDataField*
make_displacement_map(guint xres, guint yres,
                      gdouble sigma, gdouble tau,
                      GRand *rng)
{
    GwyDataField *grid, *dfield;
    guint gn, n;
    gdouble r;

    n = MAX(xres, yres);
    if (sigma*tau == 0.0)
        return gwy_data_field_new(xres, yres, 1.0, 1.0, TRUE);

    if (tau <= 1.0) {
        grid = gwy_data_field_new(n, n, 1.0, 1.0, FALSE);
        fill_displacement_map(grid, rng);
        gwy_data_field_filter_gaussian(grid, tau);
        gwy_data_field_multiply(grid, sigma/gwy_data_field_get_rms(grid));
        if (xres == yres)
            return grid;

        dfield = gwy_data_field_area_extract(grid, 0, 0, xres, yres);
        g_object_unref(grid);
        return dfield;
    }

    gn = GWY_ROUND(1.0/tau*n);
    gn = MAX(gn, 2);
    r = (gdouble)gn/n;
    grid = gwy_data_field_new(gn, gn, 1.0, 1.0, FALSE);
    fill_displacement_map(grid, rng);
    gwy_data_field_filter_gaussian(grid, r*tau);
    gwy_data_field_multiply(grid, sigma/gwy_data_field_get_rms(grid));
    dfield = gwy_data_field_new_resampled(grid, n, n, GWY_INTERPOLATION_KEY);
    g_object_unref(grid);

    if (xres != yres) {
        grid = dfield;
        dfield = gwy_data_field_area_extract(grid, 0, 0, xres, yres);
        g_object_unref(grid);
    }

    return dfield;
}

static gboolean
collides_with_another_tear(const gdouble *m,
                           gint xres, gint yres,
                           gint col, gint row, gint len)
{
    gint ifrom = MAX(row-1, 0), ito = MIN(row+2+1, yres);
    gint jfrom = MAX(col-len/2-1, 0), jto = MIN(col+len/2+2, xres);
    gint i, j;

    for (i = ifrom; i < ito; i++) {
        const gdouble *mrow = m + i*xres;
        for (j = jfrom; j < jto; j++) {
            if (mrow[j] != 0.0)
                return TRUE;
        }
    }

    return FALSE;
}

static void
fill_tear(gdouble *d, gdouble *m,
          gint xres, gint yres,
          gint col, gint row, gint len,
          gdouble value)
{
    gint jfrom = MAX(col-len/2, 0), jto = MIN(col+len/2+1, xres);
    gint j;

    if (row >= 0) {
        gdouble *drow = d + row*xres;
        gdouble *mrow = m + row*xres;
        for (j = jfrom; j < jto; j++) {
            gdouble t = (j - col)/(gdouble)(len/2), tp = 1.0 + t, tm = 1.0 - t;
            drow[j] = value*tm*tm*tp*tp;
            mrow[j] = 1.0;
        }
    }
    if (row < yres-1) {
        gdouble *drow = d + (row + 1)*xres;
        gdouble *mrow = m + (row + 1)*xres;
        for (j = jfrom; j < jto; j++) {
            gdouble t = (j - col)/(gdouble)(len/2), tp = 1.0 + t, tm = 1.0 - t;
            drow[j] = -value*tm*tm*tp*tp;
            mrow[j] = 1.0;
        }
    }
}

static GwyDataField*
make_tear_map(guint xres, guint yres,
              gdouble sigma, gdouble density, gdouble length,
              GRand *rng)
{
    GwyDataField *dfield, *mask;
    gdouble *d, *m;
    guint n, k;

    dfield = gwy_data_field_new(xres, yres, 1.0, 1.0, TRUE);
    n = GWY_ROUND(0.5*xres*yres*density/length);
    if (!n)
        return dfield;

    mask = gwy_data_field_new(xres, yres, 1.0, 1.0, TRUE);
    d = gwy_data_field_get_data(dfield);
    m = gwy_data_field_get_data(mask);

    for (k = 0; k < n; k++) {
        gint col, row, len, count = 0;
        gdouble value;

        do {
            /* The tear is a two-line structure, so start from -1 for symmetry
             * as its second line can also go one line beyond. */
            row = g_rand_int_range(rng, -1, yres);
            col = g_rand_int_range(rng, -length, xres + length);
            len = GWY_ROUND(length + (g_rand_double(rng)
                                      + g_rand_double(rng)
                                      + g_rand_double(rng) - 1.5)*length/5.0);
            count++;
        } while (count < 100
                 && (len < 2 || collides_with_another_tear(m, xres, yres,
                                                           col, row, len)));
        if (count == 100) {
            //g_printerr("Cannot place tear #%u.\n", k+1);
            break;
        }

        value = sigma*(g_rand_double(rng) - 0.5);
        fill_tear(d, m, xres, yres, col, row, len, value);
    }

    gwy_data_field_laplace_solve(dfield, mask, 0, 0.5);
    g_object_unref(mask);

    return dfield;
}

static gboolean
method_has_tau_sigma(DisplFieldMethod method)
{
    return (method == DISPL_FIELD_METHOD_1D_GAUSSIAN
            || method == DISPL_FIELD_METHOD_2D_GAUSSIAN
            || method == DISPL_FIELD_METHOD_1D_TEAR);
}

static gboolean
method_has_density(DisplFieldMethod method)
{
    return method == DISPL_FIELD_METHOD_1D_TEAR;
}

static gboolean
method_needs_xdef(DisplFieldMethod method)
{
    return (method == DISPL_FIELD_METHOD_1D_IMAGE
            || method == DISPL_FIELD_METHOD_2D_IMAGES);
}

static gboolean
method_needs_ydef(DisplFieldMethod method)
{
    return method == DISPL_FIELD_METHOD_2D_IMAGES;
}

static const gchar density_key[]   = "/module/displfield/density";
static const gchar exterior_key[]  = "/module/displfield/exterior";
static const gchar interp_key[]    = "/module/displfield/interp";
static const gchar method_key[]    = "/module/displfield/method";
static const gchar randomize_key[] = "/module/displfield/randomize";
static const gchar seed_key[]      = "/module/displfield/seed";
static const gchar sigma_key[]     = "/module/displfield/sigma";
static const gchar tau_key[]       = "/module/displfield/tau";
static const gchar update_key[]    = "/module/displfield/update";

static void
sanitize_args(DisplFieldArgs *args)
{
    args->method = MIN(args->method, DISPL_FIELD_NMODES-1);
    args->interp = gwy_enum_sanitize_value(args->interp,
                                           GWY_TYPE_INTERPOLATION_TYPE);
    args->exterior = gwy_enum_sanitize_value(args->exterior,
                                             GWY_TYPE_EXTERIOR_TYPE);
    args->sigma = CLAMP(args->sigma, 0.0, 100.0);
    args->tau = CLAMP(args->tau, 0.1, 1000.0);
    args->density = CLAMP(args->density, 0.0001, 0.25);
    args->update = !!args->update;
    args->seed = MAX(0, args->seed);
    args->randomize = !!args->randomize;

    if ((method_needs_ydef(args->method)
         && !gwy_app_data_id_verify_channel(&args->ydisplfield))
        || (method_needs_xdef(args->method)
            && !gwy_app_data_id_verify_channel(&args->xdisplfield)))
        args->method = displfield_defaults.method;
}

static void
load_args(GwyContainer *container,
          DisplFieldArgs *args)
{
    *args = displfield_defaults;

    gwy_container_gis_enum_by_name(container, method_key, &args->method);
    gwy_container_gis_enum_by_name(container, exterior_key, &args->exterior);
    gwy_container_gis_enum_by_name(container, interp_key, &args->interp);
    gwy_container_gis_double_by_name(container, sigma_key, &args->sigma);
    gwy_container_gis_double_by_name(container, tau_key, &args->tau);
    gwy_container_gis_double_by_name(container, density_key, &args->density);
    gwy_container_gis_boolean_by_name(container, update_key, &args->update);
    gwy_container_gis_int32_by_name(container, seed_key, &args->seed);
    gwy_container_gis_boolean_by_name(container, randomize_key,
                                      &args->randomize);
    args->xdisplfield = xdisplfield_id;
    args->ydisplfield = ydisplfield_id;
    sanitize_args(args);
}

static void
save_args(GwyContainer *container,
          DisplFieldArgs *args)
{
    xdisplfield_id = args->xdisplfield;
    ydisplfield_id = args->ydisplfield;
    gwy_container_set_enum_by_name(container, method_key, args->method);
    gwy_container_set_enum_by_name(container, exterior_key, args->exterior);
    gwy_container_set_enum_by_name(container, interp_key, args->interp);
    gwy_container_set_double_by_name(container, sigma_key, args->sigma);
    gwy_container_set_double_by_name(container, tau_key, args->tau);
    gwy_container_set_double_by_name(container, density_key, args->density);
    gwy_container_set_boolean_by_name(container, update_key, args->update);
    gwy_container_set_int32_by_name(container, seed_key, args->seed);
    gwy_container_set_boolean_by_name(container, randomize_key,
                                      args->randomize);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
