/**
 *    Copyright 2011 Peter Murray-Rust
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package org.xmlcml.euclid.test;

import junit.framework.Assert;

import org.apache.log4j.Logger;
import org.junit.Test;
import org.xmlcml.euclid.ParsedSymop;
import org.xmlcml.euclid.Transform3;

public class ParsedSymopTest {

	private final static Logger LOG = Logger.getLogger(ParsedSymopTest.class);

	@Test
	public void parseXYZ() {
		ParsedSymop symop = ParsedSymop.createSymop("x");
		Assert.assertEquals("xyz", "x", symop.getXyz());
		symop = ParsedSymop.createSymop("X");
		Assert.assertEquals("xyz", "x", symop.getXyz());
		symop = ParsedSymop.createSymop("y");
		Assert.assertEquals("xyz", "y", symop.getXyz());
		symop = ParsedSymop.createSymop("+z");
		Assert.assertEquals("xyz", "z", symop.getXyz());
		symop = ParsedSymop.createSymop("-z");
		Assert.assertEquals("xyz", "-z", symop.getXyz());
		Assert.assertEquals("xyz", "", symop.getXyz1());
		Assert.assertNull("xyz", symop.getNumber());
		try {
			symop = ParsedSymop.createSymop("a");
			Assert.fail();
		} catch (Exception e) {
			
		}
	}
	
	@Test
	public void parseXYZXYZ() {
		ParsedSymop symop = ParsedSymop.createSymop("+x-y");
		Assert.assertEquals("xyz", "x", symop.getXyz());
		Assert.assertEquals("xyz", "-y", symop.getXyz1());
		symop = ParsedSymop.createSymop("-x-y");
		Assert.assertEquals("xyz", "-x", symop.getXyz());
		Assert.assertEquals("xyz", "-y", symop.getXyz1());
		symop = ParsedSymop.createSymop("-x+y");
		Assert.assertEquals("xyz", "-x", symop.getXyz());
		Assert.assertEquals("xyz", "y", symop.getXyz1());
		try {
			symop = ParsedSymop.createSymop("xy");
		} catch (Exception e) {
			LOG.debug("'xy' is not picked up as error, sorry");
		}
	}
	
	@Test
	public void parseNUMB_XYZ() {
		ParsedSymop symop = ParsedSymop.createSymop("0.5+x");
		Assert.assertEquals("xyz", "x", symop.getXyz());
		Assert.assertEquals("xyz", "", symop.getXyz1());
		Assert.assertEquals("xyz", 0.5, symop.getNumber(), 0.001);
		symop = ParsedSymop.createSymop(".25-y");
		Assert.assertEquals("xyz", "-y", symop.getXyz());
		Assert.assertEquals("xyz", "", symop.getXyz1());
		Assert.assertEquals("xyz", 0.25, symop.getNumber(), 0.001);
		symop = ParsedSymop.createSymop("-0.75+z");
		Assert.assertEquals("xyz", "z", symop.getXyz());
		Assert.assertEquals("xyz", "", symop.getXyz1());
		Assert.assertEquals("xyz", -0.75, symop.getNumber(), 0.001);
	}
	
	
	@Test
	public void parseNUMB_XYZXYZ() {
		ParsedSymop symop = ParsedSymop.createSymop("0.5+x+y");
		Assert.assertEquals("xyz", "x", symop.getXyz());
		Assert.assertEquals("xyz", "y", symop.getXyz1());
		Assert.assertEquals("xyz", 0.5, symop.getNumber(), 0.001);
		symop = ParsedSymop.createSymop(".25-y-x");
		Assert.assertEquals("xyz", "-y", symop.getXyz());
		Assert.assertEquals("xyz", "-x", symop.getXyz1());
		Assert.assertEquals("xyz", 0.25, symop.getNumber(), 0.001);
		symop = ParsedSymop.createSymop("-0.75+z-x");
		Assert.assertEquals("xyz", "z", symop.getXyz());
		Assert.assertEquals("xyz", "-x", symop.getXyz1());
		Assert.assertEquals("xyz", -0.75, symop.getNumber(), 0.001);
	}
	
	@Test
	public void parseFRACT_XYZ() {
		ParsedSymop symop = ParsedSymop.createSymop("1/2+x");
		Assert.assertEquals("xyz", "x", symop.getXyz());
		Assert.assertEquals("xyz", "", symop.getXyz1());
		Assert.assertEquals("xyz", 0.5, symop.getNumber(), 0.001);
		symop = ParsedSymop.createSymop("1/4-y");
		Assert.assertEquals("xyz", "-y", symop.getXyz());
		Assert.assertEquals("xyz", "", symop.getXyz1());
		Assert.assertEquals("xyz", 0.25, symop.getNumber(), 0.001);
		symop = ParsedSymop.createSymop("-3/4+z");
		Assert.assertEquals("xyz", "z", symop.getXyz());
		Assert.assertEquals("xyz", "", symop.getXyz1());
		Assert.assertEquals("xyz", -0.75, symop.getNumber(), 0.001);
	}
	
	@Test
	public void parseFRACT_XYZXYZ() {
		ParsedSymop symop = ParsedSymop.createSymop("1/2+x+y");
		Assert.assertEquals("xyz", "x", symop.getXyz());
		Assert.assertEquals("xyz", "y", symop.getXyz1());
		Assert.assertEquals("xyz", 0.5, symop.getNumber(), 0.001);
		symop = ParsedSymop.createSymop("1/4-y-x");
		Assert.assertEquals("xyz", "-y", symop.getXyz());
		Assert.assertEquals("xyz", "-x", symop.getXyz1());
		Assert.assertEquals("xyz", 0.25, symop.getNumber(), 0.001);
		symop = ParsedSymop.createSymop("-3/4+z-x");
		Assert.assertEquals("xyz", "z", symop.getXyz());
		Assert.assertEquals("xyz", "-x", symop.getXyz1());
		Assert.assertEquals("xyz", -0.75, symop.getNumber(), 0.001);
	}
	
	
	
	@Test
	public void parseXYZ_NUMB() {
		ParsedSymop symop = ParsedSymop.createSymop("x+0.5");
		Assert.assertEquals("xyz", "x", symop.getXyz());
		Assert.assertEquals("xyz", "", symop.getXyz1());
		Assert.assertEquals("xyz", 0.5, symop.getNumber(), 0.001);
		symop = ParsedSymop.createSymop("-y+.25");
		Assert.assertEquals("xyz", "-y", symop.getXyz());
		Assert.assertEquals("xyz", "", symop.getXyz1());
		Assert.assertEquals("xyz", 0.25, symop.getNumber(), 0.001);
		symop = ParsedSymop.createSymop("z-0.75");
		Assert.assertEquals("xyz", "z", symop.getXyz());
		Assert.assertEquals("xyz", "", symop.getXyz1());
		Assert.assertEquals("xyz", -0.75, symop.getNumber(), 0.001);
	}
	
	
	@Test
	public void parseXYZXYZ_NUMB() {
		ParsedSymop symop = ParsedSymop.createSymop("x+y+0.5");
		Assert.assertEquals("xyz", "x", symop.getXyz());
		Assert.assertEquals("xyz", "y", symop.getXyz1());
		Assert.assertEquals("xyz", 0.5, symop.getNumber(), 0.001);
		symop = ParsedSymop.createSymop("-y-x+.25");
		Assert.assertEquals("xyz", "-y", symop.getXyz());
		Assert.assertEquals("xyz", "-x", symop.getXyz1());
		Assert.assertEquals("xyz", 0.25, symop.getNumber(), 0.001);
		symop = ParsedSymop.createSymop("z-x-0.75");
		Assert.assertEquals("xyz", "z", symop.getXyz());
		Assert.assertEquals("xyz", "-x", symop.getXyz1());
		Assert.assertEquals("xyz", -0.75, symop.getNumber(), 0.001);
	}
	
	@Test
	public void parseXYZ_FRACT() {
		ParsedSymop symop = ParsedSymop.createSymop("x+1/2");
		Assert.assertEquals("xyz", "x", symop.getXyz());
		Assert.assertEquals("xyz", "", symop.getXyz1());
		Assert.assertEquals("xyz", 0.5, symop.getNumber(), 0.001);
		symop = ParsedSymop.createSymop("-y+1/4");
		Assert.assertEquals("xyz", "-y", symop.getXyz());
		Assert.assertEquals("xyz", "", symop.getXyz1());
		Assert.assertEquals("xyz", 0.25, symop.getNumber(), 0.001);
		symop = ParsedSymop.createSymop("z-3/4");
		Assert.assertEquals("xyz", "z", symop.getXyz());
		Assert.assertEquals("xyz", "", symop.getXyz1());
		Assert.assertEquals("xyz", -0.75, symop.getNumber(), 0.001);
	}
	
	@Test
	public void parseXYZXYZ_FRACT() {
		ParsedSymop symop = ParsedSymop.createSymop("x+y+1/2");
		Assert.assertEquals("xyz", "x", symop.getXyz());
		Assert.assertEquals("xyz", "y", symop.getXyz1());
		Assert.assertEquals("xyz", 0.5, symop.getNumber(), 0.001);
		symop = ParsedSymop.createSymop("-y-x+1/4");
		Assert.assertEquals("xyz", "-y", symop.getXyz());
		Assert.assertEquals("xyz", "-x", symop.getXyz1());
		Assert.assertEquals("xyz", 0.25, symop.getNumber(), 0.001);
		symop = ParsedSymop.createSymop("z-x-3/4");
		Assert.assertEquals("xyz", "z", symop.getXyz());
		Assert.assertEquals("xyz", "-x", symop.getXyz1());
		Assert.assertEquals("xyz", -0.75, symop.getNumber(), 0.001);
	}
	
	@Test
	public void createTransformTest() {
		Transform3 transform = ParsedSymop.createTransform(new String[]{"x+y+1/2", "-y-x+1/4", "z-x-3/4"});
		Assert.assertEquals("matrix", "{4,4}\n"+
"(1.0,1.0,0.0,0.5)\n"+
"(-1.0,-1.0,0.0,0.25)\n"+
"(-1.0,0.0,1.0,-0.75)\n"+
"(0.0,0.0,0.0,1.0)", transform.toString());
	}
	
	
}
