/*
 * Copyright (C) 2024 Tobias Brunner
 * Copyright (C) 2018-2024 Andreas Steffen
 *
 * Copyright (C) secunet Security Networks AG
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the Licenseor (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be usefulbut
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include <crypto/crypto_tester.h>

/**
 * First four test vectors of the liboqs-0.11.0 ML-KEM-512 KAT
 */
ke_test_vector_t mlkem512_0 = {
	.method = ML_KEM_512,
	.seed = chunk_from_chars(
		0x06,0x15,0x50,0x23,0x4D,0x15,0x8C,0x5E,0xC9,0x55,
		0x95,0xFE,0x04,0xEF,0x7A,0x25,0x76,0x7F,0x2E,0x24,
		0xCC,0x2B,0xC4,0x79,0xD0,0x9D,0x86,0xDC,0x9A,0xBC,
		0xFD,0xE7,0x05,0x6A,0x8C,0x26,0x6F,0x9E,0xF9,0x7E,
		0xD0,0x85,0x41,0xDB,0xD2,0xE1,0xFF,0xA1),
	.pub_i = chunk_from_chars(
		0x40,0x08,0x65,0xED,0x10,0xB6,0x19,0xAA,0x58,0x11,
		0x13,0x9B,0xC0,0x86,0x82,0x57,0x82,0xB2,0xB7,0x12,
		0x4F,0x75,0x7C,0x83,0xAE,0x79,0x44,0x44,0xBC,0x78,
		0xA4,0x78,0x96,0xAC,0xF1,0x26,0x2C,0x81,0x35,0x10,
		0x77,0x89,0x3B,0xFC,0x56,0xF9,0x04,0x49,0xC2,0xFA,
		0x5F,0x6E,0x58,0x6D,0xD3,0x7C,0x0B,0x9B,0x58,0x19,
		0x92,0x63,0x8C,0xB7,0xE7,0xBC,0xBB,0xB9,0x9A,0xFE,
		0x47,0x81,0xD8,0x0A,0x50,0xE6,0x94,0x63,0xFB,0xD9,
		0x88,0x72,0x2C,0x36,0x35,0x42,0x3E,0x27,0x46,0x6C,
		0x71,0xDC,0xC6,0x74,0x52,0x7C,0xCD,0x72,0x89,0x68, /* 100 */

		0xCB,0xCD,0xC0,0x0C,0x5C,0x90,0x35,0xBB,0x0A,0xF2,
		0xC9,0x92,0x2C,0x78,0x81,0xA4,0x1D,0xD2,0x87,0x52,
		0x73,0x92,0x51,0x31,0x23,0x0F,0x6C,0xA5,0x9E,0x91,
		0x36,0xB3,0x9F,0x95,0x6C,0x93,0xB3,0xB2,0xD1,0x4C,
		0x64,0x1B,0x08,0x9E,0x07,0xD0,0xA8,0x40,0xC8,0x93,
		0xEC,0xD7,0x6B,0xBF,0x92,0xC8,0x05,0x45,0x66,0x68,
		0xD0,0x7C,0x62,0x14,0x91,0xC5,0xC0,0x54,0x99,0x1A,
		0x65,0x6F,0x51,0x16,0x19,0x55,0x6E,0xB9,0x77,0x82,
		0xE2,0x7A,0x3C,0x78,0x51,0x24,0xC7,0x0B,0x0D,0xAB,
		0xA6,0xC6,0x24,0xD1,0x8E,0x0F,0x97,0x93,0xF9,0x6B, /* 200 */

		0xA9,0xE1,0x59,0x9B,0x17,0xB3,0x0D,0xCC,0xC0,0xB4,
		0xF3,0x76,0x6A,0x07,0xB2,0x3B,0x25,0x73,0x09,0xCD,
		0x76,0xAB,0xA0,0x72,0xC2,0xB9,0xC9,0x74,0x43,0x94,
		0xC6,0xAB,0x9C,0xB6,0xC5,0x4A,0x97,0xB5,0xC5,0x78,
		0x61,0xA5,0x8D,0xC0,0xA0,0x35,0x19,0x83,0x2E,0xE3,
		0x2A,0x07,0x65,0x4A,0x07,0x0C,0x0C,0x8C,0x4E,0x86,
		0x48,0xAD,0xDC,0x35,0x5F,0x27,0x4F,0xC6,0xB9,0x2A,
		0x08,0x7B,0x3F,0x97,0x51,0x92,0x3E,0x44,0x27,0x4F,
		0x85,0x8C,0x49,0xCA,0xBA,0x72,0xB6,0x58,0x51,0xB3,
		0xAD,0xC4,0x89,0x36,0x95,0x50,0x97,0xCA,0xD9,0x55, /* 300 */

		0x3F,0x5A,0x26,0x3F,0x18,0x44,0xB5,0x2A,0x02,0x0F,
		0xF7,0xCA,0x89,0xE8,0x81,0xA0,0x1B,0x95,0xD9,0x57,
		0xA3,0x15,0x3C,0x0A,0x5E,0x0A,0x1C,0xCD,0x66,0xB1,
		0x82,0x1A,0x2B,0x86,0x32,0x54,0x6E,0x24,0xC7,0xCB,
		0xBC,0x4C,0xB0,0x88,0x08,0xCA,0xC3,0x7F,0x7D,0xA6,
		0xB1,0x6F,0x8A,0xCE,0xD0,0x52,0xCD,0xB2,0x56,0x49,
		0x48,0xF1,0xAB,0x0F,0x76,0x8A,0x0D,0x32,0x86,0xCC,
		0xC7,0xC3,0x74,0x9C,0x63,0xC7,0x81,0x53,0x0F,0xA1,
		0xAE,0x67,0x05,0x42,0x85,0x50,0x04,0xA6,0x45,0xB5,
		0x22,0x88,0x1E,0xC1,0x41,0x2B,0xDA,0xE3,0x42,0x08, /* 400 */

		0x5A,0x9D,0xD5,0xF8,0x12,0x6A,0xF9,0x6B,0xBD,0xB0,
		0xC1,0xAF,0x69,0xA1,0x55,0x62,0xCB,0x2A,0x15,0x5A,
		0x10,0x03,0x09,0xD1,0xB6,0x41,0xD0,0x8B,0x2D,0x4E,
		0xD1,0x7B,0xFB,0xF0,0xBC,0x04,0x26,0x5F,0x9B,0x10,
		0xC1,0x08,0xF8,0x50,0x30,0x95,0x04,0xD7,0x72,0x81,
		0x1B,0xBA,0x8E,0x2B,0xE1,0x62,0x49,0xAA,0x73,0x7D,
		0x87,0x9F,0xC7,0xFB,0x25,0x5E,0xE7,0xA6,0xA0,0xA7,
		0x53,0xBD,0x93,0x74,0x1C,0x61,0x65,0x8E,0xC0,0x74,
		0xF6,0xE0,0x02,0xB0,0x19,0x34,0x57,0x69,0x11,0x3C,
		0xC0,0x13,0xFF,0x74,0x94,0xBA,0x83,0x78,0xB1,0x1A, /* 500 */

		0x17,0x22,0x60,0xAA,0xA5,0x34,0x21,0xBD,0xE0,0x3A,
		0x35,0x58,0x9D,0x57,0xE3,0x22,0xFE,0xFA,0x41,0x00,
		0xA4,0x74,0x39,0x26,0xAB,0x7D,0x62,0x25,0x8B,0x87,
		0xB3,0x1C,0xCB,0xB5,0xE6,0xB8,0x9C,0xB1,0x0B,0x27,
		0x1A,0xA0,0x5D,0x99,0x4B,0xB5,0x70,0x8B,0x23,0xAB,
		0x32,0x7E,0xCB,0x93,0xC0,0xF3,0x15,0x68,0x69,0xF0,
		0x88,0x3D,0xA2,0x06,0x4F,0x79,0x5E,0x0E,0x2A,0xB7,
		0xD3,0xC6,0x4D,0x61,0xD2,0x30,0x3F,0xC3,0xA2,0x9E,
		0x16,0x19,0x92,0x3C,0xA8,0x01,0xE5,0x9F,0xD7,0x52,
		0xCA,0x6E,0x76,0x49,0xD3,0x03,0xC9,0xD2,0x07,0x88, /* 600 */

		0xE1,0x21,0x46,0x51,0xB0,0x69,0x95,0xEB,0x26,0x0C,
		0x92,0x9A,0x13,0x44,0xA8,0x49,0xB2,0x5C,0xA0,0xA0,
		0x1F,0x1E,0xB5,0x29,0x13,0x68,0x6B,0xBA,0x61,0x9E,
		0x23,0x71,0x44,0x64,0x03,0x1A,0x78,0x43,0x92,0x87,
		0xFC,0xA7,0x8F,0x4C,0x04,0x76,0x22,0x3E,0xEA,0x61,
		0xB7,0xF2,0x5A,0x7C,0xE4,0x2C,0xCA,0x90,0x1B,0x2A,
		0xEA,0x12,0x98,0x17,0x89,0x4B,0xA3,0x47,0x08,0x23,
		0x85,0x4F,0x3E,0x5B,0x28,0xD8,0x6B,0xA9,0x79,0xE5,
		0x46,0x71,0x86,0x2D,0x90,0x47,0x0B,0x1E,0x78,0x38,
		0x97,0x2A,0x81,0xA4,0x81,0x07,0xD6,0xAC,0x06,0x11, /* 700 */

		0x40,0x6B,0x21,0xFB,0xCC,0xE1,0xDB,0x77,0x02,0xEA,
		0x9D,0xD6,0xBA,0x6E,0x40,0x52,0x7B,0x9D,0xC6,0x63,
		0xF3,0xC9,0x3B,0xAD,0x05,0x6D,0xC2,0x85,0x11,0xF6,
		0x6C,0x3E,0x0B,0x92,0x8D,0xB8,0x87,0x9D,0x22,0xC5,
		0x92,0x68,0x5C,0xC7,0x75,0xA6,0xCD,0x57,0x4A,0xC3,
		0xBC,0xE3,0xB2,0x75,0x91,0xC8,0x21,0x92,0x90,0x76,
		0x35,0x8A,0x22,0x00,0xB3,0x77,0x36,0x5F,0x7E,0xFB,
		0x9E,0x40,0xC3,0xBF,0x0F,0xF0,0x43,0x29,0x86,0xAE,
		0x4B,0xC1,0xA2,0x42,0xCE,0x99,0x21,0xAA,0x9E,0x22,
		0x44,0x88,0x19,0x58,0x5D,0xEA,0x30,0x8E,0xB0,0x39),
	.pub_r = chunk_from_chars(
		0x52,0x1C,0x88,0x48,0x6C,0x35,0xF6,0xC2,0x45,0x83,
		0x92,0x12,0xAB,0x0E,0x23,0x66,0x0C,0xD5,0xB6,0x8F,
		0xCC,0xD5,0xA7,0xB4,0x1E,0xB5,0xA3,0xCE,0x88,0x44,
		0xA3,0x10,0x88,0xC8,0x78,0xEE,0xFE,0xB4,0x47,0x39,
		0xCF,0x91,0x30,0x01,0x3A,0x83,0xFA,0xAA,0x78,0x03,
		0x74,0x43,0xE5,0xD7,0x49,0xBA,0x4D,0x6F,0x15,0x69,
		0x34,0xCC,0x89,0xC2,0xD9,0xAB,0xC7,0x6C,0xB7,0xFF,
		0x05,0x0B,0x4E,0xEE,0xB4,0xA5,0x86,0x11,0xBE,0x33,
		0x0B,0x3F,0xDE,0xE8,0x75,0xC1,0xF3,0x66,0x21,0x6A,
		0xD6,0x59,0xFA,0xBB,0xEB,0xCE,0x37,0x11,0x4E,0x79, /* 100 */

		0x5C,0x65,0xF1,0xEE,0xCA,0x93,0x18,0x13,0x43,0x00,
		0x54,0x10,0xFE,0xBA,0xE0,0x42,0xDF,0xAE,0xEA,0xD8,
		0x73,0xCF,0x1C,0x57,0x5D,0x38,0xCE,0x26,0xEC,0x5C,
		0x02,0x94,0x0C,0x02,0x24,0xE9,0x83,0x88,0x1C,0x2A,
		0x1A,0x47,0x71,0xBA,0x31,0x66,0x28,0xA0,0xF4,0x25,
		0xEF,0x54,0xE9,0x84,0xFE,0x70,0xE3,0x86,0x6C,0x79,
		0x78,0x0B,0x75,0x72,0x46,0x2C,0xE5,0xA9,0xE1,0x16,
		0xB5,0x54,0x39,0xAE,0x92,0x1F,0xF8,0xB0,0xD8,0x9D,
		0x86,0x16,0xD4,0x05,0x13,0x5D,0xFA,0xB8,0xF1,0x4D,
		0x7D,0xA0,0x3F,0x75,0x25,0x17,0xDA,0x84,0x74,0x58, /* 200 */

		0xAB,0x83,0x64,0x6C,0xE5,0xB4,0x07,0x37,0x88,0xC6,
		0x6A,0x6B,0x60,0xFA,0xF6,0x4B,0x8F,0xED,0x50,0x7E,
		0xE2,0xA7,0xD9,0x31,0xF7,0x46,0xB9,0xF2,0x59,0x57,
		0x69,0x72,0x1A,0x59,0xD9,0x3E,0x48,0x52,0xAA,0xF8,
		0x18,0x51,0x14,0xF4,0xA0,0x4F,0x0F,0x6F,0x3C,0xA1,
		0x44,0xBA,0x8E,0xE1,0xBA,0x52,0xDB,0x4A,0xA7,0xDC,
		0x27,0x41,0x56,0x86,0x28,0x12,0xDC,0x36,0xE0,0x69,
		0x97,0x94,0x2B,0xAB,0x02,0x82,0x2B,0xFC,0x5F,0xDF,
		0xCD,0xAC,0xEA,0x86,0x9C,0x1A,0x76,0x72,0xA4,0xC7,
		0x94,0xC9,0xC0,0x9C,0xC8,0xA7,0x6D,0xF8,0x94,0x32, /* 300 */

		0x4C,0x14,0xA5,0x3E,0x99,0x61,0xCF,0x40,0xF0,0xE7,
		0x0D,0xC1,0x85,0x83,0xAA,0x5E,0x3D,0x02,0x5A,0x5B,
		0x8D,0x9C,0xED,0xA7,0x1D,0x79,0x02,0xEB,0xC5,0xD4,
		0x99,0xF0,0x59,0x38,0x6B,0x99,0x10,0xC7,0x5B,0xA8,
		0x34,0xB9,0xD0,0xC7,0x0A,0xD9,0xB9,0xEA,0x68,0x3A,
		0xA6,0x99,0x86,0x5F,0x9C,0xA7,0xF3,0xF3,0x0D,0x20,
		0xB7,0x8F,0xF9,0x98,0x50,0x21,0x6A,0x62,0xF9,0x19,
		0xA9,0xD9,0xEC,0xA4,0x82,0xA5,0x2E,0xAA,0x25,0x00,
		0xFE,0x5B,0x80,0x85,0x3C,0xBB,0x88,0xE1,0x7C,0xE5,
		0x93,0xEB,0x23,0x70,0x9B,0xAC,0x01,0xFD,0xFC,0x94, /* 400 */

		0x1B,0x52,0x7F,0x51,0x80,0xE0,0xDE,0xCC,0x37,0x85,
		0xF0,0x4D,0x91,0x20,0x09,0x8F,0x14,0xC0,0x7F,0x92,
		0x44,0xB4,0x41,0xF2,0x89,0x7F,0x24,0x3C,0x84,0x6A,
		0x1D,0x09,0x3D,0x6A,0x9C,0x0B,0x40,0xE8,0x42,0xA6,
		0xD1,0x2E,0x1D,0x2E,0x01,0xBB,0x44,0x69,0x3D,0x61,
		0xC8,0x75,0xEF,0x00,0x76,0x73,0x78,0x7A,0xAF,0x16,
		0x7C,0x1E,0xC2,0xB2,0xF6,0x1A,0xB8,0xB5,0x04,0x03,
		0x2A,0x14,0x49,0x0C,0x10,0x9A,0x0C,0x2A,0xEE,0x87,
		0x2F,0xCD,0x62,0x95,0x94,0x99,0x2E,0xBD,0x6D,0xCD,
		0xE4,0x2F,0xF6,0xA6,0x02,0xA5,0xC7,0xE1,0x5F,0x50, /* 500 */

		0xB7,0x99,0xA7,0x78,0x08,0x29,0xDB,0x1C,0xB2,0xE7,
		0x0E,0x89,0x94,0x4C,0xF5,0x43,0x22,0x4D,0x43,0x39,
		0xCC,0xF3,0x17,0xA0,0xBA,0x19,0x5A,0x07,0xDF,0x0F,
		0x43,0xD7,0xEE,0xE2,0x40,0x00,0x80,0xDA,0x25,0xA4,
		0x0F,0x32,0x00,0x61,0xB1,0x5A,0xE2,0x3E,0xA0,0xDE,
		0xE4,0x24,0x74,0xB2,0x27,0x4D,0x92,0xC7,0x2C,0x7E,
		0x82,0xF9,0x38,0xBF,0x82,0x69,0x34,0xCA,0x2A,0xAA,
		0xCA,0x49,0xCD,0x73,0xEB,0x36,0xD1,0x82,0x59,0x1B,
		0x81,0x45,0xD8,0x9A,0xC8,0xD6,0xCE,0xB7,0xBE,0x8A,
		0x1D,0x79,0x60,0xD0,0x41,0x71,0xD7,0xD0,0x3D,0x84, /* 600 */

		0x58,0x0B,0xCA,0x9B,0x59,0x76,0xAD,0x1E,0xD6,0xCC,
		0x8B,0x02,0x1B,0xEE,0xCD,0xBC,0xC8,0xB5,0x1A,0x9B,
		0x09,0x1C,0x66,0x25,0x86,0x10,0x97,0xA3,0x2F,0xB5,
		0xA4,0x1E,0x15,0xB8,0x56,0xCD,0xA1,0x35,0xC3,0xCA,
		0x29,0xC8,0x65,0x66,0x03,0xCE,0x3E,0xB7,0x80,0x71,
		0x49,0x41,0x97,0xF0,0x90,0x6D,0x8B,0x2A,0x2C,0xB2,
		0x08,0x07,0x6E,0xC8,0x9C,0xE5,0x76,0x0B,0x19,0x9E,
		0x93,0x7E,0x13,0xFE,0xBC,0x78,0x93,0x66,0x5A,0xB6,
		0xB2,0xD5,0xC8,0x5D,0xC9,0xA5,0xD8,0x73,0xCB,0xF5,
		0x5B,0x4A,0x69,0x34,0x3D,0x76,0x8F,0xBE,0xEF,0x4B, /* 700 */

		0x5E,0xB8,0x8D,0x0C,0x31,0xFF,0xD3,0x66,0xC6,0x6E,
		0x13,0x86,0x6E,0x3F,0x33,0xEE,0xCB,0xF2,0xC3,0x32,
		0x9C,0x11,0x1C,0x0C,0xDE,0x2B,0x95,0x60,0x89,0x2C,
		0xE1,0xA2,0x68,0x6A,0x2A,0x1C,0x18,0xB7,0xA7,0x26,
		0x1A,0x55,0xBD,0xA5,0x7A,0xDE,0x24,0x15,0x44,0xF3,
		0x56,0x13,0x90,0xBD,0xC6,0x95,0x14,0x42,0x9C,0x8D,
		0x5F,0xBE,0xA9,0x18,0x8B,0xAF,0x28,0x92),
	.shared = chunk_from_chars(
		0xB4,0xC8,0xE3,0xC4,0x11,0x5F,0x95,0x11,0xF2,0xFD,
		0xDB,0x28,0x8C,0x4B,0x78,0xC5,0xCD,0x7C,0x89,0xD2,
		0xD4,0xD3,0x21,0xF4,0x6B,0x4E,0xDC,0x54,0xDD,0xF0,
		0xEB,0x36),
};

ke_test_vector_t mlkem512_1 = {
	.method = ML_KEM_512,
	.seed = chunk_from_chars(
		0xD8,0x1C,0x4D,0x8D,0x73,0x4F,0xCB,0xFB,0xEA,0xDE,
		0x3D,0x3F,0x8A,0x03,0x9F,0xAA,0x2A,0x2C,0x99,0x57,
		0xE8,0x35,0xAD,0x55,0xB2,0x2E,0x75,0xBF,0x57,0xBB,
		0x55,0x6A,0xC8,0x1A,0xDD,0xE6,0xAE,0xEB,0x4A,0x5A,
		0x87,0x5C,0x3B,0xFC,0xAD,0xFA,0x95,0x8F),
	.pub_i = chunk_from_chars(
		0x4B,0x59,0x44,0x72,0x62,0xA0,0xBD,0xBA,0xBD,0xCD,
		0xDB,0x3F,0xFB,0xD1,0xC9,0x58,0xAA,0x04,0xAC,0x8A,
		0xC8,0x15,0xE0,0x54,0x03,0x9C,0xA2,0x08,0x4C,0xCD,
		0xE3,0x4B,0x1B,0x58,0x21,0x2F,0x8D,0xCB,0x9C,0x8D,
		0x78,0x67,0xD7,0x81,0x20,0xC2,0xD8,0x4C,0xCC,0x20,
		0xC0,0xF1,0x36,0x77,0x3B,0x90,0x08,0x8F,0x27,0x3B,
		0x01,0x7C,0x4C,0x29,0x40,0x40,0x43,0x59,0x31,0x92,
		0x20,0x1F,0x38,0xEB,0xA1,0xD4,0xB3,0x26,0xC6,0x53,
		0xBF,0x9B,0x91,0xA7,0xF4,0x38,0x7F,0x0A,0xE8,0x76,
		0xF6,0x27,0xB3,0xFF,0x87,0x8D,0xC3,0x8A,0xA3,0x00, /* 100 */

		0x7A,0x1A,0x98,0x38,0x7C,0x32,0x93,0x39,0x6A,0x74,
		0x33,0x7A,0xD5,0x12,0x97,0x07,0x5E,0x09,0xC1,0x19,
		0x7C,0xF5,0x0E,0xA8,0x3C,0x85,0xF7,0x08,0x5C,0x1C,
		0x81,0xB0,0x7B,0x87,0x51,0x4F,0x67,0x89,0x8C,0x64,
		0xC9,0x86,0x3B,0x08,0x74,0x4C,0xAE,0x06,0xDA,0x57,
		0xC5,0x1B,0xCC,0x7A,0xD5,0x88,0x86,0x98,0x06,0xC4,
		0xFB,0x4E,0x34,0x30,0x81,0x2A,0xF2,0x4A,0xB7,0xC1,
		0x40,0x9E,0xD8,0x9C,0x02,0x70,0xA8,0x63,0x2A,0x4A,
		0xB6,0x6A,0x10,0x09,0xEC,0x02,0x53,0xB4,0xB1,0x28,
		0x69,0x0F,0x1A,0xD7,0xB3,0xDB,0x74,0x3A,0x1F,0x85, /* 200 */

		0xAA,0x27,0xD3,0x7C,0x70,0xC8,0x32,0xB2,0xB6,0x84,
		0xB5,0xB5,0x2A,0xA6,0xA7,0x04,0x2A,0x60,0x1D,0xCD,
		0xA0,0x65,0xF8,0xF4,0x40,0x08,0x19,0x65,0x04,0x62,
		0x4E,0x72,0xA5,0x09,0x9C,0x68,0x6B,0x82,0xA2,0x83,
		0x36,0x0C,0x85,0xAD,0x42,0x5E,0xD7,0xC5,0xAF,0x71,
		0xC2,0x37,0x47,0xE2,0x14,0x76,0x46,0x22,0x20,0xF9,
		0x43,0xA9,0x38,0x14,0x93,0xE7,0x06,0x06,0x21,0xB3,
		0xD1,0x96,0x74,0xFA,0x1B,0xAC,0x0B,0xD0,0xB3,0xB1,
		0xCC,0x48,0x4D,0x7B,0x64,0x52,0xD9,0x7D,0x42,0x89,
		0x8A,0xA6,0x36,0x74,0xB2,0x5A,0x51,0xDB,0xC6,0x55, /* 300 */

		0xC9,0xE7,0xB6,0x47,0x68,0x1C,0x7A,0x0A,0x1D,0x47,
		0x14,0x63,0x47,0x54,0xCD,0x24,0x01,0xC0,0xDF,0xC4,
		0x02,0x3B,0x1B,0x47,0x5E,0xA1,0x26,0x9E,0x37,0x3E,
		0xBB,0x74,0x0C,0x92,0xD7,0xBE,0x97,0x75,0x59,0x19,
		0xC6,0x3A,0xA9,0x39,0xA4,0x71,0xBC,0x63,0xD4,0xB0,
		0x27,0x79,0x9A,0x34,0xB2,0xFB,0x24,0xA8,0x69,0x0C,
		0x47,0x45,0x38,0x59,0x22,0xB9,0x42,0xF0,0x83,0xB6,
		0xC3,0x2A,0x0D,0x5C,0x87,0xF7,0xA3,0xB9,0xFE,0x16,
		0x8F,0xA8,0xB6,0x0D,0x5F,0x32,0x47,0xF9,0x84,0x0C,
		0x13,0x25,0x9F,0x99,0x23,0x35,0xDE,0xBB,0x52,0xE7, /* 400 */

		0xE0,0x80,0x65,0x26,0x42,0x91,0xF2,0x50,0xC3,0x51,
		0xC8,0x68,0xD5,0xCC,0xB4,0x9C,0x86,0x5B,0xE3,0x3C,
		0x13,0x7C,0x25,0xBC,0xB5,0x2B,0x80,0x93,0x51,0xC4,
		0x76,0xBE,0x47,0xBB,0x3D,0xD7,0xB8,0x5A,0xAE,0x0A,
		0x99,0x4C,0xA7,0x46,0x82,0x95,0x97,0x95,0x8A,0x60,
		0xEA,0x55,0xBD,0xE8,0x87,0xC5,0x2A,0x38,0xA2,0xDB,
		0xF4,0x90,0xCD,0x50,0x6D,0x0D,0xA1,0x93,0x40,0xD6,
		0xCB,0x4D,0x1C,0x1F,0x5C,0x84,0xB4,0xFC,0x53,0x57,
		0xE2,0x1C,0x0A,0x44,0xEC,0x3A,0x49,0x8A,0xAA,0x28,
		0x8C,0x2B,0x4E,0x21,0x96,0xE1,0xAC,0x6E,0x0B,0x42, /* 500 */

		0x13,0x55,0x26,0x21,0xAF,0x04,0x91,0x45,0x68,0x7C,
		0x4E,0x69,0x36,0x0A,0x39,0xAF,0xD6,0x9B,0x38,0xD9,
		0x40,0x98,0x42,0xA6,0x5C,0x36,0xA5,0xCA,0x89,0xB1,
		0x15,0x89,0x72,0x97,0xD5,0xC0,0x1D,0x19,0x71,0x54,
		0xCA,0xEB,0x77,0xA4,0x42,0x89,0xA3,0xF9,0x3E,0x56,
		0x56,0x90,0xDE,0x39,0x95,0x50,0x18,0x04,0xDC,0x51,
		0x9F,0x75,0x69,0xBC,0xC9,0x6A,0x22,0xE7,0x19,0x84,
		0x0C,0x54,0x2D,0xA7,0x79,0xC9,0x43,0x78,0x0A,0x3C,
		0x84,0x3E,0xFE,0xEC,0x1A,0x06,0x60,0x50,0xC6,0xE4,
		0x7C,0x4A,0x39,0xAD,0x6E,0x58,0x26,0x6C,0xA5,0x10, /* 600 */

		0xFF,0x3A,0xB5,0xDF,0xF3,0x7A,0x2E,0x34,0xBA,0xF3,
		0x6A,0xBE,0x5B,0x09,0x79,0x1F,0x7B,0x8E,0x23,0x91,
		0x43,0x1D,0x5A,0x6A,0x0B,0x12,0xBA,0x07,0x70,0x27,
		0xFF,0xDB,0xB0,0xFF,0xAA,0xB7,0x28,0x06,0xD0,0x48,
		0xFA,0x06,0x6F,0xCB,0x4C,0x48,0x78,0x20,0xA5,0x6B,
		0x4B,0xD1,0x05,0x67,0xF7,0x55,0xBE,0xDE,0x04,0xC9,
		0x13,0x76,0x33,0x0F,0x79,0x7A,0x56,0xD5,0x9A,0x46,
		0x55,0x44,0xB0,0x2B,0xA6,0x8B,0x96,0xA0,0xF9,0x58,
		0x23,0xD5,0xE3,0xB2,0xC3,0x6C,0x90,0x42,0x3A,0xC5,
		0x50,0xAB,0xC7,0x96,0xBC,0x53,0x83,0x3A,0xAC,0xE0, /* 700 */

		0x05,0xA0,0x6B,0x81,0x4D,0xB1,0xD8,0xC6,0xA4,0x06,
		0xBB,0x57,0x52,0xCF,0xB2,0xB0,0x3A,0x08,0x24,0xAD,
		0xB1,0xEC,0x26,0x5E,0x52,0x30,0xB4,0x47,0xA5,0x08,
		0x29,0x23,0xEF,0x1B,0x03,0x95,0xF6,0x6C,0x0F,0xB9,
		0x13,0xA9,0xA1,0x33,0xB2,0x96,0x34,0x71,0x6B,0x65,
		0xE8,0x89,0xB0,0xF8,0x0C,0x4B,0x02,0x57,0x5A,0xD3,
		0xB8,0x01,0x83,0x31,0xA6,0x7C,0x35,0x02,0xC6,0x19,
		0x53,0x22,0xBD,0x2B,0x71,0xFD,0x95,0x8C,0xF0,0x71,
		0x80,0x65,0x87,0x64,0x8F,0xB3,0x1F,0x7E,0x4E,0xE9,
		0xEA,0xD4,0x8E,0x00,0x52,0xB0,0x62,0x44,0xF3,0xD1),
	.pub_r = chunk_from_chars(
		0xE6,0x96,0x2D,0xEA,0xEA,0xCB,0xBD,0x25,0xBA,0x54,
		0x55,0x31,0xF1,0xEB,0x0A,0x3E,0xFF,0x86,0x6F,0xB1,
		0x01,0xF5,0xBC,0x0A,0xA0,0xC5,0x0F,0x3A,0x92,0x63,
		0xD1,0x44,0x34,0xE1,0xC6,0x78,0xC3,0xD1,0x73,0x8A,
		0x97,0xEA,0x7A,0xFB,0xEA,0x27,0x5B,0x4E,0xEB,0xD9,
		0x57,0xFB,0x27,0x01,0x52,0x12,0x6F,0x6C,0xF7,0x16,
		0x78,0x18,0xA4,0xCF,0xE0,0xC6,0xE1,0x4A,0x75,0xA1,
		0x77,0xDE,0x24,0x3D,0x75,0x62,0x82,0x9A,0x27,0x2C,
		0x21,0x91,0xBD,0xF8,0xA0,0x89,0xA7,0x54,0x01,0x48,
		0x5D,0xF5,0x6A,0x5F,0x33,0x2C,0xE3,0x9A,0x39,0x97, /* 100 */

		0xDC,0xED,0x32,0x19,0x4B,0x20,0x0E,0xC5,0xAD,0x84,
		0xA8,0xBE,0x14,0xFF,0xDB,0xF8,0x68,0xDC,0x99,0x21,
		0x2C,0x37,0x6D,0xD9,0xAB,0x50,0xED,0xA3,0xAD,0x20,
		0x54,0x69,0x91,0x6A,0xD2,0x50,0x4E,0xC8,0xA3,0x8A,
		0x96,0xFA,0xF3,0xF4,0x4C,0xDC,0xE4,0xC2,0x3F,0x81,
		0xAC,0xE3,0x45,0x13,0x3A,0xCB,0xFC,0x07,0x05,0x36,
		0xA2,0x35,0xE9,0xE3,0x97,0x40,0x0B,0x02,0x6B,0xD3,
		0x48,0xEE,0x45,0x40,0x37,0x94,0x8D,0x14,0x5A,0xD8,
		0x2B,0xF9,0x2D,0xCF,0x69,0xCF,0x5F,0xC5,0x20,0x04,
		0xAA,0x04,0xFD,0xB6,0x5F,0x97,0x4F,0x43,0x80,0xCC, /* 200 */

		0xCF,0xFE,0xF8,0xF7,0x3B,0x7F,0xBB,0x20,0xE1,0xCD,
		0x89,0x4B,0xAB,0xC6,0x1B,0xD0,0x35,0x67,0x10,0x6B,
		0xD2,0xE6,0x37,0xB2,0x29,0x7A,0xBA,0x33,0x9C,0x8D,
		0x2E,0xAA,0x96,0xDC,0xE1,0x1F,0x9F,0x84,0xF2,0xD5,
		0xA3,0xCD,0xFA,0xD6,0x1E,0x3A,0x5D,0xB5,0x74,0x98,
		0xC9,0xB8,0xAB,0xC6,0x79,0xB9,0xF6,0xFF,0x0F,0x8F,
		0x99,0x62,0x77,0x7D,0xE1,0x04,0xA7,0x40,0x5A,0x5B,
		0x66,0xD8,0xD9,0xD9,0xDC,0xB9,0x57,0x18,0x3E,0x56,
		0xCD,0x17,0x24,0xBF,0xEA,0x86,0xCB,0xB9,0x22,0x7A,
		0x34,0xB9,0xEE,0x46,0xDE,0xD9,0x00,0xF7,0x67,0x6A, /* 300 */

		0xC4,0x06,0x6A,0x07,0x6D,0x05,0x6D,0xA6,0xA3,0x55,
		0x56,0xD8,0xD6,0x39,0x0B,0xD6,0x75,0x60,0x6E,0xDC,
		0x0B,0x42,0xC9,0x47,0x37,0xE4,0x42,0x4C,0x18,0x72,
		0x4A,0x42,0x68,0x79,0x3C,0x9B,0x8C,0xA0,0x19,0x30,
		0x6E,0xC5,0xA3,0x8E,0x33,0x25,0xFD,0x0F,0x23,0x6C,
		0x32,0xF7,0x29,0xD4,0x29,0x1A,0x53,0x00,0xFA,0x96,
		0x55,0x11,0x88,0x07,0xF7,0x0C,0x7F,0xBD,0x32,0x79,
		0xE1,0x18,0x20,0xCA,0xD5,0x39,0x20,0x21,0xDF,0x8D,
		0xF2,0x50,0x20,0x98,0x64,0x53,0x91,0x17,0x7C,0xC3,
		0x28,0x97,0x52,0xEE,0xE7,0xE2,0xED,0x6D,0x18,0x95, /* 400 */

		0x8B,0xD4,0x8C,0x46,0xE1,0x15,0x15,0x2E,0x9C,0xD8,
		0x9A,0xD7,0x36,0xC2,0x18,0x28,0x83,0x0D,0x5C,0x97,
		0xDE,0x0A,0x95,0x46,0x94,0x8B,0xB3,0x92,0x8F,0xAF,
		0x8D,0xCF,0x13,0xC6,0xC6,0xDF,0x22,0x70,0xB0,0xF0,
		0xFD,0x27,0xC2,0x95,0x40,0x7B,0x76,0xD3,0x23,0x2A,
		0x23,0x41,0x04,0xA3,0x64,0x2C,0x92,0x3B,0xB4,0x19,
		0xF0,0x19,0xC2,0x13,0x90,0x9C,0x5C,0xD1,0xC2,0xCF,
		0xEE,0x6C,0x19,0x8A,0xD4,0x59,0xE9,0x60,0x27,0x6B,
		0xBF,0x0B,0x9C,0xEB,0x54,0x3D,0x78,0x79,0x04,0x34,
		0x1C,0x3C,0x43,0x88,0xE3,0x75,0xDD,0x32,0x13,0x83, /* 500 */

		0x5E,0x93,0x98,0x56,0x44,0xA3,0xDC,0x7B,0x8D,0x26,
		0xF2,0x8E,0xB7,0x28,0x5A,0x5A,0x60,0x38,0x1B,0x80,
		0x1A,0x63,0xD8,0x30,0xEC,0xB6,0x3E,0x1A,0x2A,0xA4,
		0x04,0xB0,0xE8,0x84,0xC8,0xD8,0xDD,0x41,0x0D,0x75,
		0xEE,0x66,0xC5,0x27,0x07,0x56,0x81,0xBA,0xC2,0xEE,
		0x71,0xC3,0xC9,0xE2,0xF7,0x39,0xAB,0x9D,0x24,0xDC,
		0xBB,0x1C,0xA8,0x9A,0x4C,0x83,0x63,0xE4,0xE6,0xD5,
		0x5B,0xE5,0x94,0xD3,0xC6,0x86,0x89,0x2E,0xC0,0xC0,
		0x3E,0x2C,0x53,0xD0,0xC9,0x33,0x1D,0x6D,0xF9,0xAC,
		0x9C,0x08,0x34,0xC2,0x57,0x90,0xAD,0xB5,0xF3,0x0E, /* 600 */

		0x06,0xB6,0x3E,0xDB,0xD8,0xA2,0xAA,0xEE,0xB5,0xFF,
		0x85,0xB4,0x43,0xC0,0xE7,0x8C,0x40,0xE0,0x60,0x29,
		0x06,0x6B,0x7C,0x86,0x74,0xFC,0xAD,0x4B,0x15,0xD9,
		0x94,0x63,0x59,0x59,0x1D,0xBB,0xB6,0xEE,0x42,0x01,
		0x1F,0xF3,0xD1,0x45,0xCB,0x00,0xBA,0x90,0xFA,0x96,
		0xE5,0xC2,0x18,0x63,0x57,0x90,0x69,0x42,0xB5,0x92,
		0x2F,0xC6,0x98,0xFA,0xCB,0x50,0x99,0x63,0x00,0x73,
		0x31,0xE6,0x6E,0x64,0x50,0x3F,0x18,0x92,0xA7,0x7C,
		0x0F,0xA3,0x36,0x16,0xE1,0xA2,0x50,0x3C,0x8D,0x92,
		0x86,0x14,0xDF,0xA6,0xE4,0xDF,0xC0,0x3D,0x37,0x4C, /* 700 */

		0xF2,0x66,0xA2,0x7C,0xD6,0x73,0x77,0xDE,0xBB,0x55,
		0xF1,0x39,0xC7,0x04,0x98,0x6A,0xE2,0x3C,0xF7,0xE8,
		0xE0,0xFD,0x49,0x77,0x14,0xAC,0x57,0x36,0xB7,0x43,
		0xA1,0x4A,0x9B,0x52,0xCA,0xFC,0x42,0x8E,0x75,0x57,
		0x82,0x04,0x57,0xB1,0xAC,0x5C,0x16,0xAF,0x6D,0xB6,
		0xF4,0x73,0xEA,0xB1,0x51,0xB1,0x99,0xC0,0xCC,0x01,
		0x28,0x9E,0x74,0x7A,0xFC,0x83,0x26,0x7E),
	.shared = chunk_from_chars(
		0xCE,0x67,0x86,0xC3,0x1A,0x7D,0xAF,0x57,0xE0,0xFC,
		0x5D,0x38,0x57,0x68,0x9F,0xA4,0x72,0x40,0xEC,0xBD,
		0xBB,0xCB,0x3A,0xFA,0x31,0x5A,0x85,0xBD,0x8D,0xB6,
		0x82,0xC8),
};

ke_test_vector_t mlkem512_2 = {
	.method = ML_KEM_512,
	.seed = chunk_from_chars(
		0x64,0x33,0x5B,0xF2,0x9E,0x5D,0xE6,0x28,0x42,0xC9,
		0x41,0x76,0x6B,0xA1,0x29,0xB0,0x64,0x3B,0x5E,0x71,
		0x21,0xCA,0x26,0xCF,0xC1,0x90,0xEC,0x7D,0xC3,0x54,
		0x38,0x30,0x55,0x7F,0xDD,0x5C,0x03,0xCF,0x12,0x3A,
		0x45,0x6D,0x48,0xEF,0xEA,0x43,0xC8,0x68),
	.pub_i = chunk_from_chars(
		0x24,0x50,0x31,0x05,0xA7,0x3D,0xB3,0x0A,0x55,0x5A,
		0x94,0x7E,0x61,0x22,0x74,0x9B,0xB6,0x7F,0x9D,0x30,
		0x68,0x6D,0x9B,0xBD,0xE6,0xB7,0x20,0xA6,0x10,0x9D,
		0x5F,0xD3,0xB4,0xC7,0xE6,0xBC,0x1F,0x9C,0x8C,0x16,
		0xCA,0x32,0x99,0xC7,0x7B,0xDE,0xAB,0x47,0xAE,0xC2,
		0x75,0xE2,0x62,0x2E,0x20,0x10,0x4C,0x59,0xB3,0x98,
		0xF9,0x3C,0x7F,0x70,0xA0,0x47,0xC3,0x83,0x29,0x82,
		0xCC,0x78,0x73,0x16,0x3B,0xA1,0xD4,0x50,0x01,0xB8,
		0x08,0x91,0x49,0xCD,0x75,0xCB,0x9A,0xE1,0x69,0x42,
		0x07,0xC2,0x4C,0xE8,0x46,0xB2,0x1D,0x87,0xB2,0x3E, /* 100 */

		0x61,0x71,0x0E,0x72,0x7B,0x7F,0x03,0x82,0x6A,0x76,
		0x21,0x09,0x48,0x27,0x9C,0x12,0x5F,0x65,0xD4,0xA8,
		0xD4,0x09,0x6F,0x3A,0xC3,0x0F,0x68,0x3C,0x6E,0x3D,
		0xD3,0x20,0x12,0x24,0xA4,0x2D,0x67,0xB7,0xB1,0x50,
		0xAC,0xFD,0x96,0xBA,0xB8,0x99,0x2A,0xDA,0x72,0xBF,
		0xF4,0x86,0xC0,0x0D,0x78,0x53,0x6B,0x68,0xAE,0x35,
		0x9A,0x09,0xFB,0xA5,0xCB,0x47,0x83,0x89,0xB6,0x62,
		0x1C,0xFF,0x13,0x47,0x7E,0x6B,0x4F,0xA2,0x1B,0x1B,
		0x47,0xB8,0x89,0xD5,0x50,0xA4,0xC7,0x35,0x35,0xA3,
		0x13,0x6B,0x16,0x50,0x09,0x49,0x3A,0x34,0x10,0xEA, /* 200 */

		0xC0,0x77,0x95,0x4E,0x0F,0x12,0x6A,0xFA,0x5C,0x00,
		0xE4,0xBC,0x05,0xFA,0xFA,0xC6,0x3E,0xF4,0x43,0xFB,
		0x74,0x9E,0x36,0x56,0x03,0x5E,0xE4,0x9F,0x33,0xBB,
		0x21,0xF3,0xF5,0xC2,0x4A,0x2B,0x96,0x44,0x16,0x44,
		0x0F,0xFA,0x0B,0x7F,0x42,0x90,0xCA,0x49,0x19,0x4D,
		0xD9,0x11,0xCB,0x51,0x73,0xAF,0x9C,0x87,0xBD,0x55,
		0x77,0x68,0x57,0x8C,0x5A,0xB9,0xB3,0x44,0x08,0x52,
		0xDC,0x48,0xC9,0x83,0x54,0x13,0x67,0xA7,0x2D,0x17,
		0x3B,0xBA,0x67,0x15,0x69,0x08,0x4C,0x36,0xF6,0xE6,
		0xC0,0x21,0xB5,0x4E,0x95,0x19,0x67,0xED,0x28,0xB2, /* 300 */

		0x3B,0xDC,0x9F,0x4A,0xD9,0x23,0x6F,0x9A,0xAF,0x14,
		0xA4,0x7C,0x27,0x37,0xAE,0x4A,0x12,0x3E,0x37,0xA3,
		0x13,0x7E,0xA0,0x33,0x9E,0x21,0xCA,0x83,0x48,0x55,
		0x54,0x1A,0x4E,0x05,0xB5,0x22,0x9E,0x45,0x88,0xC9,
		0x83,0xA4,0xD9,0x6C,0x91,0xC8,0x3B,0xB7,0x40,0xA6,
		0x6C,0x2C,0xA4,0x85,0x3E,0x84,0x84,0x8C,0xD6,0x13,
		0xCD,0x73,0xA0,0x1A,0x34,0xA8,0x0D,0x7A,0xA1,0xCB,
		0xBA,0x41,0xA2,0x6A,0x71,0x4B,0xC4,0xB5,0x34,0x0B,
		0xAF,0x82,0xB1,0x7D,0xFB,0xEC,0x85,0x93,0xEA,0x5F,
		0xF9,0x7A,0x73,0x8C,0x52,0x2B,0xD3,0x59,0xAC,0x65, /* 400 */

		0x45,0xC1,0xCC,0xC4,0xA6,0x93,0x37,0x26,0xAB,0x02,
		0x24,0xA9,0x5B,0x73,0x54,0xBA,0x1F,0x4D,0x52,0xA0,
		0xBB,0x46,0xC8,0xF3,0xD1,0x1A,0x75,0xC6,0x7E,0x2F,
		0xCB,0x62,0x6E,0x2B,0x3B,0x61,0xE3,0x26,0xB9,0xF7,
		0x1D,0xD2,0x14,0x79,0xAF,0x3B,0x68,0x7A,0x31,0xCD,
		0xB7,0xF4,0xBB,0x9E,0xA8,0x80,0x34,0x29,0xBF,0x57,
		0x13,0xC5,0x5B,0xE6,0x28,0x5E,0x86,0x3C,0xA8,0x11,
		0x3B,0x3C,0xE4,0xC2,0x58,0xB0,0x36,0x85,0x38,0x28,
		0xB6,0x58,0x8A,0xC4,0xD0,0xC7,0xBC,0x55,0x7F,0x07,
		0xB4,0x63,0x7D,0xE3,0x75,0x4D,0x70,0x16,0x92,0xC2, /* 500 */

		0x80,0xE7,0xBB,0x8E,0xBE,0x39,0x31,0x04,0x0B,0x11,
		0x4C,0x1C,0x12,0xDF,0x0A,0xB2,0x98,0x46,0x4D,0x86,
		0xF9,0x88,0x9D,0xD6,0x75,0xED,0xF8,0x9D,0xED,0xC7,
		0xA1,0xFB,0x15,0x3E,0xC3,0x55,0x2C,0xF9,0xB5,0x9E,
		0x39,0x71,0x6A,0x4F,0x57,0x9A,0x9A,0x2C,0xCC,0x5B,
		0x6C,0x16,0x42,0x37,0xBD,0xBE,0xA1,0x74,0x5B,0x97,
		0x54,0x74,0x73,0xA8,0xBB,0x26,0x25,0x59,0xD1,0x7D,
		0xEC,0x69,0x94,0xE0,0xA5,0x28,0x0C,0xD9,0xB3,0x45,
		0xBA,0x59,0x7A,0x5C,0x38,0xC3,0xE7,0x6D,0x49,0x72,
		0x80,0x89,0x07,0xC2,0x0D,0x0B,0x04,0x40,0x17,0x0F, /* 600 */

		0x7C,0x34,0x2B,0xCB,0x93,0x48,0xA0,0x85,0x0F,0xEC,
		0xE6,0x2A,0xD0,0xE2,0x5A,0x89,0x46,0x5E,0x45,0xD8,
		0x89,0x6B,0x42,0xCC,0x12,0x53,0x4D,0xE3,0x9A,0x40,
		0xF4,0x14,0x0B,0xED,0xC6,0x7C,0xAF,0x03,0x88,0x1D,
		0x03,0x38,0xC0,0xD4,0x48,0x01,0xE4,0x8A,0x48,0xB1,
		0x91,0xD4,0x4C,0x6E,0x55,0x16,0x07,0x7A,0xF0,0x90,
		0x5D,0xF2,0xA6,0xE4,0x35,0x3A,0x39,0xBC,0x1E,0xFB,
		0x36,0xB3,0x7E,0x77,0xAD,0xC3,0x90,0x29,0xD1,0xD7,
		0x02,0x71,0xF2,0x98,0x56,0x68,0x7B,0xC5,0x42,0x8A,
		0xB4,0x29,0x16,0xE5,0xA3,0x90,0xEE,0x19,0x8A,0xC5, /* 700 */

		0x5C,0x3E,0xFB,0x8B,0x7E,0x1B,0x7A,0x15,0x41,0xA1,
		0x2B,0x91,0xF7,0x99,0x92,0x3B,0x57,0xA3,0x61,0x3C,
		0x6E,0x75,0x59,0x3F,0x33,0xA8,0x56,0xC5,0x81,0x4A,
		0x20,0x9D,0xAE,0x26,0x88,0xC2,0xDC,0x2B,0xF1,0xA1,
		0x15,0x49,0x1A,0x94,0xBF,0x9A,0x2E,0x81,0xB0,0x27,
		0x6E,0xF5,0x17,0x6B,0x29,0x81,0x27,0x86,0x8C,0xFF,
		0x46,0x34,0xDF,0x11,0x97,0xF5,0x46,0x3B,0x23,0xF1,
		0x26,0xF1,0xAE,0x62,0x5B,0xD2,0xBE,0x8C,0xA1,0x22,
		0x94,0x57,0xF9,0x6A,0xB5,0xC2,0x5D,0xBB,0x61,0x64,
		0xA8,0x0B,0xAE,0x45,0xC7,0x86,0x8A,0xC1,0x74,0x85),
	.pub_r = chunk_from_chars(
		0x02,0x56,0xF7,0x21,0xFD,0x95,0x63,0x98,0xB7,0x3A,
		0xFC,0xD3,0xA4,0xFA,0x22,0xA2,0x59,0xAF,0xAA,0x00,
		0x67,0x54,0x53,0x86,0xD2,0xAF,0xC2,0x17,0x6E,0xF3,
		0x64,0x93,0xD6,0x42,0xDC,0x84,0x02,0xAF,0x4A,0xBF,
		0xA1,0x02,0x49,0x3B,0xCB,0x35,0x7C,0x4A,0xBA,0xAE,
		0xE1,0x38,0xFA,0xFD,0x03,0x5E,0x5A,0x12,0xF9,0xDE,
		0xC9,0x25,0x5E,0x17,0x1A,0xD0,0x8F,0x05,0x1A,0x8D,
		0xFA,0x87,0x1D,0x90,0x81,0x66,0x35,0xDE,0xE9,0xCD,
		0xCC,0xB1,0xB5,0xF7,0x82,0x4B,0x34,0x6B,0x62,0x76,
		0x1A,0x52,0x37,0x34,0x1E,0x3A,0xBF,0xD9,0xAF,0xA6, /* 100 */

		0x29,0xE3,0x4D,0xA3,0x68,0xA7,0xB8,0xD2,0x68,0xBE,
		0x24,0x8A,0xB3,0x08,0x52,0x09,0xF2,0x2F,0x00,0xCA,
		0x67,0xC5,0x9B,0x9B,0xAA,0x4B,0xBC,0x4B,0xE7,0xC6,
		0x77,0x06,0x98,0x71,0x5D,0xB4,0x53,0xB3,0x10,0x5D,
		0x77,0x66,0x9E,0x02,0x83,0xFA,0x87,0xAD,0x8B,0xAA,
		0x17,0x12,0x11,0x07,0xD6,0x18,0x74,0x3F,0xB2,0x0C,
		0xCC,0xA9,0x9D,0x2A,0xE1,0x62,0x5C,0xCB,0x54,0x5F,
		0x6D,0x79,0x78,0xCF,0xEC,0x0A,0xCE,0x6B,0x3D,0xC8,
		0x09,0xD4,0x5D,0x0E,0x6D,0x67,0x19,0x08,0x24,0x4C,
		0x6B,0x50,0xB9,0x44,0xA1,0x13,0x3B,0x4F,0x57,0xD3, /* 200 */

		0x58,0x10,0xBB,0x85,0xA6,0x72,0x3C,0x61,0x3E,0x04,
		0x11,0x33,0x40,0x5E,0x50,0x3E,0xFC,0x79,0x2B,0x6C,
		0xA1,0xDC,0x30,0x4E,0x44,0x6C,0x07,0x3D,0x02,0x56,
		0x7C,0xDC,0xDF,0xD2,0xB9,0xBA,0xE6,0x42,0xDC,0x10,
		0xE7,0x99,0x2E,0x16,0x60,0xD0,0xCB,0xBE,0x59,0x2C,
		0xE3,0x36,0x1D,0x53,0x30,0x80,0xF3,0x3D,0xD5,0x9A,
		0x46,0x98,0x65,0x3F,0xD4,0x1F,0xA2,0xE5,0x03,0xA7,
		0x9C,0xA4,0xDA,0xB4,0x6A,0x5B,0x0B,0x22,0x87,0x9D,
		0x16,0x2B,0x99,0x48,0x2B,0x98,0x2D,0xA3,0x03,0x22,
		0xB7,0x8B,0xAB,0x3D,0xF0,0xFF,0xC3,0x0C,0xF3,0x42, /* 300 */

		0x97,0x22,0x7B,0x6A,0x8F,0x03,0x5F,0x5A,0x79,0x44,
		0xAD,0xAC,0x6D,0xB3,0x4B,0x43,0xF5,0x15,0x0B,0x55,
		0xFF,0xA3,0x4E,0x25,0xB8,0x0D,0xAA,0xB6,0x55,0x76,
		0x5D,0x3A,0x1B,0x1A,0x2E,0x74,0xA3,0xA7,0xF1,0xFB,
		0x9D,0x18,0x0F,0x52,0x0C,0xDD,0x7B,0xE8,0x00,0xEF,
		0x1B,0x17,0xC1,0x0C,0x98,0xE2,0x7B,0xA2,0x7E,0xED,
		0x7B,0xD1,0xA8,0x9D,0xFB,0x76,0xE0,0xC4,0xC4,0x4F,
		0x8F,0xCD,0x4C,0x94,0xF6,0x7C,0x78,0x5E,0xF1,0x30,
		0x99,0x6E,0xA5,0x05,0xF0,0x9F,0x7E,0x38,0x73,0xF1,
		0xAC,0x23,0x4D,0x17,0x89,0xDA,0x53,0x4F,0x3D,0x2A, /* 400 */

		0xF0,0xF7,0x3B,0xC4,0xF5,0x68,0x4B,0x33,0x25,0xC9,
		0x44,0x9F,0x0D,0x67,0x8B,0x56,0x88,0x6D,0x26,0xC5,
		0xBA,0x20,0xEC,0x32,0x4A,0x69,0x04,0x5D,0x37,0xE1,
		0x19,0xAD,0x80,0xD8,0x7A,0x59,0xAD,0xFA,0x29,0x87,
		0xA5,0x52,0x01,0xF1,0xD2,0xF9,0x86,0x74,0x17,0x2F,
		0xE2,0x9B,0xD3,0xA5,0xD9,0xDB,0xC5,0x64,0x18,0x95,
		0xAD,0x60,0x52,0x6D,0xCD,0xE6,0x0B,0x2F,0x27,0xEC,
		0x89,0x68,0x6C,0x72,0x2A,0xFC,0x57,0x73,0xBA,0x47,
		0xBE,0x2B,0xB4,0x32,0x1B,0xEC,0x20,0xD2,0x13,0x57,
		0xC5,0x02,0x65,0x04,0x57,0x0D,0x01,0x43,0x20,0x4C, /* 500 */

		0x0E,0x44,0xB3,0xF4,0x1E,0xF5,0x1D,0xCD,0x59,0xBC,
		0x1B,0xBB,0xA0,0x00,0x65,0x37,0x2B,0x09,0x85,0x0E,
		0x78,0x1D,0x20,0x72,0x5C,0xA6,0x0C,0x45,0xBD,0x98,
		0x81,0x71,0x7E,0xE8,0x9B,0xBE,0x4A,0xD0,0x1A,0x5B,
		0xB1,0x0F,0x1D,0x8A,0x11,0x63,0x05,0xCE,0xB2,0x54,
		0xC5,0x63,0x6A,0x84,0xAD,0xE3,0x17,0x53,0x22,0x2B,
		0xC8,0xFA,0xEF,0x34,0x1E,0xEE,0x58,0x85,0xD1,0xD3,
		0x71,0xE7,0x4E,0x20,0xCF,0x77,0x59,0xB5,0xB4,0x13,
		0x0A,0x6A,0xAC,0xE1,0xA0,0xCF,0x94,0xF2,0xE3,0x47,
		0xD0,0xFC,0xAA,0x80,0x91,0xF4,0x87,0x0D,0xA8,0x62, /* 600 */

		0x53,0x0A,0xF1,0xF4,0x77,0x3D,0x52,0x5A,0x0C,0x5A,
		0x55,0x45,0x59,0x16,0x75,0x3E,0x36,0x73,0x0C,0x4C,
		0xBD,0xF7,0x8F,0xCF,0x01,0xA5,0x5C,0xF6,0x08,0x51,
		0x77,0x22,0xC1,0x33,0x7C,0x88,0xBB,0xA0,0xD3,0xB3,
		0x37,0x5E,0x5B,0xCA,0x24,0xC6,0x7E,0x7D,0xC1,0x48,
		0xCF,0x6D,0x05,0x76,0x19,0x0E,0x8E,0x2A,0xBC,0xD7,
		0x93,0x9F,0xFC,0xE6,0xF6,0xA7,0x16,0x7F,0x04,0x0F,
		0xA6,0x08,0xC4,0x6E,0xE9,0xBA,0x7E,0x94,0x1E,0x2E,
		0x8E,0x09,0x32,0xC0,0x13,0xD0,0xC3,0xCA,0x44,0x3F,
		0x76,0x89,0xBB,0x70,0x02,0x72,0x5E,0x77,0x02,0xC0, /* 700 */

		0x98,0xCF,0x19,0x94,0xEB,0x39,0x54,0x24,0x08,0xB2,
		0x24,0x72,0x3E,0x72,0x12,0xCB,0x20,0x13,0x69,0x2F,
		0x1A,0x71,0x10,0x14,0x11,0xBC,0x79,0xEA,0x0F,0x03,
		0xAE,0x83,0xD4,0xF1,0xF3,0xE9,0xEB,0x42,0x79,0xE7,
		0x2F,0x1F,0x4A,0x65,0x9F,0xD3,0x75,0xE7,0x90,0x06,
		0x6C,0x57,0x57,0xF0,0x48,0xED,0x47,0xCF,0x85,0x1D,
		0x52,0x47,0xC7,0xAA,0xB6,0x5D,0x20,0x64),
	.shared = chunk_from_chars(
		0xD0,0xDD,0x93,0x75,0x23,0xA0,0x36,0x99,0x69,0x6F,
		0xAF,0xF1,0x64,0x1D,0xF3,0x92,0x5B,0x4F,0xF9,0x54,
		0x45,0x48,0xD3,0xD2,0xD8,0xF6,0x49,0x46,0x63,0x59,
		0x49,0xD2),
};

ke_test_vector_t mlkem512_3 = {
	.method = ML_KEM_512,
	.seed = chunk_from_chars(
		0x22,0x5D,0x5C,0xE2,0xCE,0xAC,0x61,0x93,0x0A,0x07,
		0x50,0x3F,0xB5,0x9F,0x7C,0x2F,0x93,0x6A,0x3E,0x07,
		0x54,0x81,0xDA,0x3C,0xA2,0x99,0xA8,0x0F,0x8C,0x5D,
		0xF9,0x22,0x3A,0x07,0x3E,0x7B,0x90,0xE0,0x2E,0xBF,
		0x98,0xCA,0x22,0x27,0xEB,0xA3,0x8C,0x1A),
	.pub_i = chunk_from_chars(
		0xBF,0xF4,0x84,0x45,0x1C,0x17,0x3F,0x41,0x41,0x21,
		0xE1,0x53,0x63,0x27,0x15,0xCB,0xD1,0x85,0x60,0x8C,
		0x4B,0xA0,0xE3,0x59,0x23,0xAC,0x85,0xA3,0x11,0x32,
		0xD4,0xD2,0x8B,0xA8,0x45,0x2B,0xCD,0xB7,0x72,0xB1,
		0xD4,0x14,0x30,0x30,0x68,0xC6,0xA4,0xA2,0xD7,0xBA,
		0x83,0xF7,0x2B,0xC0,0x0C,0xB4,0xC9,0x9A,0x9B,0x6B,
		0x26,0xF4,0xBE,0x08,0xC3,0xAD,0xD6,0x12,0xCA,0x7B,
		0x6A,0xA4,0x22,0xF3,0x2F,0x59,0x91,0xBE,0x0B,0x4C,
		0xC5,0xDE,0x56,0x44,0xAB,0x17,0xB2,0x50,0x44,0x42,
		0x28,0xE6,0x2F,0x91,0x82,0x28,0x99,0xC6,0x94,0xE1, /* 100 */

		0x13,0x3C,0xA5,0xDB,0x14,0xD8,0x84,0xA4,0x45,0x26,
		0x14,0x69,0x05,0x56,0x59,0xEC,0xAC,0x02,0xA0,0x53,
		0xB9,0x93,0x1F,0x16,0x00,0x52,0xCA,0x58,0xA2,0x45,
		0x92,0x55,0x68,0x93,0x55,0x5F,0x8B,0x13,0xAB,0x08,
		0xB0,0xE9,0x79,0x0C,0xE1,0xB4,0x2D,0xA5,0x47,0xA7,
		0xAE,0x91,0xCD,0x77,0xF3,0xC6,0xDE,0xDC,0xCC,0x64,
		0x58,0x12,0xA9,0xD2,0xCF,0x4F,0x74,0x9F,0x7D,0xC6,
		0x1A,0x58,0xA6,0x70,0x12,0xDC,0x51,0x77,0x62,0xBE,
		0xA3,0x28,0x20,0x68,0x50,0x46,0x98,0x43,0x78,0x40,
		0xCC,0x24,0xBB,0x65,0x94,0x37,0x21,0x4B,0x79,0x93, /* 200 */

		0xC0,0xB6,0x24,0x85,0x14,0xE3,0x2F,0x07,0x26,0x53,
		0x77,0xB0,0x19,0x1F,0x75,0xAB,0x52,0x38,0xAC,0x85,
		0xB1,0x81,0xAF,0x15,0x3B,0xAB,0x90,0x74,0x24,0x61,
		0xCA,0xA8,0x90,0x1D,0x8A,0x34,0x89,0xC5,0x04,0x50,
		0x24,0x78,0xC7,0x54,0x32,0x56,0x44,0x68,0xAE,0x2C,
		0xC7,0x19,0xED,0x23,0xAE,0xEC,0x64,0x60,0xCB,0x79,
		0x14,0x71,0xF7,0x02,0xC2,0x09,0x0E,0x21,0xB2,0x73,
		0xC6,0x94,0xAE,0xC9,0x36,0x9E,0xE7,0x57,0x24,0xC9,
		0x99,0xCF,0x7C,0x95,0xCD,0xB8,0x63,0x62,0x5D,0x83,
		0x07,0x6B,0x7B,0x03,0xDE,0xAA,0x22,0x1D,0xEA,0x89, /* 300 */

		0x91,0x94,0x2F,0x06,0x88,0x87,0x73,0x0B,0x65,0x27,
		0xD8,0xCF,0xFA,0x7A,0x69,0xA4,0x52,0x1E,0xAB,0x81,
		0x8D,0x32,0x06,0x5A,0x6F,0xE3,0x61,0x60,0xAA,0x76,
		0x7A,0x6C,0xBC,0x93,0x01,0x19,0xD3,0x8B,0x49,0x60,
		0x26,0x09,0x41,0x92,0x72,0xE1,0xEB,0x62,0x6F,0x38,
		0x02,0xEB,0x15,0xBD,0x31,0x82,0x08,0xCA,0x82,0x73,
		0x17,0xC6,0x4A,0x58,0xD2,0x9B,0x43,0x06,0x56,0xF3,
		0xF9,0x85,0x25,0x46,0x14,0x1F,0xE9,0x8E,0xC2,0xD3,
		0x4E,0x63,0x6B,0xB1,0xA3,0xE8,0x13,0x01,0x11,0xC9,
		0xBC,0x59,0x71,0x9D,0xF3,0x61,0x27,0x99,0x19,0x95, /* 400 */

		0x06,0x18,0xD1,0xD0,0x43,0x42,0x27,0x23,0x3F,0x87,
		0x7B,0x62,0x47,0x3B,0x55,0x88,0x73,0xBF,0x15,0x49,
		0x6F,0xD1,0x76,0x9C,0xC5,0x9D,0xEB,0x7B,0x4E,0x1B,
		0x26,0x8A,0x7F,0x6A,0x19,0x2A,0xDC,0x2C,0x7E,0xAB,
		0xB0,0x01,0x60,0x11,0x2B,0x7C,0x8B,0x3A,0x67,0x9B,
		0x43,0x2B,0x95,0xBF,0x34,0x68,0x56,0x17,0xA4,0xC7,
		0x63,0x13,0xFA,0x25,0x8D,0x85,0xBA,0x5D,0xDF,0x63,
		0x01,0xEB,0x39,0x61,0x4B,0x6A,0x82,0xE7,0x07,0x50,
		0x96,0x05,0xA9,0x0C,0x40,0x8A,0xA7,0xCA,0xA8,0xED,
		0x3C,0x59,0xB0,0xA5,0x10,0x2C,0x78,0x9A,0xFE,0x42, /* 500 */

		0x13,0x46,0xD9,0x19,0x87,0x3B,0x9E,0xF5,0x02,0x9C,
		0xCF,0xD4,0x68,0x76,0xB9,0x87,0x15,0xF7,0xB1,0x2E,
		0x5C,0xA8,0x2F,0x50,0x2E,0x50,0xC2,0x88,0xA8,0x24,
		0x78,0x23,0x58,0x5E,0x52,0x6B,0xA2,0x91,0x32,0x6E,
		0x02,0xCB,0xB0,0x3D,0x4C,0x76,0x8C,0x76,0x58,0x5C,
		0xB6,0xA0,0xA1,0xE5,0x05,0x08,0x28,0xAA,0xA9,0x45,
		0x7B,0x69,0x1C,0x58,0x29,0x05,0x34,0x8B,0x11,0x59,
		0x31,0x39,0x00,0x8A,0xB9,0x2B,0xD5,0x44,0x5D,0x0A,
		0x63,0xC7,0xB7,0x56,0xC5,0xE2,0xCA,0x32,0x8D,0x1C,
		0x5F,0x68,0x32,0xB5,0x62,0x45,0xCC,0x8E,0xCC,0x2C, /* 600 */

		0x7D,0x61,0x64,0xB2,0xB5,0xA8,0x74,0xA0,0x1F,0xE2,
		0x7A,0x1E,0x1B,0x56,0x40,0x80,0x9A,0xBB,0xED,0xD0,
		0x42,0xDC,0x18,0x48,0x9A,0x39,0x1C,0x75,0xE8,0x94,
		0x07,0x81,0x7E,0x69,0xD3,0xA4,0x9C,0xFB,0x96,0x72,
		0x9C,0x7A,0xF1,0xB4,0x8B,0x9A,0x40,0x89,0xA7,0x64,
		0x71,0xB2,0x59,0x74,0x79,0xF4,0xB5,0x8D,0xC7,0x7C,
		0x03,0x40,0xBA,0xF6,0xE1,0x03,0xB7,0xB6,0x1A,0xB1,
		0xF0,0x82,0xF6,0x35,0x79,0x5C,0xE6,0x6D,0x0C,0x3B,
		0x45,0x01,0x11,0x45,0x82,0xDA,0x86,0x63,0xD5,0x78,
		0x3C,0x98,0x1C,0x26,0x25,0x6B,0xFC,0x05,0x84,0x33, /* 700 */

		0x76,0x7D,0x10,0x73,0xC7,0x76,0x94,0xCB,0x47,0x2B,
		0x21,0xBD,0x05,0x22,0x55,0x18,0x9F,0x45,0xCB,0x11,
		0x00,0x99,0x4C,0xE7,0xC9,0x70,0x52,0x86,0x46,0xEF,
		0x30,0x7C,0xDD,0x43,0x76,0x99,0x17,0xA7,0xFF,0xE4,
		0x29,0xBA,0x65,0xA9,0xE6,0x43,0x79,0x6D,0x7B,0x53,
		0x95,0xCA,0x9C,0x5D,0xE7,0x1E,0xA5,0x23,0x9D,0x8A,
		0x0C,0xB5,0x00,0x2A,0xAC,0x9D,0xFA,0x7B,0x70,0xAF,
		0xA1,0x1E,0x30,0x61,0x3F,0xBB,0x4E,0xB8,0x14,0xA6,
		0xD7,0x81,0xFD,0xAD,0x9E,0xB7,0x23,0x5E,0xB6,0x76,
		0x84,0x41,0x60,0x54,0xAB,0xD5,0xBA,0x87,0xE2,0x55),
	.pub_r = chunk_from_chars(
		0xCB,0x6B,0x5E,0x38,0x7B,0x7B,0x8F,0x7B,0x13,0xFE,
		0x6C,0xF9,0x85,0x6C,0x27,0xDE,0xC7,0x56,0x8C,0x6A,
		0xE0,0x76,0x2F,0x63,0x6F,0xA7,0x48,0xB1,0x91,0xAF,
		0x56,0xE9,0x2D,0x6A,0xA5,0x00,0x20,0x70,0x68,0xBC,
		0xAC,0xAF,0x70,0xFA,0x50,0xDD,0x6D,0xCF,0xC3,0xA6,
		0xDA,0x4B,0xDF,0x8C,0x96,0xC5,0xC7,0xDF,0x18,0x2B,
		0x1A,0x94,0x35,0xE1,0x71,0xEB,0x6B,0x11,0x89,0x41,
		0xDF,0xCF,0xFB,0x4C,0x10,0xD5,0x5A,0x2C,0xE2,0x00,
		0xC2,0x26,0x0B,0xAF,0x2C,0x4C,0x38,0x82,0x5F,0x0E,
		0xCF,0xE6,0x83,0x21,0xDC,0x7D,0xED,0xA6,0x5C,0x90, /* 100 */

		0x9A,0x17,0x89,0x18,0xE4,0x3F,0x60,0x08,0x80,0x2B,
		0x02,0xB6,0xAA,0xB7,0xEA,0x39,0x9E,0xCD,0x20,0x4B,
		0xDB,0x7E,0xAE,0x80,0x6E,0x4B,0x0E,0xDA,0xD9,0x20,
		0xB7,0xCB,0x43,0xF4,0x37,0x97,0x9F,0x03,0x3A,0xC0,
		0x1C,0x9E,0xD0,0x19,0x24,0xCC,0x11,0x8B,0xDF,0x37,
		0x0B,0x4F,0xAD,0x92,0xF5,0xBA,0x23,0xD9,0xE3,0x85,
		0xAB,0xF5,0x06,0x9E,0xCF,0x8D,0xCB,0x3B,0x4F,0x1D,
		0x88,0x92,0x84,0x91,0xD8,0xD8,0xFE,0xC4,0x13,0x13,
		0x29,0x4A,0x41,0xBB,0x68,0x50,0x34,0x37,0xCB,0x76,
		0x31,0x3C,0x5B,0xD5,0xD5,0x0B,0xFC,0x4B,0xB8,0x2C, /* 200 */

		0x8B,0x15,0x39,0x3F,0x58,0xC1,0xEF,0xC5,0x1A,0x60,
		0x67,0xE2,0xC6,0x82,0xCA,0x9B,0x11,0xA5,0xF1,0x1D,
		0xE9,0xCD,0xA0,0xFE,0x72,0xA2,0xEB,0x66,0x5E,0x95,
		0xE2,0x0D,0x1C,0x59,0x96,0x37,0x03,0x03,0xDF,0x8B,
		0xB9,0xED,0xE0,0xE2,0x99,0x60,0xDA,0x60,0x3D,0x85,
		0xD3,0x0A,0x54,0xA8,0xFE,0xC4,0xF1,0x6B,0xB9,0x83,
		0x82,0x57,0x16,0x5C,0x45,0xA9,0x1F,0x9F,0x31,0x3F,
		0xD9,0xDA,0x8E,0x5D,0x44,0x76,0x46,0x15,0x1E,0xAD,
		0x3B,0x10,0xC3,0x99,0xBE,0x14,0xB9,0x04,0x7D,0x51,
		0x8C,0x39,0x70,0x9D,0xDF,0x8D,0x8D,0x2B,0xA4,0x7F, /* 300 */

		0x1A,0x41,0x20,0x39,0x69,0x8E,0xEA,0x9F,0x77,0x4A,
		0x06,0x6C,0xB5,0x34,0x38,0xE3,0x5D,0xD8,0xBB,0xC1,
		0xA8,0x37,0x8F,0xA1,0x16,0x55,0x65,0x24,0xC0,0x89,
		0xA9,0x4E,0xC0,0x21,0x73,0x3B,0x2B,0x87,0x63,0x5F,
		0xBF,0xD7,0xD5,0x35,0x11,0xA9,0xAA,0x1D,0x66,0x5B,
		0x04,0x4B,0xFE,0x45,0x69,0xBB,0x47,0x58,0x4D,0x72,
		0x98,0x10,0x68,0x4B,0x3A,0x63,0x73,0x9E,0x3B,0xAE,
		0xF0,0xCD,0xAF,0x00,0x77,0x4A,0xBA,0xEB,0x90,0xB9,
		0xE7,0x8B,0xBF,0x2C,0xB5,0xF8,0x6D,0x25,0xA7,0xD4,
		0x48,0xD7,0xCD,0xC9,0xB8,0x74,0xAF,0x4D,0x83,0x66, /* 400 */

		0x4F,0x08,0x5D,0xB4,0xED,0x9C,0xCD,0x01,0xDF,0xE1,
		0xA5,0x5A,0x3A,0x47,0xBB,0xD7,0x44,0x0B,0xD4,0x3E,
		0x0F,0xA3,0x36,0xF6,0x91,0x9A,0xAB,0xD1,0xCA,0x0B,
		0x14,0xB5,0xB3,0x73,0x86,0x6A,0xB3,0x78,0x63,0xCE,
		0x20,0x05,0xB0,0x03,0x24,0x27,0xCF,0xF1,0xA4,0xDF,
		0x9A,0x79,0x5B,0x6A,0x4B,0xC6,0xE0,0xBE,0x3E,0x10,
		0xFC,0x0D,0xE5,0xD9,0x81,0x30,0xC2,0x7B,0xCF,0x6E,
		0xDA,0x26,0x82,0x24,0x15,0x8A,0x0B,0x82,0x69,0xE5,
		0x8B,0x5E,0x43,0x71,0x32,0x51,0xAD,0x3B,0x87,0xB2,
		0x3D,0x87,0xC1,0x35,0x05,0xCB,0xB6,0x5C,0x83,0xBC, /* 500 */

		0x00,0x94,0x20,0x48,0x9A,0x85,0x77,0x45,0x22,0x1F,
		0xEA,0x1E,0x1D,0x2C,0xFF,0x23,0x22,0x61,0xC6,0xC2,
		0x01,0xD8,0xB7,0xF2,0xAA,0x9F,0xEA,0x11,0x94,0xBC,
		0x80,0x6C,0xE4,0xE6,0x93,0xF6,0x71,0xD8,0x30,0x1C,
		0x7B,0xD6,0x21,0xAD,0xCD,0x15,0xC8,0xD3,0x24,0x78,
		0x96,0xBB,0x18,0x6B,0x1B,0x97,0x02,0x16,0x4F,0xAE,
		0xEE,0xD1,0x8E,0xCD,0xFA,0xE8,0x0F,0x7D,0xDB,0xB3,
		0xC5,0x18,0xEA,0x47,0x3A,0x79,0xAB,0x69,0xD9,0x0B,
		0x49,0xD9,0x1F,0x94,0x83,0x4D,0xE6,0xE5,0x55,0x9E,
		0xA0,0xAC,0x8E,0xDD,0x51,0xC7,0x47,0xA0,0x09,0x67, /* 600 */

		0x6A,0x6F,0x79,0xDF,0x4D,0x3F,0x9B,0xF2,0x74,0x94,
		0x8A,0xDF,0xAF,0x78,0x2D,0x26,0x17,0xF1,0xB5,0xF6,
		0x11,0x3D,0x61,0x7A,0xD7,0x91,0x94,0xB6,0x8E,0xEB,
		0x96,0x67,0x44,0x9C,0xB8,0x4B,0x89,0xB1,0x1E,0xCE,
		0xF2,0x61,0x83,0x9B,0x4C,0xF8,0xBC,0x66,0xFC,0xF8,
		0x9A,0xCC,0x36,0x43,0xF9,0x1C,0x6C,0x66,0xCA,0xBC,
		0x80,0x49,0x4E,0xE3,0x90,0xFE,0xC7,0x1F,0x81,0x9A,
		0x47,0xAD,0x10,0x20,0xD5,0xD7,0x72,0xFD,0x79,0xBD,
		0xBD,0x86,0x9C,0x5C,0x36,0x50,0xB9,0xAD,0xD4,0x7D,
		0x65,0xF2,0xF1,0xBA,0x03,0x5E,0x72,0x11,0xDA,0x8D, /* 700 */

		0x40,0x9A,0x84,0xC6,0xF6,0x9F,0x46,0x8F,0x27,0x7D,
		0x02,0x7D,0x84,0xFC,0xD0,0x3B,0x46,0xC0,0x92,0x88,
		0xFE,0x2C,0xE6,0xF9,0x57,0x30,0x6F,0x3A,0x96,0xDA,
		0xE3,0xA9,0xBE,0xE7,0xC3,0x41,0x09,0x90,0x5F,0x52,
		0x08,0x15,0xB0,0x4E,0x18,0xDF,0x98,0x9A,0xBC,0x30,
		0xA9,0x5B,0xE6,0x66,0xA4,0xDB,0xD4,0xC1,0xE7,0xEF,
		0x5E,0x0D,0xE4,0x95,0x86,0xBF,0x31,0x4B),
	.shared = chunk_from_chars(
		0x72,0xE4,0x22,0x3D,0x6D,0x11,0x06,0x11,0x02,0x2A,
		0x0F,0xDB,0x2C,0xBD,0x33,0x3D,0x11,0xA0,0xF3,0x81,
		0x29,0x3C,0xF4,0x5D,0xD0,0xC8,0x97,0x1E,0xCC,0xE8,
		0x30,0xFB),
};

/**
 * First four test vectors of the liboqs-0.11.0 ML-KEM-768 KAT
 */
ke_test_vector_t mlkem768_0 = {
	.method = ML_KEM_768,
	.seed = chunk_from_chars(
		0x06,0x15,0x50,0x23,0x4D,0x15,0x8C,0x5E,0xC9,0x55,
		0x95,0xFE,0x04,0xEF,0x7A,0x25,0x76,0x7F,0x2E,0x24,
		0xCC,0x2B,0xC4,0x79,0xD0,0x9D,0x86,0xDC,0x9A,0xBC,
		0xFD,0xE7,0x05,0x6A,0x8C,0x26,0x6F,0x9E,0xF9,0x7E,
		0xD0,0x85,0x41,0xDB,0xD2,0xE1,0xFF,0xA1),
	.pub_i = chunk_from_chars(
		0xA8,0xE6,0x51,0xA1,0xE6,0x85,0xF2,0x24,0x78,0xA8,
		0x95,0x4F,0x00,0x7B,0xC7,0x71,0x1B,0x93,0x07,0x72,
		0xC7,0x8F,0x09,0x2E,0x82,0x87,0x8E,0x3E,0x93,0x7F,
		0x36,0x79,0x67,0x53,0x29,0x13,0xA8,0xD5,0x3D,0xFD,
		0xF4,0xBF,0xB1,0xF8,0x84,0x67,0x46,0x59,0x67,0x05,
		0xCF,0x34,0x51,0x42,0xB9,0x72,0xA3,0xF1,0x63,0x25,
		0xC4,0x0C,0x29,0x52,0xA3,0x7B,0x25,0x89,0x7E,0x5E,
		0xF3,0x5F,0xBA,0xEB,0x73,0xA4,0xAC,0xBE,0xB6,0xA0,
		0xB8,0x99,0x42,0xCE,0xB1,0x95,0x53,0x1C,0xFC,0x0A,
		0x07,0x99,0x39,0x54,0x48,0x3E,0x6C,0xBC,0x87,0xC0, /* 100 */

		0x6A,0xA7,0x4F,0xF0,0xCA,0xC5,0x20,0x7E,0x53,0x5B,
		0x26,0x0A,0xA9,0x8D,0x11,0x98,0xC0,0x7D,0xA6,0x05,
		0xC4,0xD1,0x10,0x20,0xF6,0xC9,0xF7,0xBB,0x68,0xBB,
		0x34,0x56,0xC7,0x3A,0x01,0xB7,0x10,0xBC,0x99,0xD1,
		0x77,0x39,0xA5,0x17,0x16,0xAA,0x01,0x66,0x0C,0x8B,
		0x62,0x8B,0x2F,0x56,0x02,0xBA,0x65,0xF0,0x7E,0xA9,
		0x93,0x33,0x6E,0x89,0x6E,0x83,0xF2,0xC5,0x73,0x1B,
		0xBF,0x03,0x46,0x0C,0x5B,0x6C,0x8A,0xFE,0xCB,0x74,
		0x8E,0xE3,0x91,0xE9,0x89,0x34,0xA2,0xC5,0x7D,0x4D,
		0x06,0x9F,0x50,0xD8,0x8B,0x30,0xD6,0x96,0x6F,0x38, /* 200 */

		0xC3,0x7B,0xC6,0x49,0xB8,0x26,0x34,0xCE,0x77,0x22,
		0x64,0x5C,0xCD,0x62,0x50,0x63,0x36,0x46,0x46,0xD6,
		0xD6,0x99,0xDB,0x57,0xB4,0x5E,0xB6,0x74,0x65,0xE1,
		0x6D,0xE4,0xD4,0x06,0xA8,0x18,0xB9,0xEA,0xE1,0xCA,
		0x91,0x6A,0x25,0x94,0x48,0x97,0x08,0xA4,0x3C,0xEA,
		0x88,0xB0,0x2A,0x4C,0x03,0xD0,0x9B,0x44,0x81,0x5C,
		0x97,0x10,0x1C,0xAF,0x50,0x48,0xBB,0xCB,0x24,0x7A,
		0xE2,0x36,0x6C,0xDC,0x25,0x4B,0xA2,0x21,0x29,0xF4,
		0x5B,0x3B,0x0E,0xB3,0x99,0xCA,0x91,0xA3,0x03,0x40,
		0x28,0x30,0xEC,0x01,0xDB,0x7B,0x2C,0xA4,0x80,0xCF, /* 300 */

		0x35,0x04,0x09,0xB2,0x16,0x09,0x4B,0x7B,0x0C,0x3A,
		0xE3,0x3C,0xE1,0x0A,0x91,0x24,0xE8,0x96,0x51,0xAB,
		0x90,0x1E,0xA2,0x53,0xC8,0x41,0x5B,0xD7,0x82,0x5F,
		0x02,0xBB,0x22,0x93,0x69,0xAF,0x97,0x20,0x28,0xF2,
		0x28,0x75,0xEA,0x55,0xAF,0x16,0xD3,0xBC,0x69,0xF7,
		0x0C,0x2E,0xE8,0xB7,0x5F,0x28,0xB4,0x7D,0xD3,0x91,
		0xF9,0x89,0xAD,0xE3,0x14,0x72,0x9C,0x33,0x1F,0xA0,
		0x4C,0x19,0x17,0xB2,0x78,0xC3,0xEB,0x60,0x28,0x68,
		0x51,0x28,0x21,0xAD,0xC8,0x25,0xC6,0x45,0x77,0xCE,
		0x1E,0x63,0xB1,0xD9,0x64,0x4A,0x61,0x29,0x48,0xA3, /* 400 */

		0x48,0x3C,0x7F,0x1B,0x9A,0x25,0x80,0x00,0xE3,0x01,
		0x96,0x94,0x4A,0x40,0x36,0x27,0x60,0x9C,0x76,0xC7,
		0xEA,0x6B,0x5D,0xE0,0x17,0x64,0xD2,0x43,0x79,0x11,
		0x7B,0x9E,0xA2,0x98,0x48,0xDC,0x55,0x5C,0x45,0x4B,
		0xCE,0xAE,0x1B,0xA5,0xCC,0x72,0xC7,0x4A,0xB9,0x6B,
		0x9C,0x91,0xB9,0x10,0xD2,0x6B,0x88,0xB2,0x56,0x39,
		0xD4,0x77,0x8A,0xE2,0x6C,0x7C,0x61,0x51,0xA1,0x9C,
		0x6C,0xD7,0x93,0x84,0x54,0x37,0x24,0x65,0xE4,0xC5,
		0xEC,0x29,0x24,0x5A,0xCB,0x3D,0xB5,0x37,0x9D,0xE3,
		0xDA,0xBF,0xA6,0x29,0xA7,0xC0,0x4A,0x83,0x53,0xA8, /* 500 */

		0x53,0x0C,0x95,0xAC,0xB7,0x32,0xBB,0x4B,0xB8,0x19,
		0x32,0xBB,0x2C,0xA7,0xA8,0x48,0xCD,0x36,0x68,0x01,
		0x44,0x4A,0xBE,0x23,0xC8,0x3B,0x36,0x6A,0x87,0xD6,
		0xA3,0xCF,0x36,0x09,0x24,0xC0,0x02,0xBA,0xE9,0x0A,
		0xF6,0x5C,0x48,0x06,0x0B,0x37,0x52,0xF2,0xBA,0xDF,
		0x1A,0xB2,0x72,0x20,0x72,0x55,0x4A,0x50,0x59,0x75,
		0x35,0x94,0xE6,0xA7,0x02,0x76,0x1F,0xC9,0x76,0x84,
		0xC8,0xC4,0xA7,0x54,0x0A,0x6B,0x07,0xFB,0xC9,0xDE,
		0x87,0xC9,0x74,0xAA,0x88,0x09,0xD9,0x28,0xC7,0xF4,
		0xCB,0xBF,0x80,0x45,0xAE,0xA5,0xBC,0x66,0x78,0x25, /* 600 */

		0xFD,0x05,0xA5,0x21,0xF1,0xA4,0xBF,0x53,0x92,0x10,
		0xC7,0x11,0x3B,0xC3,0x7B,0x3E,0x58,0xB0,0xCB,0xFC,
		0x53,0xC8,0x41,0xCB,0xB0,0x37,0x1D,0xE2,0xE5,0x11,
		0xB9,0x89,0xCB,0x7C,0x70,0xC0,0x23,0x36,0x6D,0x78,
		0xF9,0xC3,0x7E,0xF0,0x47,0xF8,0x72,0x0B,0xE1,0xC7,
		0x59,0xA8,0xD9,0x6B,0x93,0xF6,0x5A,0x94,0x11,0x4F,
		0xFA,0xF6,0x0D,0x9A,0x81,0x79,0x5E,0x99,0x5C,0x71,
		0x15,0x2A,0x46,0x91,0xA5,0xA6,0x02,0xA9,0xE1,0xF3,
		0x59,0x9E,0x37,0xC7,0x68,0xC7,0xBC,0x10,0x89,0x94,
		0xC0,0x66,0x9F,0x3A,0xDC,0x95,0x7D,0x46,0xB4,0xB6, /* 700 */

		0x25,0x69,0x68,0xE2,0x90,0xD7,0x89,0x2E,0xA8,0x54,
		0x64,0xEE,0x7A,0x75,0x0F,0x39,0xC5,0xE3,0x15,0x2C,
		0x2D,0xFC,0x56,0xD8,0xB0,0xC9,0x24,0xBA,0x8A,0x95,
		0x9A,0x68,0x09,0x65,0x47,0xF6,0x64,0x23,0xC8,0x38,
		0x98,0x2A,0x57,0x94,0xB9,0xE1,0x53,0x37,0x71,0x33,
		0x1A,0x9A,0x65,0x6C,0x28,0x82,0x8B,0xEB,0x91,0x26,
		0xA6,0x0E,0x95,0xE8,0xC5,0xD9,0x06,0x83,0x2C,0x77,
		0x10,0x70,0x55,0x76,0xB1,0xFB,0x95,0x07,0x26,0x9D,
		0xDA,0xF8,0xC9,0x5C,0xE9,0x71,0x9B,0x2C,0xA8,0xDD,
		0x11,0x2B,0xE1,0x0B,0xCC,0x9F,0x4A,0x37,0xBD,0x1B, /* 800 */

		0x1E,0xEE,0xB3,0x3E,0xCD,0xA7,0x6A,0xE9,0xF6,0x9A,
		0x5D,0x4B,0x29,0x23,0xA8,0x69,0x57,0x67,0x1D,0x61,
		0x93,0x35,0xBE,0x1C,0x4C,0x2C,0x77,0xCE,0x87,0xC4,
		0x1F,0x98,0xA8,0xCC,0x46,0x64,0x60,0xFA,0x30,0x0A,
		0xAF,0x5B,0x30,0x1F,0x0A,0x1D,0x09,0xC8,0x8E,0x65,
		0xDA,0x4D,0x8E,0xE6,0x4F,0x68,0xC0,0x21,0x89,0xBB,
		0xB3,0x58,0x4B,0xAF,0xF7,0x16,0xC8,0x5D,0xB6,0x54,
		0x04,0x8A,0x00,0x43,0x33,0x48,0x93,0x93,0xA0,0x74,
		0x27,0xCD,0x3E,0x21,0x7E,0x6A,0x34,0x5F,0x6C,0x2C,
		0x2B,0x13,0xC2,0x7B,0x33,0x72,0x71,0xC0,0xB2,0x7B, /* 900 */

		0x2D,0xBA,0xA0,0x0D,0x23,0x76,0x00,0xB5,0xB5,0x94,
		0xE8,0xCF,0x2D,0xD6,0x25,0xEA,0x76,0xCF,0x0E,0xD8,
		0x99,0x12,0x2C,0x97,0x96,0xB4,0xB0,0x18,0x70,0x04,
		0x25,0x80,0x49,0xA4,0x77,0xCD,0x11,0xD6,0x8C,0x49,
		0xB9,0xA0,0xE7,0xB0,0x0B,0xCE,0x8C,0xAC,0x78,0x64,
		0xCB,0xB3,0x75,0x14,0x00,0x84,0x74,0x4C,0x93,0x06,
		0x26,0x94,0xCA,0x79,0x5C,0x4F,0x40,0xE7,0xAC,0xC9,
		0xC5,0xA1,0x88,0x40,0x72,0xD8,0xC3,0x8D,0xAF,0xB5,
		0x01,0xEE,0x41,0x84,0xDD,0x5A,0x81,0x9E,0xC2,0x4E,
		0xC1,0x65,0x12,0x61,0xF9,0x62,0xB1,0x7A,0x72,0x15, /* 1000 */

		0xAA,0x4A,0x74,0x8C,0x15,0x83,0x6C,0x38,0x91,0x37,
		0x67,0x82,0x04,0x83,0x8D,0x71,0x95,0xA8,0x5B,0x4F,
		0x98,0xA1,0xB5,0x74,0xC4,0xCD,0x79,0x09,0xCD,0x1F,
		0x83,0x3E,0xFF,0xD1,0x48,0x55,0x43,0x22,0x9D,0x37,
		0x48,0xD9,0xB5,0xCD,0x6C,0x17,0xB9,0xB3,0xB8,0x4A,
		0xEF,0x8B,0xCE,0x13,0xE6,0x83,0x73,0x36,0x59,0xC7,
		0x95,0x42,0xD6,0x15,0x78,0x2A,0x71,0xCD,0xEE,0xE7,
		0x92,0xBA,0xB5,0x1B,0xDC,0x4B,0xBF,0xE8,0x30,0x8E,
		0x66,0x31,0x44,0xED,0xE8,0x49,0x18,0x30,0xAD,0x98,
		0xB4,0x63,0x4F,0x64,0xAB,0xA8,0xB9,0xC0,0x42,0x27, /* 1100 */

		0x26,0x53,0x92,0x0F,0x38,0x0C,0x1A,0x17,0xCA,0x87,
		0xCE,0xD7,0xAA,0xC4,0x1C,0x82,0x88,0x87,0x93,0x18,
		0x1A,0x6F,0x76,0xE1,0x97,0xB7,0xB9,0x0E,0xF9,0x09,
		0x43,0xBB,0x38,0x44,0x91,0x29,0x11,0xD8,0x55,0x1E,
		0x54,0x66,0xC5,0x76,0x7A,0xB0,0xBC,0x61,0xA1,0xA3,
		0xF7,0x36,0x16,0x2E,0xC0,0x98,0xA9,0x00,0xB1,0x2D,
		0xD8,0xFA,0xBB,0xFB,0x3F,0xE8,0xCB,0x1D,0xC4,0xE8,
		0x31,0x5F,0x2A,0xF0,0xD3,0x2F,0x00,0x17,0xAE,0x13,
		0x6E,0x19,0xF0,0x28),
	.pub_r = chunk_from_chars(
		0x3B,0x83,0x5A,0x5F,0xA1,0x45,0x38,0x7A,0x08,0x19,
		0xC4,0xDA,0xA1,0xE6,0x5F,0xBE,0x2B,0xA5,0x40,0x0A,
		0xFC,0xD6,0x40,0xBB,0xDD,0xBB,0xE3,0x58,0x5F,0x24,
		0xBE,0xDD,0x51,0x28,0x96,0x94,0xA4,0xFE,0x64,0x3C,
		0xD5,0xAF,0x9C,0x8E,0xB2,0x77,0xC3,0xF1,0x87,0x7A,
		0x34,0x7A,0x97,0xEB,0xEA,0x8A,0x03,0x79,0x71,0xC6,
		0xB3,0x79,0x93,0xE4,0x33,0xCF,0xAF,0x58,0x0E,0xBA,
		0x4B,0x7F,0xDA,0x99,0x0D,0x54,0xBF,0x4D,0x60,0xCA,
		0xF9,0xD1,0xCA,0xFC,0x47,0x7F,0xD9,0x56,0xF8,0xE6,
		0x07,0x0B,0x6A,0xEE,0xC6,0x77,0x6E,0xB8,0x14,0x83, /* 100 */

		0x54,0x07,0xB5,0xF7,0x05,0xDB,0x94,0x72,0x70,0x1D,
		0x16,0xE0,0x06,0x55,0x02,0x4A,0x30,0x9B,0x14,0xDD,
		0xBF,0x36,0xD2,0x22,0xBB,0x50,0x96,0x47,0xA5,0xA0,
		0x49,0xD5,0x81,0x6F,0x49,0xAD,0x9F,0x29,0x75,0xDD,
		0xB6,0x4C,0x2D,0xF0,0x5F,0xFE,0xB2,0x4C,0x6A,0x3F,
		0x24,0xA7,0x86,0xDB,0xF4,0xF6,0xD5,0x66,0x6F,0xC5,
		0x5F,0xB7,0x35,0x39,0x67,0x9D,0xC1,0x5B,0x72,0xFB,
		0x4F,0x6C,0xE3,0x8F,0xEB,0x28,0x1D,0x28,0xC9,0x08,
		0xD5,0x19,0x5D,0xB7,0x00,0x83,0x15,0x97,0x8E,0xF9,
		0xD2,0xC6,0x7D,0xC4,0xDB,0xCC,0x49,0x62,0x46,0x7A, /* 200 */

		0x2D,0x44,0xF7,0x23,0x5F,0xA5,0x4E,0xBD,0x88,0xBD,
		0xEC,0x32,0x40,0x8B,0x1F,0x7A,0xFF,0x1B,0x84,0x20,
		0x64,0x07,0x56,0x51,0xF0,0x3A,0x3A,0xFD,0x27,0x21,
		0xED,0x1F,0xE4,0xFF,0x1A,0x87,0x75,0xC6,0xB4,0xD9,
		0x57,0x64,0x55,0x54,0x12,0xCF,0xF2,0xF8,0xAA,0x44,
		0x04,0x90,0x0F,0x33,0x58,0x5F,0x0B,0xD1,0xB7,0x09,
		0x55,0xCF,0xF8,0x01,0x30,0xDC,0xC2,0x40,0x39,0x20,
		0xE9,0x74,0x4A,0x3D,0x0D,0xA9,0x14,0x40,0x55,0x61,
		0xEC,0xB2,0xBB,0x32,0x12,0x0B,0x7A,0xDB,0xD2,0xF4,
		0xD8,0xE9,0xA0,0x7B,0x46,0x30,0x48,0x0B,0x8D,0xF8, /* 300 */

		0xC0,0x68,0x93,0x4F,0xFD,0x9B,0xC9,0xB8,0x55,0xA8,
		0x88,0xEE,0xCA,0x09,0x0F,0x21,0x19,0x05,0xE0,0x74,
		0xA0,0x78,0xAB,0x68,0x91,0x7E,0x74,0x45,0xA6,0xC7,
		0xC7,0xE3,0x94,0x03,0x75,0x3C,0xE1,0x9B,0x66,0x14,
		0xB9,0xD2,0x22,0xAB,0x99,0xF2,0x63,0xA6,0x81,0xCE,
		0xC6,0xC0,0x37,0x58,0x7E,0xF0,0x51,0xF0,0xF7,0x29,
		0x4E,0x37,0x65,0x28,0xB3,0x17,0x89,0xA5,0x30,0x34,
		0x22,0x58,0x24,0x1C,0x99,0xAE,0x7D,0x38,0x4B,0xCD,
		0x61,0x01,0x2A,0x32,0xA9,0x77,0xC6,0x38,0xB0,0x9A,
		0x3B,0xC1,0x6A,0x33,0xAA,0x47,0xCF,0x2D,0x7F,0x12, /* 400 */

		0xD7,0x9D,0x8A,0xA5,0x0F,0x63,0xC8,0xC5,0x3C,0x43,
		0x98,0x00,0xB2,0xED,0x9B,0xBA,0x94,0x81,0xEB,0x18,
		0x1B,0x42,0x44,0xED,0x06,0x7D,0x62,0x69,0x5D,0x6A,
		0x99,0xDF,0xD7,0xBF,0x87,0x88,0xC1,0x59,0xCA,0xAF,
		0x94,0xE9,0xFD,0xA9,0x2A,0xC5,0xA9,0x3F,0x59,0xA0,
		0xDF,0x7C,0x0F,0x9B,0xBD,0x41,0x7C,0xB8,0xCF,0x45,
		0xD1,0x07,0x60,0x06,0xE0,0x8A,0x9E,0x58,0x5E,0xE4,
		0xD7,0x39,0x42,0x65,0x58,0x2A,0x87,0x64,0x1F,0x16,
		0x53,0xBE,0x9E,0xDF,0x19,0x44,0x01,0xE6,0xE4,0xEE,
		0x93,0xC4,0xAB,0x05,0x4A,0x1B,0x6E,0x81,0xE3,0xBF, /* 500 */

		0x01,0xFD,0x26,0xF2,0xE9,0xA6,0xDB,0x5B,0xF6,0xC0,
		0xDB,0xD2,0x1E,0x14,0xC2,0xE1,0xA5,0xA4,0xCF,0xF0,
		0xB2,0x67,0xED,0x95,0x42,0x7B,0x0B,0x04,0x9E,0xFF,
		0x7F,0xBC,0x09,0x3B,0x05,0x45,0x10,0x57,0x85,0x23,
		0xAC,0x7A,0x32,0xCC,0x1F,0x8E,0xDF,0xCF,0x07,0x8A,
		0x6C,0x71,0xE6,0xE6,0x78,0x8E,0xDF,0xDA,0x7D,0x7B,
		0xAD,0xD3,0x75,0xF7,0xD9,0x11,0xEF,0xAF,0xB9,0xCB,
		0x40,0x6E,0x96,0x8B,0xC5,0x98,0x94,0x18,0xFB,0x09,
		0x72,0x9E,0xD5,0x1C,0x92,0xC4,0xAE,0xAE,0x10,0x84,
		0x63,0x84,0xF4,0xA0,0x91,0xC4,0x05,0xAD,0x85,0x77, /* 600 */

		0x3F,0xE0,0xAD,0xE8,0x16,0xED,0xDF,0xD6,0x18,0xBA,
		0x0E,0xA5,0xDE,0xB7,0x3C,0xC4,0x35,0x92,0xE0,0x63,
		0x01,0x51,0x18,0x02,0x55,0x42,0x87,0x1E,0x7A,0x60,
		0xF8,0x44,0xA6,0xB2,0xC3,0xD6,0x30,0xF9,0xC6,0xF8,
		0x57,0x91,0xE8,0xD2,0xBD,0xF3,0x57,0x8F,0xF9,0x26,
		0x28,0xE8,0xAC,0xAF,0x02,0xB8,0x8D,0x79,0x79,0x7F,
		0xB1,0xAC,0x30,0x15,0x32,0x01,0xFC,0xAD,0x22,0x34,
		0xFB,0xD4,0xF2,0xFC,0x84,0xFA,0x7D,0x2A,0xB6,0xFB,
		0x2E,0x4D,0x9B,0x55,0xF1,0x1D,0xD9,0x1A,0x79,0x87,
		0x26,0x10,0x7C,0x68,0x42,0xC3,0xE7,0xA1,0xCA,0x89, /* 700 */

		0x50,0x35,0xA8,0xFE,0x70,0x10,0x58,0xE3,0x42,0x6E,
		0x17,0xBB,0xF0,0x4C,0x23,0xE7,0x8F,0xFB,0x28,0x3E,
		0x02,0x7E,0x1C,0x63,0x6B,0x1C,0xF9,0xDE,0xD3,0xF5,
		0x90,0x9E,0xBC,0xB0,0xFC,0x63,0x60,0x8E,0x91,0x8C,
		0x9E,0xA9,0xA7,0xF7,0xB6,0xD3,0xEC,0xE7,0x27,0xDA,
		0xC1,0x28,0xD3,0x1B,0x7C,0x0F,0xFD,0x9E,0x43,0x04,
		0x6A,0xE6,0xA5,0x3C,0x25,0x88,0x8D,0x0E,0x60,0x2B,
		0x23,0x02,0xE2,0x55,0xDC,0xA8,0xC5,0x8C,0x10,0xC0,
		0x10,0x26,0x91,0x52,0x58,0x2C,0x59,0x8F,0xDD,0xA0,
		0xB8,0xF4,0x3E,0x31,0x1E,0xA1,0x5B,0xA9,0x6E,0x0D, /* 800 */

		0x9F,0xF3,0x93,0x6F,0x5F,0x18,0x63,0x1F,0xB9,0xD0,
		0x30,0x20,0xE3,0x42,0x64,0x7B,0xE0,0x78,0xC1,0x2A,
		0x94,0x75,0x47,0x4B,0x3D,0xEE,0x55,0xAB,0xC0,0xE3,
		0xDD,0x80,0x4D,0x73,0xFD,0x92,0x9B,0x6A,0xF9,0x4A,
		0x67,0xDD,0x27,0xC3,0x5B,0x5F,0xC2,0xC9,0xBC,0xE5,
		0x00,0xB8,0x10,0x3B,0x98,0x44,0x23,0xCE,0xC7,0x46,
		0x23,0x1A,0x5B,0x81,0x9A,0xCD,0xEA,0x13,0x88,0x16,
		0xE7,0x0A,0x95,0x00,0x5E,0xA9,0x2F,0x72,0x32,0xB6,
		0x66,0xE7,0x72,0xC0,0x60,0xF9,0x5E,0x20,0x61,0x2E,
		0xB7,0xDA,0xD3,0x29,0x7A,0x34,0x2A,0x78,0x17,0xC7, /* 900 */

		0x3E,0x24,0x31,0x8A,0x0B,0x76,0x15,0x62,0xD1,0xCC,
		0xB6,0xB5,0xD6,0x18,0xCB,0xE0,0x6F,0x4B,0x1E,0x7B,
		0x35,0x1B,0x6B,0x83,0x1F,0xC8,0x34,0x79,0xEB,0x34,
		0xBF,0x94,0x7B,0x68,0xB3,0xA1,0xB5,0x57,0xAD,0x86,
		0x68,0x72,0x65,0x6C,0x9F,0x59,0xE7,0x57,0x80,0x61,
		0xE8,0x4D,0xBA,0xE9,0x00,0xAF,0x33,0x01,0xBE,0xF1,
		0xEA,0xA0,0xC6,0x42,0x47,0x46,0x30,0x29,0x30,0xBB,
		0x68,0x5C,0x8F,0x3D,0x97,0x21,0x52,0x1E,0xD6,0x1B,
		0xB6,0x48,0xA4,0xD5,0x33,0x5C,0x4E,0xBF,0x30,0x61,
		0xF8,0x86,0x39,0x41,0x95,0x52,0x42,0xFE,0xEE,0xC8, /* 1000 */

		0x64,0x62,0x82,0x82,0x39,0xF4,0x60,0xF5,0x5C,0xF9,
		0xDE,0x10,0xBA,0xDA,0x56,0x27,0xF9,0xD3,0x32,0x83,
		0x62,0xD6,0xAD,0xA0,0x8F,0x70,0xF0,0xC6,0x5C,0x5A,
		0x15,0x5B,0x2D,0xA6,0x61,0x56,0xA6,0xAA,0xE5,0x55,
		0xC0,0x37,0x13,0x28,0x92,0x49,0x28,0xE0,0x46,0x13,
		0x5D,0xAA,0xF4,0x8B,0x86,0xC1,0xEA,0x78,0xB5,0x6F,
		0x40,0xAF,0xB2,0x79,0x4F,0xB7,0x4B,0x96,0x27,0xE2,
		0xA4,0x3A,0xAB,0xF3,0xE1,0x7A,0x84,0xEE,0x7A,0xD3,
		0x0C,0xF7,0x9E,0xB2,0x0A,0x72,0xAC,0x69),
	.shared = chunk_from_chars(
		0xAC,0x86,0x5F,0x83,0x9F,0xEF,0x1B,0xF3,0xD5,0x28,
		0xDD,0x75,0x04,0xBE,0xD2,0xF6,0x4B,0x55,0x02,0xB0,
		0xFA,0x81,0xD1,0xC3,0x27,0x63,0x65,0x8E,0x4A,0xAC,
		0x50,0x37),
};

ke_test_vector_t mlkem768_1 = {
	.method = ML_KEM_768,
	.seed = chunk_from_chars(
		0xD8,0x1C,0x4D,0x8D,0x73,0x4F,0xCB,0xFB,0xEA,0xDE,
		0x3D,0x3F,0x8A,0x03,0x9F,0xAA,0x2A,0x2C,0x99,0x57,
		0xE8,0x35,0xAD,0x55,0xB2,0x2E,0x75,0xBF,0x57,0xBB,
		0x55,0x6A,0xC8,0x1A,0xDD,0xE6,0xAE,0xEB,0x4A,0x5A,
		0x87,0x5C,0x3B,0xFC,0xAD,0xFA,0x95,0x8F),
	.pub_i = chunk_from_chars(
		0x93,0xC1,0x40,0xF6,0xC4,0x7B,0x7E,0x53,0xB9,0x6F,
		0x72,0xBB,0x18,0x44,0x7D,0x27,0x7C,0xC0,0x21,0xC1,
		0x44,0xA0,0xF7,0xA3,0x5E,0x30,0xB5,0x73,0x86,0xA7,
		0x8A,0xC9,0x76,0x37,0x62,0x62,0x32,0x0A,0x5E,0x7E,
		0x1C,0xB4,0x2E,0x29,0x0D,0xE6,0x84,0x46,0x2C,0xE1,
		0x06,0x7E,0x92,0x0E,0xE8,0x6C,0x32,0x41,0x8B,0x13,
		0x0A,0x5A,0x41,0xA0,0xE8,0x26,0x8C,0xFA,0x7E,0x0D,
		0xB2,0xB4,0x41,0xCB,0x92,0x7D,0x78,0x97,0xC4,0x2B,
		0x1D,0x50,0xF9,0xB3,0x28,0x68,0xA3,0x5A,0x2C,0x04,
		0xCF,0xE9,0x10,0x40,0xE9,0xA9,0x20,0x89,0x02,0xF2, /* 100 */

		0x0C,0x47,0x7E,0x1B,0x1E,0xE5,0xC2,0x90,0xD2,0xE5,
		0x24,0x4E,0xB1,0xB4,0xB7,0xB4,0xC6,0xAD,0x07,0x45,
		0x33,0xB5,0x8D,0x99,0x14,0xA6,0xAA,0x88,0x29,0xF9,
		0x67,0x89,0xF5,0xCB,0x87,0x60,0x75,0x69,0x98,0x30,
		0x03,0xF3,0xA2,0x46,0x1C,0x33,0xC8,0x1A,0x36,0x72,
		0xAF,0x59,0x24,0xC4,0xBA,0x37,0xE6,0x82,0x7F,0xCC,
		0xF8,0x6D,0x8B,0x41,0x03,0xFB,0xE9,0xC0,0xF6,0x22,
		0x6D,0xD0,0xA2,0x14,0x5A,0x6B,0x7A,0xEC,0x76,0xB1,
		0x86,0x46,0x6F,0x9C,0x67,0xBF,0x16,0x90,0x39,0x25,
		0x95,0x74,0x45,0x64,0x97,0x14,0x0C,0x8C,0xF4,0xAC, /* 200 */

		0x05,0x09,0x19,0x73,0xAC,0x8C,0x08,0xD8,0x09,0x46,
		0x57,0x85,0xA6,0x77,0xA0,0x32,0xAC,0x09,0xAD,0x1D,
		0x66,0x6E,0x8C,0x48,0x46,0x28,0x13,0xCE,0x5A,0xC7,
		0x5F,0x18,0x4B,0x38,0x25,0x1C,0x30,0xE3,0x62,0xB0,
		0xE2,0x50,0x1D,0x68,0x00,0xC8,0xAD,0x10,0x3C,0x8B,
		0x77,0x37,0x80,0xB6,0x71,0x7C,0xF1,0x5C,0x40,0x11,
		0x39,0xAC,0xD5,0x4B,0x15,0x98,0xB2,0xB7,0xC7,0x94,
		0x92,0xA8,0x66,0x31,0x09,0x02,0x68,0xC7,0x0D,0x87,
		0x5B,0xC0,0x40,0xCB,0x2B,0x75,0xA3,0x86,0xFA,0x96,
		0xB0,0x92,0xB8,0xCB,0xC2,0x5C,0x47,0xF7,0x0A,0xA7, /* 300 */

		0x6C,0xD8,0xB9,0xAF,0xC1,0x2B,0x42,0xB5,0x36,0xE2,
		0x7C,0x55,0x78,0x83,0x1A,0x96,0xDB,0xBA,0xB7,0x13,
		0x8C,0x3F,0x24,0x7E,0x95,0x5A,0x6C,0x08,0xB4,0x40,
		0x7D,0x47,0x08,0xF1,0x91,0x4B,0xFA,0x48,0xAF,0x4B,
		0x28,0x53,0x3F,0x74,0x7B,0x86,0x0B,0x70,0x76,0xC0,
		0x28,0xE2,0x45,0xC9,0x72,0x7B,0x42,0xF3,0x24,0x8F,
		0xB0,0x40,0x8B,0x3E,0xF0,0xC4,0x91,0x8A,0xB7,0x6A,
		0xB9,0x6D,0xAC,0xA8,0x1A,0xFB,0x12,0x11,0xAB,0x3A,
		0x03,0x29,0xBA,0x1A,0x5B,0x06,0x9A,0x68,0x93,0x4C,
		0x1C,0xE8,0x4C,0x2F,0x72,0x83,0x93,0x11,0x25,0x7F, /* 400 */

		0xA1,0x9E,0x72,0xC6,0x2F,0xB5,0x68,0x6B,0x61,0x41,
		0x6C,0xAF,0x8B,0x22,0xD2,0xB2,0x6A,0x6D,0xD0,0x1B,
		0xB7,0x38,0x7F,0x88,0xEB,0x86,0x06,0x98,0x0A,0x5E,
		0x22,0x59,0xCB,0xD5,0x6C,0xA1,0xEC,0x05,0x1C,0xFC,
		0x66,0xF9,0x62,0x39,0x99,0x1A,0x23,0x60,0xD7,0x5D,
		0xAF,0xA1,0x53,0x4C,0x8A,0x05,0xAB,0x9C,0x95,0xE2,
		0xE5,0x86,0x85,0x37,0x14,0x60,0x0A,0x34,0x55,0x51,
		0x1B,0x62,0xD9,0x45,0x25,0xD8,0xB1,0xA9,0x65,0xCE,
		0x69,0xF9,0x86,0x73,0x18,0x88,0xD1,0xEC,0xCE,0x53,
		0x60,0x00,0xE4,0x86,0x33,0x22,0xA8,0x3E,0x3C,0x84, /* 500 */

		0x75,0xD9,0xEB,0xA5,0x41,0x4A,0x65,0xB0,0x56,0x1D,
		0x24,0xA7,0xB0,0x9D,0xCA,0x6F,0xFE,0x23,0x83,0x60,
		0x58,0xBE,0xC0,0xA8,0x56,0x71,0x64,0x10,0x08,0xC8,
		0x31,0x23,0x3C,0x78,0x90,0x78,0x3D,0xAF,0xE8,0xC3,
		0x59,0xA2,0x63,0xE7,0x96,0x86,0x91,0x18,0xC4,0xE7,
		0x40,0x02,0xDC,0x3B,0x0E,0xC5,0xB1,0xC4,0xAC,0x64,
		0x1B,0xAF,0xE3,0x0E,0xB2,0xBB,0x74,0x37,0x13,0xBF,
		0xED,0x2C,0x1E,0x79,0xC4,0xA6,0xF4,0x65,0x93,0x00,
		0x6A,0xCB,0x35,0xF7,0x03,0x1F,0x19,0x48,0x07,0xB8,
		0x93,0xAA,0x7A,0x93,0x23,0x16,0x21,0x68,0xD3,0x84, /* 600 */

		0xD0,0xE9,0x6F,0x2F,0x23,0x36,0x86,0x26,0xA2,0xE0,
		0x27,0x89,0x1A,0xC9,0x4F,0x12,0xF3,0x0A,0x04,0x4B,
		0x85,0xF0,0x32,0x2C,0xE2,0xB1,0x65,0x17,0xD3,0x6A,
		0xA3,0xF2,0x3B,0x5F,0x39,0x1B,0xE3,0x94,0xB5,0x4D,
		0xDC,0x28,0xE5,0xA9,0x77,0x53,0x72,0x86,0x9A,0x7B,
		0x50,0xB4,0x89,0x8C,0x07,0x2C,0x06,0x69,0xD4,0x3C,
		0x64,0x1A,0x3C,0xB5,0xE8,0xC1,0x70,0xBA,0x63,0xBE,
		0xD6,0xAD,0x10,0x08,0x61,0x52,0x01,0x30,0x5E,0x29,
		0x2E,0x1C,0x81,0x8E,0x1E,0xF6,0x4F,0x3E,0x0C,0x52,
		0xFB,0xA6,0x7E,0x72,0x49,0xC1,0xEC,0x4A,0x14,0x0D, /* 700 */

		0xC8,0x9C,0xE0,0x05,0x06,0x47,0xF1,0xC1,0x9F,0x58,
		0x97,0xC7,0x7B,0x90,0x59,0x04,0x0B,0x4B,0x5B,0x42,
		0x82,0xE0,0x66,0x9C,0xB4,0xC6,0x58,0xF4,0x8F,0x67,
		0x33,0x84,0x13,0xC9,0x2F,0xC7,0x41,0x2E,0x44,0x4B,
		0xC6,0x35,0xDA,0xBC,0x93,0xC5,0x1B,0x03,0x07,0x84,
		0xF2,0x14,0x61,0xDC,0xA2,0x10,0xCC,0x6A,0x54,0xFD,
		0xC2,0x5B,0xBA,0x09,0x70,0x5B,0xAA,0x77,0xC1,0x82,
		0x66,0x36,0xB1,0x66,0x44,0xA5,0xAD,0xA5,0x12,0x70,
		0xD8,0x31,0x7A,0x78,0x7A,0xBF,0x53,0x19,0x2D,0x40,
		0xDC,0x76,0x5B,0xFB,0x45,0xC7,0xDC,0xAE,0x58,0x48, /* 800 */

		0x1A,0xF3,0xA4,0x9F,0xE7,0x62,0xB9,0x0B,0xAC,0x1F,
		0x6F,0x69,0xA5,0xDE,0x80,0x45,0x98,0x1C,0x67,0x90,
		0x6B,0x9F,0x69,0x26,0x25,0xFC,0xB6,0xA4,0xDF,0x26,
		0x41,0x68,0x8A,0xBA,0x61,0xFA,0x49,0xA8,0x4B,0x47,
		0xC9,0x66,0x61,0xE4,0x70,0x1E,0x58,0x26,0xC1,0x10,
		0x80,0x69,0xC4,0xB9,0xC6,0xE4,0x80,0x25,0x1C,0x4B,
		0x02,0x1D,0x11,0xCF,0xDA,0xCA,0x12,0xBB,0xF9,0xC0,
		0x9A,0x02,0x34,0x18,0x66,0x68,0xE3,0x89,0x45,0x4A,
		0xC4,0x04,0x6F,0x5B,0x24,0x68,0xAA,0x2F,0xA9,0xB1,
		0x17,0x86,0x85,0x74,0xE7,0x29,0x57,0x4E,0xE5,0x7E, /* 900 */

		0xA7,0x49,0x5A,0xF3,0x10,0x62,0x7B,0x85,0x91,0x6D,
		0x6B,0x4C,0x70,0x83,0x74,0xF9,0xFB,0x0C,0x9C,0x3A,
		0xA4,0x66,0x4C,0x66,0x51,0x28,0x1D,0x8E,0xEB,0x98,
		0x13,0x2C,0x1A,0xF1,0x3A,0x81,0x6C,0xA5,0xB8,0x69,
		0xF5,0x0B,0x50,0xB2,0x91,0xEF,0xD4,0x66,0xDE,0xA0,
		0x90,0x91,0x86,0x30,0xC6,0xFC,0x77,0x24,0x7C,0x45,
		0x8A,0xA8,0x25,0x68,0xA4,0x1D,0xD9,0xD4,0x70,0x8A,
		0xF3,0xBB,0xCA,0x5A,0x57,0x16,0x44,0x7E,0x8C,0x2C,
		0x24,0xAD,0x78,0x8A,0x86,0x32,0x47,0x3F,0xDC,0x4A,
		0x25,0x71,0x96,0xBD,0x2B,0x54,0x90,0x91,0x17,0x6F, /* 1000 */

		0xE6,0x54,0xD3,0x69,0x4E,0x10,0xA1,0x18,0xF7,0xB6,
		0x6D,0x1C,0xD5,0xAF,0x19,0x9B,0x6E,0x90,0xB3,0xBF,
		0xFA,0x88,0xA9,0x2E,0x36,0x71,0x7F,0xD4,0xBB,0xFC,
		0xC7,0x8B,0x0E,0x08,0x63,0x4B,0xC0,0x80,0xC8,0x68,
		0x85,0xC9,0x2C,0x8E,0xF6,0x7A,0xA4,0x93,0x3C,0x7D,
		0xF0,0x97,0x89,0x13,0x69,0xBA,0xC5,0xFC,0x5D,0x0C,
		0x36,0x73,0x7A,0xA6,0x09,0x71,0x31,0x0A,0xDF,0x02,
		0xC2,0x57,0xF7,0x6A,0x7A,0x7B,0x3F,0x7E,0x13,0x2C,
		0x2A,0x71,0xC5,0x2F,0xF4,0xAF,0xFB,0x68,0x46,0x39,
		0x1C,0x08,0x68,0x24,0x1A,0x3D,0xF1,0x34,0x08,0xD4, /* 1100 */

		0x19,0xA7,0x8B,0xCF,0xFD,0x49,0x61,0x9B,0x03,0xA1,
		0x92,0xC6,0x83,0x90,0x0A,0x24,0x42,0x89,0xC9,0xF7,
		0xB4,0x56,0x48,0x23,0x90,0x0C,0xEA,0xCF,0xFD,0x9A,
		0x93,0x71,0x26,0x0D,0x5E,0x57,0xA8,0x27,0x11,0x96,
		0xF4,0x75,0x9E,0xEA,0xD0,0xCE,0xAC,0x31,0x89,0x66,
		0xE7,0x6F,0x68,0xDE,0x95,0xAB,0x9D,0xB2,0xBA,0x4F,
		0xBF,0x83,0xC3,0xB2,0x70,0x92,0xCD,0x33,0x9C,0xFE,
		0x48,0xD5,0xCA,0x0B,0xA1,0x15,0x91,0xD0,0x45,0x66,
		0xF4,0xED,0x24,0xA5),
	.pub_r = chunk_from_chars(
		0x4D,0xDF,0x4E,0x29,0x19,0x94,0x38,0x17,0xAB,0x3B,
		0x4C,0xB0,0x85,0xDA,0xEF,0xE5,0xFE,0xC7,0x74,0xD8,
		0x3D,0x96,0xC2,0xCE,0xE7,0x41,0x49,0xE2,0x5A,0xDD,
		0xD1,0x5F,0x71,0xF2,0x6C,0x93,0x12,0xD6,0xEE,0xE0,
		0xB9,0xA9,0xFA,0x77,0xB0,0xBD,0x20,0x2C,0xD7,0x52,
		0xF7,0x5A,0x7D,0xA5,0x75,0xA2,0xA6,0x6C,0xD1,0xC6,
		0x08,0xF5,0x4E,0x3C,0x60,0x9F,0xE0,0x96,0xCB,0x6C,
		0x5B,0x83,0x2E,0xFF,0x66,0x20,0x48,0xD8,0xA7,0x89,
		0x74,0x5A,0x83,0x16,0x13,0x2F,0x3B,0x5B,0x35,0x21,
		0x64,0xEA,0x69,0x65,0x37,0x90,0x3A,0x26,0x66,0x23, /* 100 */

		0xDD,0x35,0x2A,0x3F,0xA8,0xCD,0xDE,0x4A,0x25,0xD3,
		0x95,0x01,0x9E,0x5A,0x41,0x8F,0xF4,0x20,0x25,0x9B,
		0x1B,0xB5,0xA3,0xBA,0xC5,0xA7,0xDD,0xA1,0xE3,0x86,
		0xB8,0x2E,0x58,0x8A,0x79,0xF2,0xB8,0x41,0xB1,0x17,
		0x58,0xB3,0x6B,0x89,0x82,0x13,0x80,0x25,0x8C,0x8E,
		0xD6,0x67,0x25,0xE9,0x0D,0x7A,0xF0,0x73,0x39,0xBA,
		0xD6,0x17,0x10,0xB1,0xBE,0x0A,0x72,0xA9,0x8B,0x3E,
		0x59,0x0D,0xDA,0xF4,0x81,0x65,0xCE,0xFF,0x91,0x8F,
		0xD3,0x23,0x84,0xCF,0x61,0x2E,0xB7,0x13,0xE8,0x56,
		0xBC,0x61,0x78,0xEE,0xAA,0x0C,0x8A,0xA6,0xFF,0x63, /* 200 */

		0x8A,0xF2,0xD6,0xAA,0x33,0x91,0x68,0x7B,0xD7,0xB3,
		0xEB,0x6E,0xB9,0x21,0x75,0xC3,0x02,0x39,0xAC,0x7C,
		0xE0,0x26,0xCD,0xFB,0x3D,0x52,0xAF,0x16,0x8E,0x3E,
		0xEE,0x00,0x5F,0x8D,0xBA,0x59,0x30,0x58,0x8B,0x41,
		0x48,0x98,0x7C,0x82,0x04,0xDF,0xFA,0x60,0xEC,0x1A,
		0xB4,0x49,0xA0,0x9F,0x9D,0x39,0xA6,0x0F,0xFE,0xF1,
		0xAA,0xDC,0x44,0xBB,0xDB,0x2A,0x6B,0xB3,0xB6,0x1B,
		0x94,0x9C,0x1B,0xDB,0x7C,0x1C,0x63,0x86,0xBB,0x89,
		0x78,0x9D,0x09,0xBD,0x71,0xE4,0xF5,0xCA,0x72,0xC7,
		0xF4,0x71,0x81,0x06,0xB2,0x75,0xAE,0x17,0x05,0x55, /* 300 */

		0xE7,0xB9,0xF3,0x2B,0xFE,0x75,0xDD,0x44,0x5B,0xD3,
		0xE8,0x6E,0xB1,0xB1,0x04,0x49,0x75,0x9E,0xE2,0xEC,
		0xAD,0xD4,0x12,0x62,0xAF,0xE5,0x04,0x52,0xFE,0xA9,
		0xC1,0x37,0x15,0x61,0xF6,0x94,0x6E,0x8F,0x79,0x7F,
		0x2F,0x6B,0xF8,0x42,0xB5,0x0D,0xC7,0x2F,0x38,0x6B,
		0x4B,0x41,0xCD,0x03,0x80,0x85,0xCC,0xD6,0x1E,0xBE,
		0xE0,0x77,0x30,0x09,0x88,0xB1,0x75,0x64,0x45,0xD7,
		0x98,0x2B,0x93,0x95,0xAE,0x02,0x49,0x3B,0x83,0xB0,
		0x64,0x86,0xE7,0x21,0x5E,0xCF,0xC3,0x25,0x4E,0x01,
		0x2B,0xC0,0x58,0x62,0x6B,0xF2,0xD5,0xB2,0xE0,0xB3, /* 400 */

		0x41,0xAD,0x37,0x80,0x11,0x1F,0x7D,0x79,0xDE,0x7B,
		0x5F,0xCD,0xAB,0x1C,0x93,0xB1,0xF6,0xF9,0xEE,0xF1,
		0x6D,0x30,0x5B,0x83,0x3C,0x65,0x53,0x4D,0x24,0xF7,
		0xF7,0x4E,0xB0,0x0F,0xFD,0x18,0xFE,0x47,0x69,0x83,
		0x82,0x3B,0xF9,0xB9,0xBC,0xB7,0x00,0x4C,0x57,0xBB,
		0x88,0x49,0x78,0xE3,0x6B,0x2F,0x5B,0x2E,0xCF,0x2D,
		0x9C,0x7B,0xA4,0xED,0xFE,0x4B,0xBD,0x7A,0xFB,0x11,
		0xD3,0x62,0x71,0x7E,0xA4,0x73,0xF1,0x19,0xE5,0xEC,
		0xC2,0xDB,0x58,0x0A,0xE0,0x13,0xC6,0x0A,0xE6,0xB7,
		0x4D,0xDA,0xF8,0x57,0xF4,0xF1,0x88,0x42,0xF3,0x37, /* 500 */

		0x44,0x6B,0xD5,0x62,0x6E,0xD3,0x60,0x26,0x51,0x32,
		0x37,0x77,0xEE,0x1E,0x78,0xB2,0x8F,0x79,0x5C,0x19,
		0xF7,0x18,0x01,0x4B,0xC9,0x2E,0x71,0x17,0x7D,0x6A,
		0x44,0x59,0xFF,0x13,0x84,0xCB,0xA0,0xF4,0x3D,0x1F,
		0x46,0xD6,0xEA,0xC0,0x2C,0x29,0xFC,0xCB,0x95,0x9E,
		0x14,0xAB,0xD3,0x72,0x3C,0x93,0xD3,0x50,0x5B,0x5B,
		0x44,0x4A,0x7A,0xB0,0x47,0x0E,0x2D,0xDF,0x85,0xF1,
		0xAD,0x79,0xF5,0x5F,0x4D,0x1F,0xAB,0x17,0x0D,0x02,
		0x75,0x4C,0x1C,0x28,0x88,0x1C,0x4A,0x6A,0x14,0x06,
		0x73,0x9A,0x44,0x8D,0xFE,0x29,0xF2,0x4C,0x8F,0xBA, /* 600 */

		0x86,0x5C,0x7B,0x6D,0x57,0x18,0x02,0xC8,0x12,0x19,
		0x94,0x51,0x94,0xE9,0x3A,0x51,0x8C,0xBF,0x42,0x2C,
		0xE4,0x7D,0x8D,0x80,0x89,0xA7,0x99,0xE3,0x7E,0x9B,
		0x5A,0x9C,0xDA,0x35,0x1F,0x2C,0xD4,0x02,0x43,0xC2,
		0xED,0xD2,0x8B,0x64,0x33,0x3A,0x5A,0x17,0x57,0x83,
		0x0F,0x8C,0x32,0xF3,0x2F,0x4F,0x9D,0x0F,0x7E,0x65,
		0x48,0xB3,0xDD,0x2F,0xB9,0xC5,0x2B,0xBC,0xBC,0x81,
		0x76,0xB2,0x9B,0x9B,0xC1,0x01,0xDB,0x66,0x90,0x17,
		0x72,0x50,0x5A,0x48,0xF6,0xC6,0x3E,0x50,0xBB,0x3E,
		0xD8,0x33,0x1D,0x27,0xD8,0xC0,0xB5,0x76,0x8A,0x16, /* 700 */

		0x46,0x96,0x0A,0xD4,0xC0,0x44,0x56,0x29,0x59,0x9D,
		0x4B,0x4E,0x34,0x6F,0x8B,0x63,0xF2,0x57,0xD2,0xDE,
		0x7A,0xBA,0xE1,0x99,0x89,0x0C,0x3F,0x2A,0x55,0xE3,
		0xA6,0x29,0x46,0xA5,0x60,0xF6,0x04,0x13,0x96,0x72,
		0xFF,0x9E,0xD5,0xD8,0xDF,0xBA,0x8E,0x61,0x90,0x54,
		0xE8,0x3B,0xF2,0xE8,0x6F,0xEC,0x1D,0x3B,0x3C,0x4E,
		0x5E,0xC0,0xFD,0x9F,0x22,0x64,0x44,0x2B,0x4C,0x25,
		0xFC,0xC3,0x06,0x20,0xD0,0x60,0xD1,0x38,0xB1,0x7F,
		0xE7,0x83,0x8E,0x1C,0x8D,0x3D,0x0A,0x0E,0x8A,0xC6,
		0x4A,0x57,0xD9,0x8E,0x99,0x72,0x96,0x2D,0xDF,0x94, /* 800 */

		0x3B,0x6F,0x80,0x12,0xD4,0x0D,0x37,0x00,0x2F,0x3B,
		0x9C,0x94,0x52,0xDA,0xBB,0x26,0x33,0xD3,0x87,0xE8,
		0x38,0x75,0xA0,0x11,0x3B,0x24,0x07,0x7A,0xB8,0x86,
		0x6F,0xC6,0x23,0x2F,0x73,0x5C,0x03,0x19,0x78,0x23,
		0x4D,0x35,0xEE,0xC0,0x2E,0x1B,0xAA,0x80,0x89,0xF0,
		0x4D,0xB2,0x50,0x2F,0xB3,0x24,0xF3,0xBD,0x71,0x5C,
		0x08,0x36,0x9C,0xBD,0xFE,0xDF,0xDD,0x07,0x92,0x76,
		0x50,0x06,0xDA,0xAF,0x92,0x14,0x94,0xC7,0x01,0x27,
		0xCC,0xE8,0x24,0x79,0xAB,0x5E,0x88,0xBB,0x02,0xA1,
		0x90,0x62,0x79,0xCB,0xF6,0x9A,0x26,0xC0,0x19,0x10, /* 900 */

		0xD7,0x8B,0x9F,0xA1,0x1A,0x4B,0xED,0x36,0x2F,0xF4,
		0x54,0xC1,0x29,0xC2,0x28,0xBE,0x14,0xCA,0x49,0xE3,
		0xA3,0x62,0x7A,0xB2,0x06,0x8C,0x3C,0xB7,0x79,0xD4,
		0x7D,0xA9,0x10,0xE0,0xB7,0x6E,0x08,0x2F,0x0C,0x71,
		0x0C,0xCE,0x75,0xCD,0xB5,0xDF,0xE3,0xA1,0x20,0x45,
		0xA7,0xFE,0xD3,0xF0,0xAE,0x17,0x1F,0xF6,0x72,0xC4,
		0xA4,0xEA,0x30,0x76,0x77,0xB9,0xB1,0x3F,0x5C,0x5E,
		0x39,0x9E,0x9D,0xF1,0xCE,0xE8,0x55,0xB3,0x25,0x95,
		0x3F,0x14,0xAE,0x4D,0xAD,0x7A,0x91,0x18,0x90,0x39,
		0x73,0xFF,0x96,0xBB,0x30,0xDF,0xBE,0x74,0x4F,0x2B, /* 1000 */

		0x3E,0x54,0xF2,0x41,0xAC,0xC4,0x41,0xA8,0x9C,0x10,
		0x46,0xC2,0xCD,0x6B,0x48,0x5B,0x22,0xA7,0x6E,0x8C,
		0xDE,0x39,0x71,0x63,0x83,0xFF,0xDE,0xD3,0x35,0x26,
		0xD1,0xCD,0xA2,0x48,0x72,0x33,0x1D,0x7A,0x7F,0x2F,
		0x27,0x3D,0xE0,0x64,0x21,0x6B,0x71,0x72,0xF1,0xC9,
		0x00,0x6A,0x86,0x4B,0xF9,0x38,0x21,0x3E,0x16,0xC2,
		0x1B,0xB3,0x6B,0xA3,0xFB,0xC0,0x9C,0xA4,0x0C,0x80,
		0x81,0xA7,0xB2,0xE1,0x64,0xD9,0x05,0x7F,0x90,0x65,
		0xC8,0xB4,0xD6,0x50,0x05,0x12,0xBE,0x46),
	.shared = chunk_from_chars(
		0x95,0x04,0x92,0xE9,0x40,0xBF,0xA8,0x6A,0xC2,0x81,
		0x83,0xDE,0x02,0xEF,0xC0,0x40,0x17,0xD1,0x75,0x51,
		0x06,0xA8,0x54,0xFB,0xB2,0xAC,0x68,0x17,0x4F,0x70,
		0x5B,0x41),
};

ke_test_vector_t mlkem768_2 = {
	.method = ML_KEM_768,
	.seed = chunk_from_chars(
		0x64,0x33,0x5B,0xF2,0x9E,0x5D,0xE6,0x28,0x42,0xC9,
		0x41,0x76,0x6B,0xA1,0x29,0xB0,0x64,0x3B,0x5E,0x71,
		0x21,0xCA,0x26,0xCF,0xC1,0x90,0xEC,0x7D,0xC3,0x54,
		0x38,0x30,0x55,0x7F,0xDD,0x5C,0x03,0xCF,0x12,0x3A,
		0x45,0x6D,0x48,0xEF,0xEA,0x43,0xC8,0x68),
	.pub_i = chunk_from_chars(
		0xC0,0x2B,0x82,0x02,0x92,0xC4,0x6C,0x8A,0x8A,0x92,
		0x97,0x5A,0x2E,0x27,0x29,0x2C,0xF8,0x93,0xA7,0xF5,
		0x4A,0x84,0xA6,0x70,0x22,0x84,0x5D,0x66,0x23,0x6D,
		0xCA,0x3C,0x48,0x44,0x60,0x4B,0x16,0x87,0x9E,0x84,
		0x9C,0x7F,0xE3,0x47,0x5D,0x03,0xF8,0x23,0x23,0xD8,
		0x22,0xAE,0x2C,0xB6,0x76,0x38,0x42,0xA3,0x19,0x11,
		0x4F,0x5C,0x77,0x75,0x46,0x13,0xD5,0xFB,0xAE,0xC0,
		0x5A,0xA8,0x01,0xA0,0x3A,0xC1,0x98,0x9A,0x72,0x53,
		0x37,0xEA,0x38,0xA6,0x68,0xEA,0x65,0xCD,0xB3,0x63,
		0x43,0xEC,0x3D,0x69,0x0C,0xCD,0xFA,0x47,0xAB,0x1F, /* 100 */

		0xE4,0x73,0x24,0x7B,0xC4,0x88,0xD9,0x8E,0x1E,0x33,
		0x17,0x59,0x20,0xA0,0xF2,0x27,0x2A,0x03,0x98,0x2C,
		0x83,0x86,0x48,0x5B,0xC2,0xC4,0x24,0x58,0x84,0xF1,
		0xD9,0x41,0x75,0x0A,0x1D,0x27,0xFA,0x85,0xF0,0x06,
		0x7F,0x5C,0x37,0x75,0x32,0x30,0x42,0x61,0xAA,0x90,
		0x8B,0xFB,0x95,0x08,0x94,0xCE,0xCC,0x1B,0xBF,0xD1,
		0x97,0xCE,0xD7,0x8A,0x64,0x13,0x66,0xB9,0x19,0x9A,
		0x4B,0xA1,0x90,0x44,0x6E,0x03,0xAE,0x51,0x54,0x9B,
		0xE7,0x95,0x9C,0x87,0xA6,0x29,0xAB,0xF2,0x4C,0xBB,
		0x7A,0x58,0xD0,0x1C,0xAD,0x49,0x49,0x73,0x4F,0x79, /* 200 */

		0x7A,0xE3,0x00,0xA7,0x8F,0x45,0xC7,0x0F,0x66,0x85,
		0x06,0x97,0x7D,0x7C,0x85,0xA0,0xCF,0x64,0xC7,0x00,
		0xA5,0xB1,0xF7,0x87,0x98,0x01,0xBC,0x3A,0xC7,0x55,
		0x84,0x30,0x2C,0x8B,0xC8,0x1C,0x0D,0x7F,0x6B,0x23,
		0x7A,0x94,0x00,0x68,0x1B,0xAC,0xFC,0x71,0x92,0xF9,
		0xC6,0x53,0xF8,0xA9,0x55,0x1F,0x79,0x6C,0xCF,0xC2,
		0x62,0xD3,0xA4,0x38,0xA5,0x51,0xAD,0x74,0xC8,0x0A,
		0xE9,0xE3,0xB5,0xA7,0xF1,0x2C,0x52,0x04,0xAD,0x44,
		0xC2,0x1F,0x48,0x8B,0x78,0x24,0xA4,0xB1,0xEE,0xD7,
		0x9D,0xA9,0xB4,0x4E,0x60,0xD8,0xB2,0x25,0x49,0x0C, /* 300 */

		0x06,0x2B,0x92,0x4D,0xA0,0x4F,0x0E,0xF0,0x25,0x23,
		0x04,0xCC,0xA3,0x4B,0xB4,0xFC,0x70,0xCE,0x6B,0x18,
		0x00,0xFB,0x6C,0x5D,0xC7,0x71,0xBF,0x7B,0xDA,0x9E,
		0x9D,0xB2,0xB6,0xC6,0x04,0x6A,0x16,0x72,0xC6,0xF6,
		0xF0,0x1A,0x04,0x06,0xB9,0x03,0x41,0x1C,0x27,0x1B,
		0x20,0xE1,0x39,0xA6,0x91,0x38,0xAD,0xFB,0xF5,0x0C,
		0x11,0x34,0xAE,0x08,0x99,0xC1,0x66,0xB1,0xB1,0xA5,
		0xD8,0x18,0x51,0x5B,0x0D,0xD9,0x48,0x95,0xA2,0x53,
		0x05,0x9B,0x16,0x76,0x30,0xF3,0xA0,0xDD,0x3C,0x64,
		0xCF,0x25,0x7C,0xDF,0x93,0x59,0x6E,0x54,0x28,0x8C, /* 400 */

		0x7A,0x45,0x80,0x40,0x4B,0xA8,0xDA,0x54,0x1A,0xBB,
		0x1B,0x6A,0xEC,0x7E,0x53,0x0C,0x58,0xA6,0x1A,0x64,
		0xD9,0xB9,0xBC,0x32,0xD8,0x70,0xBC,0x10,0x5E,0xF7,
		0x06,0x3B,0x9A,0x97,0x51,0x9B,0x87,0x1A,0x60,0xD7,
		0xAA,0xF8,0xC1,0x93,0xE9,0x48,0xC2,0x67,0xC9,0xC8,
		0x15,0x06,0x3D,0x6A,0x60,0x3E,0xE8,0xD2,0x24,0x9B,
		0x38,0x6D,0x94,0x39,0x58,0x04,0xA9,0xAF,0x8C,0xD2,
		0x18,0x48,0x00,0xC7,0xCC,0xFB,0xAE,0xD9,0x8C,0x4F,
		0x9F,0x95,0xB7,0x01,0x45,0xB4,0x48,0x38,0x71,0x83,
		0xF5,0xCC,0x8F,0x4B,0x5F,0x35,0xA9,0x49,0xBC,0x74, /* 500 */

		0x70,0x6B,0xD0,0x3D,0xB7,0x64,0x5C,0x2D,0x08,0x46,
		0x24,0x84,0x4A,0x55,0xA0,0xA5,0xAB,0xB9,0x24,0x4C,
		0xB5,0xB8,0x73,0x6B,0x83,0x5B,0x51,0x2B,0xCD,0x91,
		0x23,0x61,0xFB,0x66,0x04,0x79,0xB2,0xC5,0x1B,0x99,
		0x73,0x49,0x70,0x80,0x8B,0xA5,0xC6,0xDB,0x01,0x69,
		0x75,0x6C,0xFC,0x98,0x6D,0xBD,0xA9,0x49,0x15,0x56,
		0x3C,0x15,0xA1,0x09,0xAE,0xB7,0x7C,0x01,0xA7,0xCF,
		0xF5,0x5C,0x84,0x30,0x69,0x6D,0x6E,0x83,0xAE,0xA5,
		0xD4,0x92,0x92,0x46,0x2D,0x72,0x5C,0x42,0x8B,0x98,
		0x4D,0x15,0xE9,0x7E,0xD4,0xA7,0x64,0x03,0x09,0x46, /* 600 */

		0x99,0xE1,0x33,0x64,0x26,0xB5,0x42,0x81,0x06,0x7E,
		0x67,0x57,0x88,0xB3,0xA4,0xFC,0xEB,0xCA,0x44,0x31,
		0xBF,0xDC,0x8A,0x49,0xC2,0x15,0xBA,0x11,0x00,0x4A,
		0x34,0x15,0x16,0x01,0x84,0x19,0xAB,0xEA,0xB2,0x37,
		0x41,0x84,0xB0,0xE2,0xB7,0xB7,0xA5,0x4E,0xFC,0x67,
		0x8E,0x81,0x25,0x88,0x96,0x0A,0x0D,0x00,0x94,0x18,
		0xE5,0x28,0x9E,0xDD,0xCA,0xC5,0x82,0xD3,0xC7,0x3D,
		0xD9,0x3E,0x12,0x34,0x51,0x8D,0xA0,0x03,0x64,0x88,
		0x53,0xC5,0xB5,0xA2,0x4B,0x38,0x9E,0x88,0x30,0x0A,
		0xEA,0xE7,0x8C,0x41,0xE6,0x2E,0x5E,0xE3,0x03,0xA5, /* 700 */

		0x6B,0x54,0x83,0x75,0x75,0x7C,0xD4,0x2E,0x21,0xA7,
		0xC5,0x9F,0xF2,0x9A,0x13,0x38,0x68,0xD1,0xD0,0x78,
		0x41,0x97,0x57,0x34,0x64,0x29,0x25,0x47,0x16,0x1B,
		0x3A,0x86,0x89,0xDC,0x0C,0x7D,0xE1,0x11,0x39,0xD4,
		0x07,0xF0,0xB5,0x78,0x4D,0x94,0x22,0x78,0xD4,0x94,
		0x62,0x96,0x5F,0x21,0xF7,0x2C,0xB9,0x92,0xAD,0xD1,
		0x60,0x62,0x0F,0x6A,0x82,0xF7,0x0A,0xA9,0xA8,0x30,
		0x19,0x0C,0xE3,0xC9,0xFA,0x80,0x15,0x07,0x1C,0x28,
		0x77,0x48,0x3C,0x22,0x03,0xBA,0xB7,0x22,0x8E,0xE2,
		0xF3,0x7B,0xBB,0x17,0xC1,0xA4,0xBC,0x49,0x9E,0xC2, /* 800 */

		0x49,0xFF,0xBB,0x42,0xE4,0x98,0x99,0x05,0xA8,0x78,
		0xDC,0x87,0x3E,0x45,0x6A,0x7D,0x9D,0xB9,0x46,0x57,
		0x29,0x36,0xD8,0xD8,0x17,0x77,0xD1,0xB9,0x92,0xC0,
		0x0D,0xB1,0x2A,0x35,0xDF,0x13,0x48,0xCE,0x8C,0x9D,
		0x1A,0x02,0x89,0x8C,0x74,0x03,0xD4,0x10,0x7F,0x45,
		0xA6,0xBE,0xE8,0x64,0x37,0x5B,0x62,0x3D,0x30,0xE0,
		0xC1,0x91,0xF8,0xAD,0x83,0x5A,0x3F,0x37,0x09,0x36,
		0x2C,0xC7,0x63,0xD6,0x3C,0x2B,0x91,0x64,0x9D,0x72,
		0x28,0x25,0x40,0x2C,0x9E,0xC1,0xB1,0xC7,0xAA,0x02,
		0xB1,0x54,0xD6,0x9E,0x39,0xDA,0x19,0x6E,0x72,0xA2, /* 900 */

		0x59,0x28,0x21,0x2F,0xE0,0x71,0x1B,0xBB,0x2F,0xD2,
		0xF9,0x55,0x0A,0x36,0x1A,0x28,0xD4,0x29,0x5B,0xF0,
		0xC0,0x36,0xD7,0xB7,0x9B,0xE9,0x13,0x02,0x32,0x06,
		0x02,0x06,0xC9,0x19,0xF8,0xB8,0x64,0xF0,0x3D,0x6F,
		0x02,0x34,0x8E,0xD2,0xC8,0x00,0x1A,0x30,0x12,0x4B,
		0x40,0xC6,0xD6,0x1D,0xC3,0xE1,0x8B,0xCB,0x27,0xC6,
		0x0B,0xD1,0x32,0x4D,0x83,0x86,0xDA,0x99,0xC3,0xE0,
		0x98,0xCF,0xB9,0x4B,0x01,0x80,0x31,0xA4,0xD6,0xAA,
		0x57,0x88,0xD2,0x7A,0x75,0xA6,0x7B,0x63,0x03,0x27,
		0xAD,0xD8,0x04,0xAD,0xC2,0x16,0x42,0xD5,0x80,0x80, /* 1000 */

		0x95,0x6A,0x03,0xF0,0x3C,0x81,0x93,0xAC,0x30,0x2C,
		0x92,0xE9,0x49,0xCE,0xA3,0x17,0x21,0x81,0x33,0x90,
		0x38,0x2C,0x81,0x4B,0x2C,0x5A,0x6D,0x34,0xA1,0xEB,
		0x44,0xB8,0xB6,0x2A,0xA7,0x7B,0xF8,0x8B,0x50,0x67,
		0x45,0x5F,0xC0,0xC7,0xBE,0x96,0x7C,0x21,0xBC,0x28,
		0x1A,0xD3,0x8E,0xA6,0x97,0x8D,0x22,0x8A,0x83,0xA8,
		0x20,0xC5,0x7B,0x40,0x39,0x17,0x5C,0x91,0x0A,0x45,
		0x0E,0x30,0x01,0x01,0xCA,0xE4,0x5B,0x07,0x36,0xA4,
		0xDC,0xBB,0x0E,0x6E,0xB5,0x84,0x91,0xDA,0x31,0xDE,
		0x3B,0x65,0xCB,0xA0,0xA7,0xC8,0x0C,0x78,0x15,0xF6, /* 1100 */

		0x77,0xEE,0xA7,0xCE,0x5E,0xC5,0x71,0xBB,0xF7,0xC4,
		0x51,0x6C,0x65,0xE4,0xB2,0x52,0x8E,0x16,0x09,0x59,
		0xC8,0x15,0xE1,0xBB,0xCA,0xD0,0xA2,0x9C,0xE5,0x95,
		0x73,0xD7,0x49,0x45,0x3D,0x9B,0x6B,0x52,0xA9,0x51,
		0xDE,0x9B,0x88,0x9C,0x79,0xCB,0x35,0x5C,0x34,0x56,
		0x31,0x07,0x0B,0x7C,0x30,0xF1,0x93,0x41,0x23,0xC9,
		0xEC,0x68,0xEB,0xB3,0x24,0xD2,0x54,0xE5,0xD2,0x46,
		0xB1,0x4A,0x02,0x59,0xE4,0x9B,0x2C,0xCE,0xC7,0x2B,
		0x8C,0x49,0x48,0x34),
	.pub_r = chunk_from_chars(
		0xC1,0xC4,0xD5,0x86,0x6B,0x3F,0xDD,0xBF,0x2D,0xF4,
		0x55,0x3F,0xBF,0xE2,0xFB,0xD2,0x8D,0x11,0xDF,0x51,
		0x32,0xD6,0xE4,0x02,0x00,0x87,0xA5,0x2E,0x91,0x63,
		0xB7,0x6F,0xAA,0x22,0x84,0xFE,0x8B,0x82,0xD6,0x8F,
		0x31,0x70,0xE5,0x46,0x32,0xC7,0x7D,0x5F,0x4E,0xD4,
		0x2A,0x74,0x45,0x73,0x18,0x99,0xE6,0x79,0x3D,0x85,
		0xF1,0x66,0x80,0x44,0x48,0xCB,0x4B,0x03,0x16,0x08,
		0x6F,0x47,0x8B,0x06,0x4A,0x52,0x19,0xC0,0x62,0x5B,
		0x96,0x47,0xAF,0xB0,0x7B,0x66,0x61,0x21,0xC5,0xA9,
		0x7D,0x16,0xE2,0xAF,0x0B,0xA7,0xA3,0x67,0x72,0x43, /* 100 */

		0x1F,0x8B,0xFF,0xFD,0x08,0x6D,0x83,0x32,0xF2,0xCC,
		0xF1,0x75,0xF7,0x0A,0x96,0x9A,0x82,0x27,0x20,0xE7,
		0xFB,0x16,0x18,0xF2,0xA5,0x1D,0xE2,0xA2,0x5E,0xE3,
		0xCB,0xFB,0x21,0x16,0xBE,0x73,0x4E,0x8D,0x4B,0x8E,
		0xEC,0x54,0x9F,0xC7,0x41,0x60,0x52,0x2E,0x77,0x6B,
		0x8E,0xC1,0xB7,0x0B,0x8D,0x4F,0x69,0xFD,0x03,0x92,
		0x29,0x1D,0xAF,0xB0,0x61,0x49,0x14,0xA7,0xFD,0xAB,
		0xF8,0x19,0xAF,0xDF,0xDB,0xE0,0x88,0x8C,0xE3,0x46,
		0xCC,0x9A,0xFE,0x96,0x6D,0x1C,0xD7,0x68,0x77,0x91,
		0xB5,0xFE,0x13,0xE5,0x11,0x4F,0x67,0x0E,0x99,0x02, /* 200 */

		0x02,0x24,0xF7,0xBB,0x65,0x37,0x11,0x37,0x0C,0x99,
		0x17,0x26,0x0F,0x1C,0x8F,0x9C,0x74,0xFE,0x78,0xFB,
		0xF6,0x88,0x22,0xE8,0x06,0x54,0x19,0x87,0xBF,0xFE,
		0x14,0x86,0x73,0x1E,0x6B,0x78,0x41,0xFA,0x91,0xCE,
		0x97,0x3A,0x4A,0x35,0x80,0x72,0xF2,0x19,0xEA,0x3B,
		0x07,0x0C,0x4A,0xE3,0x51,0x43,0xE6,0xA3,0x88,0xFB,
		0x1D,0x0B,0x80,0xFC,0x95,0xA1,0x07,0x31,0x84,0xC8,
		0x79,0x8F,0x0E,0x2D,0x62,0x55,0x97,0xA4,0x59,0xF3,
		0x8E,0xAA,0xB3,0x78,0xB4,0xFC,0xC0,0x9A,0x38,0xB4,
		0xB0,0xAE,0x32,0xD4,0x17,0xDA,0x16,0xBE,0xC1,0xD2, /* 300 */

		0xE9,0x07,0xB1,0x2B,0x12,0xBB,0x30,0xD9,0xF9,0x5F,
		0x84,0x9B,0xF7,0x95,0x22,0x62,0x2C,0xD4,0x06,0x8F,
		0x7F,0xA2,0x06,0xD0,0x40,0x0D,0xCA,0x69,0xA4,0x04,
		0xD4,0x8A,0xB4,0xEC,0x46,0x9F,0xED,0xEF,0x37,0xEC,
		0xA0,0xB4,0x1B,0xBD,0x4C,0xAE,0x6E,0x8D,0xE1,0x06,
		0x63,0x94,0xE9,0x44,0x34,0x92,0x30,0x88,0x25,0x53,
		0xF6,0x5B,0xB9,0xB2,0x9B,0x74,0xA1,0x21,0xEF,0x7F,
		0x1C,0x68,0x19,0x95,0xEC,0x82,0x9C,0xAB,0xB8,0x90,
		0xE2,0x26,0xDE,0xFB,0xA9,0xDE,0x92,0x97,0xF9,0x84,
		0x1D,0x00,0x67,0xBC,0x18,0xA9,0x31,0x31,0xDA,0x91, /* 400 */

		0x74,0x58,0x39,0xF3,0xF0,0x11,0xA1,0xEB,0x96,0xA2,
		0xDF,0x3F,0x78,0x59,0xF6,0x75,0xD9,0x1F,0x97,0x44,
		0x6D,0xDD,0xE2,0xE9,0x6B,0x07,0x3B,0x3B,0x01,0x6B,
		0x52,0x6D,0x9C,0x4B,0x5F,0xE2,0xC8,0xEF,0x98,0x6D,
		0x12,0x17,0x3C,0x6B,0xFB,0xC4,0xB5,0x28,0xE7,0xB9,
		0x0F,0x23,0x20,0x25,0xF8,0xAF,0x4C,0xFA,0xA0,0x29,
		0xA9,0xE8,0x1E,0xF1,0xC2,0xE0,0x12,0xD6,0x38,0x7B,
		0xE2,0xFB,0xDE,0xD4,0x05,0x90,0x27,0x05,0xDE,0xCC,
		0xD0,0x73,0xFC,0xBC,0x4D,0x76,0xB5,0x5A,0x7A,0x26,
		0xB2,0xBF,0xC4,0x88,0xBA,0xFE,0x2C,0xF5,0x95,0x67, /* 500 */

		0x88,0x39,0xE0,0x35,0xF5,0x31,0xD0,0xBA,0x36,0x5E,
		0xE4,0xD9,0x8D,0x55,0x24,0x76,0xEE,0x10,0x71,0x25,
		0x8D,0xEC,0xA1,0x25,0xB9,0x2C,0x0A,0x04,0x81,0xBD,
		0xEE,0x6D,0x48,0xF5,0x02,0xF5,0x77,0x97,0x8B,0x1F,
		0xC8,0x18,0xD6,0xCC,0xC3,0xB0,0x50,0x4C,0x1B,0x3B,
		0xB5,0xEA,0x34,0xD6,0x45,0x5E,0xE6,0x54,0x35,0xC9,
		0xA1,0x02,0x04,0xB9,0xCF,0x04,0x1B,0x7F,0xD0,0x15,
		0xF4,0xFA,0xE3,0x46,0x26,0x9B,0x7D,0x2E,0x7D,0x6F,
		0x1F,0xF6,0x26,0x1C,0x43,0xC1,0x15,0x09,0x28,0x7F,
		0x3F,0x64,0x4B,0x14,0xCD,0xDE,0xEA,0x5A,0xC9,0xF6, /* 600 */

		0x59,0x86,0x7F,0x47,0xEA,0xAD,0x6F,0xE3,0x6E,0xA5,
		0x8B,0x4A,0xD2,0xB2,0xF7,0xF4,0x7E,0x99,0xA3,0x23,
		0x9D,0x74,0x05,0x06,0x07,0xE1,0xBE,0xC9,0xAF,0xAE,
		0x12,0x21,0xD9,0xD0,0xE0,0xF6,0x34,0xAD,0x69,0xEB,
		0x61,0x8D,0xF4,0xE6,0x7A,0xEA,0x43,0x9D,0xAD,0x2C,
		0x3B,0x63,0x87,0x5B,0x71,0x65,0x13,0xC8,0x85,0xD1,
		0xFB,0x76,0xD9,0x99,0x44,0x81,0x39,0x7F,0xEA,0x2D,
		0x5B,0xC8,0xA5,0x63,0x64,0xD4,0xA8,0x4A,0xD7,0x6B,
		0x69,0x87,0x63,0x2D,0xA9,0x97,0xB3,0x2D,0x11,0xF0,
		0x88,0xCB,0x42,0x44,0x11,0x1C,0xAB,0x00,0x81,0x22, /* 700 */

		0xE6,0x3C,0x7D,0x4A,0x50,0xB1,0x8F,0x8D,0x46,0xAF,
		0x7C,0xC6,0x27,0xDD,0x9C,0x03,0x67,0x94,0x8F,0xE4,
		0x80,0x4C,0x48,0x2A,0x56,0x32,0xAF,0x94,0xB0,0xD0,
		0xAE,0x3F,0x05,0x23,0xE1,0x9E,0x3B,0x1A,0x8E,0xC7,
		0xFD,0xE1,0xED,0x66,0xE0,0xA5,0x0D,0x17,0xC1,0x95,
		0x26,0x91,0x9C,0xCC,0x81,0xD5,0x93,0xB1,0xF6,0x5E,
		0xC8,0x69,0xDE,0xE8,0xF2,0x56,0xDA,0xB0,0x83,0x06,
		0xB8,0x5C,0xD3,0xB4,0x06,0x2D,0xF7,0x25,0x32,0x55,
		0x02,0xF3,0xDF,0xD0,0xF1,0xF1,0x9B,0xE6,0xFD,0xDC,
		0x94,0x01,0xB8,0xC6,0x59,0x4F,0xE1,0x00,0xF7,0x77, /* 800 */

		0x71,0xF7,0x63,0xBE,0x0C,0x97,0x26,0xB8,0xF2,0xC1,
		0x38,0x98,0x1F,0x91,0xF0,0x00,0xAD,0xB4,0x08,0x9E,
		0x17,0x0D,0x35,0x59,0x67,0x89,0xA4,0x3B,0x7A,0xBB,
		0xFA,0x62,0xFA,0xD1,0x21,0x50,0x75,0x92,0xD6,0x4E,
		0x04,0xEE,0x85,0xC9,0xC2,0x73,0x22,0xD8,0xA6,0x7D,
		0x38,0x1E,0x55,0x8D,0x6F,0x9D,0x7B,0x75,0x22,0x0C,
		0x40,0x0A,0xF4,0xE1,0xE1,0x54,0x24,0x50,0x35,0xD4,
		0xAA,0xF8,0xD8,0x74,0xAF,0x51,0xD5,0xA0,0x15,0x3D,
		0xC0,0xED,0xAB,0x33,0xD9,0xB1,0x47,0x02,0xE2,0x42,
		0xEA,0xDB,0x8D,0x57,0x81,0x89,0x4D,0xD5,0x09,0xCC, /* 900 */

		0xE0,0xC7,0xCD,0xA6,0xC5,0xEA,0x8F,0xAF,0x28,0x41,
		0xF1,0x60,0x9A,0x25,0xB0,0x69,0x9A,0x34,0x95,0x50,
		0x70,0x37,0x39,0xCA,0x2F,0x45,0x81,0x06,0x6C,0x80,
		0xFA,0x80,0xAD,0x0A,0x62,0xBE,0x3E,0x7F,0x84,0x6C,
		0x30,0xD3,0x96,0x7A,0xF9,0x33,0x3F,0x8F,0xC3,0xA8,
		0xD6,0xF6,0xE4,0x11,0xED,0xA8,0xEB,0xDD,0x0D,0x7F,
		0x7D,0x00,0x76,0xF9,0x93,0x39,0x85,0xD8,0xF6,0xB3,
		0xF9,0x7F,0x60,0x5F,0xF6,0x5A,0x50,0xCA,0xA2,0xC7,
		0x95,0xB2,0xE4,0xA7,0x1C,0xE5,0x4B,0x50,0xBC,0x67,
		0x68,0x24,0x9F,0xBE,0xD1,0xAE,0x45,0xB2,0x01,0xE8, /* 1000 */

		0xD0,0x5A,0xC8,0xC1,0x10,0xDB,0x40,0x69,0x29,0x33,
		0x79,0xDD,0x9F,0xB2,0x97,0xE6,0x93,0x05,0x54,0xE8,
		0x0A,0x1D,0xB2,0x4E,0x08,0xDB,0xBB,0x03,0xAE,0xCF,
		0x3E,0x30,0x8F,0x4A,0x6F,0x0E,0x9C,0xF1,0x26,0xBA,
		0xE8,0x88,0xD0,0xBC,0x3F,0x30,0xBC,0x25,0x90,0xDF,
		0x64,0x2B,0xAB,0x34,0x6C,0xC2,0x20,0xAC,0x31,0x01,
		0x63,0x1A,0x36,0x07,0xE3,0xBE,0xDC,0x25,0x7C,0xD8,
		0xD4,0x03,0x56,0x7C,0x7D,0xC0,0x71,0x93,0xBE,0x40,
		0x44,0x0F,0x94,0xE2,0xB6,0xE2,0xCD,0xA2),
	.shared = chunk_from_chars(
		0xDF,0x7F,0xEF,0xB0,0xFE,0x72,0x98,0x0E,0x15,0x3B,
		0x5E,0x99,0xA9,0x66,0x91,0x5B,0xD4,0xC2,0x22,0x6B,
		0x24,0x90,0xC6,0x55,0x10,0x9A,0x07,0x23,0xAB,0x8A,
		0xAD,0xE0),
};

ke_test_vector_t mlkem768_3 = {
	.method = ML_KEM_768,
	.seed = chunk_from_chars(
		0x22,0x5D,0x5C,0xE2,0xCE,0xAC,0x61,0x93,0x0A,0x07,
		0x50,0x3F,0xB5,0x9F,0x7C,0x2F,0x93,0x6A,0x3E,0x07,
		0x54,0x81,0xDA,0x3C,0xA2,0x99,0xA8,0x0F,0x8C,0x5D,
		0xF9,0x22,0x3A,0x07,0x3E,0x7B,0x90,0xE0,0x2E,0xBF,
		0x98,0xCA,0x22,0x27,0xEB,0xA3,0x8C,0x1A),
	.pub_i = chunk_from_chars(
		0x25,0xC1,0x21,0x5D,0x62,0x41,0x56,0x82,0x80,0x88,
		0xB6,0x3D,0xA6,0x22,0xAF,0xA9,0xFC,0x54,0x76,0xB3,
		0x60,0x34,0x74,0xB2,0x4D,0x83,0x94,0xB9,0x3A,0xB4,
		0x94,0xEA,0x3E,0xB0,0xF3,0x3F,0xF5,0xD9,0x1A,0x33,
		0xD3,0x4A,0x2F,0x66,0x62,0x35,0xEA,0x00,0x9E,0x35,
		0x54,0x4C,0xF1,0x35,0xEF,0xF5,0x4E,0x35,0x75,0xAF,
		0xE2,0xFB,0x15,0xB6,0xB5,0x9A,0x4D,0xB8,0x43,0xC2,
		0x51,0x39,0xED,0x34,0x5F,0xA1,0x70,0x00,0x3F,0xDA,
		0x60,0xB2,0x40,0x06,0x5F,0x76,0x15,0x70,0x47,0x1C,
		0x49,0x14,0xAE,0xFB,0x01,0xC3,0x8F,0xE3,0x6B,0x3C, /* 100 */

		0x7B,0x80,0xF7,0x78,0x87,0xA0,0x3B,0x74,0xBA,0xFB,
		0x64,0xD3,0xB9,0x22,0x70,0x1A,0x86,0xB0,0x27,0x65,
		0x1F,0xDB,0xC7,0x77,0xBC,0x4D,0x90,0xD3,0x87,0xFC,
		0x15,0xB8,0xC1,0x24,0x43,0xAF,0x86,0x36,0xE3,0x52,
		0x24,0xC0,0x43,0x05,0x20,0xBC,0x98,0x4D,0x71,0xC5,
		0x6D,0xF2,0x6E,0x84,0xB6,0x83,0xC1,0xDB,0xA0,0x81,
		0xB0,0xA5,0x84,0xB5,0x79,0x3E,0x72,0x57,0x75,0x48,
		0x2E,0xC4,0x5B,0x1D,0xB7,0xC8,0x49,0x49,0x64,0x4D,
		0x40,0x55,0x79,0x65,0x9C,0xC3,0x1D,0xB1,0x30,0xBD,
		0x19,0x0D,0x22,0x47,0x21,0x0C,0x04,0x3B,0xC8,0xF2, /* 200 */

		0x27,0x45,0x0A,0xC2,0x06,0xC1,0x3E,0x8B,0xAA,0x89,
		0x6E,0x47,0x1F,0x7C,0xAB,0x2B,0xDA,0xB2,0x92,0x20,
		0xC0,0x51,0xBA,0x35,0x4D,0xC4,0x04,0x76,0x7E,0x81,
		0x27,0x65,0xC6,0x6E,0xC1,0x9A,0xCB,0xA3,0xB9,0x93,
		0xCA,0xA2,0xC0,0xBC,0x23,0x67,0xDD,0x50,0x84,0x6B,
		0xB4,0x0D,0x65,0x87,0xB5,0x1A,0x76,0x27,0x7F,0x68,
		0x65,0xA5,0xC6,0x9D,0x88,0x55,0x53,0xDD,0xFC,0x20,
		0x0E,0x97,0x0B,0x9C,0x06,0x66,0xC1,0x85,0x4E,0xFC,
		0x93,0x2A,0xF5,0x7A,0x1F,0x3B,0xC0,0xA8,0x05,0x02,
		0x4E,0xFC,0xF9,0x78,0xF7,0x7B,0x1D,0xD7,0x39,0x60, /* 300 */

		0x4B,0x8A,0x87,0xB4,0x4B,0x52,0x2E,0xA8,0x0E,0x87,
		0xCA,0x2C,0xBA,0x72,0x20,0xE6,0x2C,0x44,0x1A,0x42,
		0x9D,0x19,0xB7,0x64,0x25,0xD5,0x1C,0xD1,0xE6,0x1E,
		0x2A,0x29,0x50,0x92,0x26,0x37,0xA1,0xAB,0x84,0xFB,
		0xF6,0x45,0xA1,0xF6,0x41,0x3C,0x28,0x8B,0x73,0x19,
		0x45,0xD4,0x02,0x99,0x58,0xD5,0x6D,0xB9,0x0A,0xA0,
		0x1B,0x71,0x2C,0x83,0xB7,0x3A,0xA8,0x85,0x81,0x52,
		0x27,0x57,0xD5,0x05,0xB8,0x3E,0x61,0xAE,0x06,0xEB,
		0x6F,0x52,0x70,0x02,0x1C,0x61,0xB7,0xDA,0xBC,0xB0,
		0x22,0x98,0x78,0xD8,0x5C,0x8F,0x35,0x2A,0x9A,0x43, /* 400 */

		0x49,0xBD,0xEC,0xB2,0xAB,0x0C,0x2A,0xB4,0x03,0x99,
		0x40,0xA8,0x87,0xB6,0x65,0xF8,0xB9,0xDD,0x10,0x99,
		0x2B,0x3B,0x04,0x93,0xDB,0x60,0x74,0xFA,0x85,0xE1,
		0x38,0x2A,0x97,0x09,0x53,0xEB,0xF1,0x13,0xD4,0x83,
		0x5E,0x9F,0x03,0x1D,0x39,0x08,0x36,0x73,0x2B,0x3E,
		0xA6,0xA8,0x91,0x45,0x18,0x3F,0x72,0xDB,0xB0,0x95,
		0xD7,0x61,0x7A,0x3A,0xBD,0xC2,0x89,0x12,0x97,0x20,
		0x35,0xF1,0xCC,0x30,0xAE,0xD3,0x10,0x1D,0x91,0xC9,
		0xEB,0x39,0x76,0x80,0xC8,0xAE,0x22,0x2A,0xA6,0x1E,
		0x51,0x5A,0x93,0x96,0xCE,0xFC,0xC7,0x5C,0x0C,0xC2, /* 500 */

		0x55,0x53,0x5A,0x3F,0x41,0x21,0x21,0x41,0xC1,0x25,
		0x58,0x88,0x1E,0x23,0x79,0x8E,0x17,0x48,0xBB,0xB5,
		0x46,0x4A,0xDF,0xB1,0x67,0xA2,0xC4,0x43,0x88,0x04,
		0x47,0x54,0xA8,0x7A,0x5A,0xD1,0xCE,0x75,0x2C,0x38,
		0xB5,0x04,0xBC,0x44,0x3A,0x24,0xFF,0x73,0xA6,0xCE,
		0xA4,0x76,0x8E,0x22,0x68,0x15,0x54,0xB5,0x6D,0xD5,
		0x9D,0xA1,0xC7,0xB8,0x8D,0xB1,0x26,0xB7,0xEC,0x64,
		0x0C,0x1B,0xCA,0xE8,0xA2,0x0B,0x5F,0x86,0x4E,0xA6,
		0x38,0x5D,0xBB,0xA1,0xC4,0xDD,0xD3,0x17,0x75,0xCA,
		0x9F,0x72,0xEC,0x6F,0x13,0x75,0x70,0xFA,0x93,0x56, /* 600 */

		0xDE,0x55,0xC5,0xC2,0xEC,0x92,0x04,0x8A,0x0C,0xD9,
		0x68,0x08,0x37,0xF9,0x87,0x63,0xFB,0x35,0xD1,0x43,
		0x18,0x05,0xA7,0x42,0x0D,0x2B,0xB8,0x03,0xCC,0x37,
		0xBD,0xC9,0x0F,0xA9,0x21,0xBD,0x4D,0x18,0xB4,0xF3,
		0x34,0x19,0xCF,0x03,0xAD,0x0E,0x23,0x60,0xD9,0x21,
		0x92,0xD7,0xD8,0x7D,0xF7,0x1B,0xAE,0x83,0xD2,0xB5,
		0x38,0x74,0x19,0x14,0x89,0x2C,0x0D,0x77,0x8C,0xFD,
		0xF4,0x29,0xBE,0x26,0x8A,0x22,0xF4,0x04,0x3C,0x29,
		0xB6,0x2F,0xF7,0x4F,0xA7,0x70,0x15,0x6D,0x93,0x12,
		0x7C,0x3A,0x7F,0xE2,0xC6,0xC6,0xC6,0x54,0x70,0xD1, /* 700 */

		0x93,0x1F,0x85,0xA3,0xAF,0x1A,0x96,0x3B,0xD9,0x19,
		0xCA,0x3C,0x43,0x7B,0x43,0xD3,0x68,0xE6,0xE7,0xA1,
		0x99,0x40,0x72,0x58,0xB8,0xBD,0xE8,0x8B,0x47,0x42,
		0x19,0x26,0x73,0x49,0x9A,0xC4,0x40,0xB2,0x47,0xD9,
		0x72,0x25,0xE7,0x55,0x44,0x82,0xC6,0xAA,0x65,0x67,
		0x48,0x60,0x74,0xF1,0x5B,0x19,0x58,0x8C,0x6D,0x9D,
		0xE4,0x61,0x9D,0x4C,0x49,0x3B,0xF7,0x3F,0x2B,0xDB,
		0xC0,0x53,0x43,0xA5,0x81,0xE9,0x51,0x97,0xB6,0x4D,
		0xCD,0x88,0x26,0x44,0x25,0x65,0xBF,0x62,0x9D,0x5C,
		0xE4,0x04,0x52,0x18,0xBB,0x90,0x87,0x01,0xA5,0xE5, /* 800 */

		0x3B,0x2D,0x06,0x79,0x72,0x05,0x98,0xFA,0x1B,0x08,
		0xF2,0x86,0xB6,0x0E,0x5C,0x37,0x7C,0xA7,0xAF,0xAF,
		0xB2,0x06,0xED,0x47,0x12,0xD6,0xD1,0x6A,0x47,0xB9,
		0x93,0x60,0x42,0x66,0x19,0xF9,0x7D,0xB0,0x05,0x9A,
		0x15,0x59,0x23,0x53,0x57,0x77,0x65,0xA8,0x36,0x8D,
		0x93,0x92,0x88,0x02,0x70,0x48,0x36,0xC2,0x27,0x92,
		0x05,0x44,0x6B,0xBF,0x7A,0x24,0x65,0x04,0x28,0x16,
		0xD7,0x90,0x7F,0x14,0x36,0x35,0xB1,0x43,0x46,0x80,
		0x15,0x65,0x23,0xB0,0x2C,0xF1,0xD8,0x56,0x5C,0xA5,
		0xC7,0xCE,0xAA,0x12,0x6A,0x27,0x53,0xDC,0x29,0x9C, /* 900 */

		0x10,0xE4,0x96,0x8A,0x47,0x29,0x99,0x1A,0x0F,0x5C,
		0xE3,0x1C,0x59,0x56,0x29,0x30,0x1C,0x13,0x5B,0x58,
		0x10,0xA7,0x12,0x91,0x0A,0x84,0xCE,0x1F,0x21,0x64,
		0xC3,0xAB,0xA2,0x48,0xCB,0x89,0xC5,0x91,0xCA,0x0D,
		0x24,0xA7,0x68,0x1C,0x0A,0x14,0x06,0x72,0xBF,0x01,
		0xC7,0xF5,0xC9,0x67,0xF3,0xE4,0x1B,0x9A,0x92,0x5A,
		0xF6,0xEA,0x75,0x40,0x8A,0x24,0x68,0x4B,0x72,0x5B,
		0x44,0x6A,0x30,0xD0,0x19,0x9F,0x67,0xAF,0x94,0xB9,
		0x58,0xC4,0xB9,0x79,0xC2,0xA8,0x41,0xB2,0x88,0x78,
		0xB5,0xDC,0x51,0x54,0xB7,0x9C,0x94,0x4C,0x71,0x56, /* 1000 */

		0x53,0x8E,0x6A,0x13,0x03,0x79,0x67,0x4B,0xBA,0x54,
		0x38,0xD6,0x47,0x75,0xFE,0x1A,0x87,0xD3,0xAC,0xA6,
		0x77,0x54,0x1B,0xF8,0x2C,0x27,0xC6,0xD1,0x8F,0x3B,
		0xB1,0x82,0xBC,0x74,0x94,0xDC,0x91,0xB2,0x3C,0x35,
		0x39,0x2F,0x21,0xAC,0x49,0xE6,0x9E,0x3C,0x70,0x45,
		0x4B,0xDB,0x16,0xB2,0x39,0x4D,0x00,0xA5,0x49,0x73,
		0x4A,0x1A,0xA4,0x79,0x0B,0x01,0x50,0x8B,0xE6,0x70,
		0x84,0x1E,0x09,0x04,0x2E,0x5A,0x1A,0x4A,0x73,0xC3,
		0xB8,0x30,0xAC,0x5A,0x35,0xB8,0x26,0xF3,0x64,0x11,
		0x63,0x9E,0x20,0x57,0x6A,0x89,0x19,0x74,0x26,0x00, /* 1100 */

		0x5C,0x08,0x38,0x0C,0x13,0xA4,0xA4,0x64,0x89,0x2B,
		0xC6,0x99,0x2E,0x23,0x67,0x71,0xE7,0xB1,0x35,0xAE,
		0xE0,0x96,0xE9,0xF6,0x57,0x38,0xC5,0x31,0xB8,0xE6,
		0x3C,0x3F,0x4A,0x92,0x2A,0x79,0x22,0x2C,0xF8,0xB8,
		0x30,0x9A,0x04,0xC3,0xC7,0x46,0x4A,0x69,0x98,0x2A,
		0x07,0x2F,0xB8,0x95,0x71,0x4E,0xB8,0xF4,0xE6,0xD0,
		0xA4,0xC4,0x47,0xAC,0x97,0x28,0x0A,0x7A,0x7B,0x63,
		0xBD,0x58,0x5C,0x52,0x1A,0x49,0xBF,0x92,0x30,0x58,
		0x8D,0xDD,0x70,0x34),
	.pub_r = chunk_from_chars(
		0x29,0xB0,0xDE,0x2C,0x9B,0xDE,0x97,0xA5,0x00,0x7F,
		0x60,0x20,0xF6,0xF2,0xDA,0xEE,0x3A,0x00,0x22,0x57,
		0x36,0x6C,0xCD,0xEE,0xEB,0x9A,0x55,0x65,0x52,0x48,
		0x0A,0x83,0xD4,0x49,0x27,0x27,0x5E,0x85,0x5C,0x9D,
		0x31,0xFA,0xB8,0x89,0xD1,0xE3,0x4A,0x1E,0xBD,0x8A,
		0x6B,0xAA,0xA7,0x2B,0x68,0x16,0xA7,0x61,0x61,0x88,
		0xCB,0x9E,0x99,0xE9,0x23,0xFA,0x53,0x78,0x2A,0x42,
		0x3A,0xEA,0x1B,0x78,0x91,0xAC,0xF5,0x1A,0x25,0x2E,
		0x25,0x21,0x4D,0x16,0xBC,0xAA,0x13,0xF8,0x22,0xCE,
		0x23,0xFE,0x19,0xE9,0x4D,0xC9,0xDF,0x70,0x23,0x99, /* 100 */

		0xFE,0xDD,0xF6,0x30,0x93,0xC8,0x58,0x5B,0x3A,0x74,
		0x90,0x42,0x97,0xDD,0x77,0xBE,0x8D,0x5E,0x15,0x9F,
		0x51,0xD2,0x27,0x6B,0x26,0x9A,0x9B,0x94,0xE2,0x01,
		0x8A,0x54,0xE1,0x13,0xA9,0xEE,0xBB,0xF5,0xA7,0xF0,
		0x32,0xB9,0x27,0x26,0xCB,0x02,0x89,0x64,0x66,0xAE,
		0xBD,0x62,0x6C,0x64,0x43,0x34,0x42,0x43,0xF0,0x5B,
		0x0C,0x1B,0x0D,0x68,0x42,0x1C,0xB7,0xE7,0x1C,0x42,
		0xFA,0x1C,0x98,0x91,0x69,0x56,0xE1,0xC0,0x31,0x46,
		0xD0,0xD5,0x86,0xE5,0x4F,0x19,0xDD,0xF8,0x26,0x51,
		0x73,0x95,0x4D,0xA6,0x50,0xB8,0x4E,0x27,0x53,0xF6, /* 200 */

		0x3E,0x69,0xE1,0x54,0x9B,0x0C,0x54,0x8A,0xE1,0x04,
		0x33,0x6D,0x75,0x68,0xEF,0xBC,0xD2,0xCC,0x5E,0x64,
		0x86,0xA5,0x1C,0xD4,0x25,0x80,0x89,0xE7,0x21,0x56,
		0x34,0x59,0x2D,0x22,0xDF,0xF9,0xD4,0x48,0xD0,0x06,
		0x1C,0x78,0xE1,0xEA,0x87,0xF2,0x23,0x12,0x97,0x62,
		0x71,0x2E,0xDB,0xBF,0x03,0x62,0x5E,0xB4,0xB1,0x89,
		0x1C,0x51,0x51,0xD4,0xE8,0x78,0x71,0x38,0xF7,0x39,
		0xB5,0x3B,0x8D,0xF2,0xFE,0x75,0x9C,0x38,0xDC,0x2A,
		0xE8,0xE5,0x62,0x95,0x29,0xA1,0x78,0xEC,0xD3,0x52,
		0x25,0x3A,0x2E,0xC2,0xE4,0xF5,0x8F,0x0C,0xDE,0x48, /* 300 */

		0x9E,0x5D,0x1A,0x79,0x0E,0x82,0x37,0xF9,0x7B,0x1E,
		0xC4,0xD7,0xE5,0xFF,0x65,0xF2,0x29,0xBD,0x2C,0xEC,
		0x4B,0x59,0xAE,0x4E,0xDB,0x38,0xE6,0x5E,0xEA,0x8D,
		0x86,0x4B,0x6D,0xD6,0x45,0xDB,0x85,0x52,0x8D,0xC3,
		0x16,0xD9,0xE5,0x57,0x21,0xB2,0x5D,0xAA,0x68,0xA5,
		0xA8,0x4B,0x14,0x60,0xF3,0x03,0xE5,0x7E,0xEC,0xCE,
		0xF9,0x4D,0xC3,0x70,0x81,0x65,0xC8,0x0D,0xA1,0xAA,
		0xCD,0x8E,0x2A,0x6A,0x69,0xE5,0xA3,0x73,0xE5,0x48,
		0xB8,0xF4,0x6A,0xB4,0x12,0x18,0xF8,0x49,0x7D,0xD4,
		0x6F,0x37,0x41,0x50,0x74,0x7F,0x27,0x68,0x5A,0x79, /* 400 */

		0x85,0x9D,0xFC,0x16,0xFD,0x79,0x4C,0x9B,0x27,0x54,
		0x8D,0xE8,0x3A,0x76,0x47,0x2C,0x15,0xBB,0xFC,0xD3,
		0xB0,0x7F,0xF6,0xA5,0xBF,0xBD,0x32,0x99,0xD9,0xD5,
		0xD1,0x3E,0xF8,0x3C,0x1B,0x0D,0xD9,0x76,0x1C,0xA5,
		0x75,0x08,0x6F,0xF2,0x98,0x05,0xC3,0xE3,0xFB,0xEB,
		0x98,0x17,0x41,0x55,0xC1,0x01,0x4F,0xC9,0x24,0x92,
		0x15,0xD7,0xCD,0x71,0xBC,0x41,0x00,0xE1,0xA2,0xDA,
		0xB4,0xAB,0xA2,0x1B,0x9C,0x26,0xC5,0x6C,0x34,0x59,
		0x4A,0x66,0xEF,0xB3,0xFC,0xD0,0xFB,0x99,0x4E,0xDD,
		0x99,0xAD,0xD8,0xCB,0xC7,0x85,0xC4,0xFC,0xCE,0xF1, /* 500 */

		0xFC,0x16,0x5F,0xA5,0x44,0x98,0x66,0xF0,0xB1,0xA4,
		0xF6,0x6B,0x2E,0x7F,0xF6,0x7E,0x7A,0x73,0x13,0xF4,
		0x19,0x68,0x1A,0x38,0xA0,0x3E,0xA9,0x17,0x3D,0x02,
		0xA2,0x1E,0xEF,0x66,0x7A,0xE5,0xA1,0xC7,0xED,0xF3,
		0xE4,0xC7,0xDB,0x3B,0xCD,0x94,0xE2,0xA5,0x60,0x93,
		0xB0,0x59,0xEE,0xC2,0xD3,0x97,0x46,0x80,0xA2,0xF3,
		0x7C,0xC1,0xDF,0x22,0x7B,0xED,0x2D,0xAE,0x85,0xA1,
		0xE6,0x24,0x04,0x09,0xF2,0xF0,0xFD,0x69,0xA9,0xA4,
		0x5F,0x16,0x24,0xB6,0x9C,0xF0,0x72,0x27,0x7A,0x55,
		0xFD,0x44,0x9F,0x88,0xBE,0x3C,0x58,0x8F,0x9F,0xE1, /* 600 */

		0xFA,0xC6,0xB1,0xA1,0x00,0xAD,0x64,0x9D,0xD1,0x41,
		0xAE,0xAE,0x3A,0xB8,0xDE,0x5B,0x5C,0xF4,0x49,0x1B,
		0xC6,0x7F,0x21,0x09,0x29,0xFD,0x08,0x1E,0xF2,0xCD,
		0x74,0x8D,0x20,0xFA,0xEB,0x0C,0xF1,0xC0,0xD8,0xDA,
		0x61,0x1A,0x24,0x5F,0x98,0x5A,0xDF,0xDB,0x6A,0x07,
		0xED,0x99,0x51,0xC0,0xB7,0x35,0xE3,0x67,0x74,0xF9,
		0xBF,0xAC,0xD3,0xC8,0x5E,0xEA,0x16,0xD5,0xD6,0xB4,
		0x71,0x35,0x8E,0xA6,0x12,0x78,0xC6,0xC5,0xE3,0xFE,
		0x3C,0x3E,0x91,0xC8,0xEA,0x89,0xB2,0xC8,0x70,0x02,
		0x52,0x33,0xCA,0x94,0x16,0x55,0x9B,0x44,0xEA,0x2A, /* 700 */

		0x00,0xE3,0x65,0x94,0x47,0xC1,0xE4,0xF4,0x80,0xB3,
		0xDA,0x8E,0xC2,0x6F,0xDB,0x6B,0xAA,0xE4,0x7B,0x3B,
		0xC4,0x60,0xD6,0x7B,0x63,0x20,0x5A,0x51,0xF1,0x7A,
		0x79,0x0F,0xF1,0xBD,0x19,0x16,0xB7,0x1B,0x27,0x60,
		0x38,0xA7,0xD7,0x9C,0x7F,0x90,0x3B,0x42,0x7F,0x3D,
		0x53,0xB5,0x61,0x77,0xD4,0xAA,0x50,0x85,0x09,0xEA,
		0x4B,0x6F,0x0F,0xF4,0x69,0x34,0x9B,0xF1,0x6A,0xF4,
		0xCD,0x60,0xE5,0x90,0x3D,0x37,0x77,0x44,0x61,0x44,
		0x62,0x45,0x0C,0xA2,0xBF,0x6B,0xD2,0x5A,0xE8,0xB1,
		0xC4,0x27,0xB3,0xA3,0x41,0x44,0x07,0xE4,0xFA,0xA1, /* 800 */

		0xB8,0xAE,0x7D,0x84,0x69,0xB4,0x1F,0xCA,0x5C,0xCD,
		0x29,0xBE,0xA9,0x17,0xF7,0x1E,0x01,0xDF,0x4C,0x00,
		0x27,0x5E,0xAB,0x51,0x80,0xE6,0xEC,0x9A,0x45,0x5D,
		0x34,0x1B,0xD8,0xCD,0x90,0x56,0x9D,0xB8,0x92,0x9A,
		0x62,0x99,0xAC,0x39,0x1C,0xB5,0x02,0xA3,0x5E,0x76,
		0x55,0x9B,0x99,0xCE,0x4F,0x2C,0x8A,0x7F,0x76,0x72,
		0xCB,0x73,0xDB,0xD4,0x81,0xE0,0xDF,0x77,0x67,0x83,
		0x5A,0x18,0xC6,0x51,0xC3,0xD2,0x97,0xD7,0xD4,0xE6,
		0x5A,0x76,0x27,0xC9,0xA7,0x84,0x87,0x48,0xAB,0x55,
		0x52,0x38,0xF8,0xB6,0x2F,0x61,0xCD,0xC4,0x26,0x92, /* 900 */

		0x86,0x98,0x41,0x0E,0x80,0x59,0xD3,0xC4,0x52,0x21,
		0x25,0xAC,0xD5,0x75,0xD0,0x65,0x28,0xC4,0x49,0x12,
		0xF3,0x1B,0xE4,0x9A,0xB5,0xD2,0x98,0xF7,0x1A,0xC6,
		0x4E,0xE3,0xD9,0x6C,0xDC,0x6B,0x7B,0xBF,0x83,0x46,
		0x94,0xEE,0x81,0x5B,0x3F,0xA5,0xF3,0xE1,0xCB,0x76,
		0x0B,0xFA,0x26,0x38,0x89,0x67,0x2E,0x27,0x85,0xF1,
		0x62,0x52,0xD3,0x54,0x5F,0x51,0xEB,0x1C,0x9D,0xF2,
		0x32,0xE5,0x83,0xA5,0x56,0xC5,0xEB,0x79,0x38,0xFC,
		0xA6,0x95,0xA5,0x11,0x7E,0x97,0x98,0x2A,0xB8,0x15,
		0xBA,0xD0,0x59,0x91,0x8B,0xBB,0xDE,0x8B,0xB2,0x77, /* 1000 */

		0x91,0xC0,0x8A,0x66,0x9D,0x07,0x22,0x77,0x95,0x24,
		0x8E,0xDF,0x99,0xAD,0xE2,0x43,0x18,0x73,0x5A,0x1C,
		0x97,0x7B,0x7C,0xEC,0xAC,0x2E,0x3B,0x94,0x25,0xED,
		0xCE,0x53,0x44,0xAD,0x52,0xBD,0x3B,0x94,0xF5,0x50,
		0x7E,0x21,0x31,0xE9,0x8F,0x3E,0x58,0x74,0x31,0x3D,
		0x6C,0x1A,0xB0,0xF9,0x85,0x56,0xCC,0x9A,0x3B,0x17,
		0x90,0x22,0xF1,0x15,0x7D,0x7B,0x11,0x7C,0x3E,0x70,
		0xE4,0x11,0xDF,0x1B,0xFE,0xB0,0xBA,0x6B,0x7D,0xF2,
		0x34,0x35,0x73,0x75,0x7B,0x1C,0xF4,0x2A),
	.shared = chunk_from_chars(
		0xDA,0x72,0x3F,0x95,0xD4,0x3D,0x70,0xE6,0x7C,0x85,
		0x8A,0x10,0xFB,0x7C,0x55,0x40,0xF9,0x14,0x06,0xAE,
		0xAA,0xE8,0x06,0xAB,0xC0,0xED,0xF4,0x8D,0x24,0xFD,
		0x94,0x7F),
};

/**
 * First four test vectors of the liboqs-0.11.0 ML-KEM-1024 KAT
 */
ke_test_vector_t mlkem1024_0 = {
	.method = ML_KEM_1024,
	.seed = chunk_from_chars(
		0x06,0x15,0x50,0x23,0x4D,0x15,0x8C,0x5E,0xC9,0x55,
		0x95,0xFE,0x04,0xEF,0x7A,0x25,0x76,0x7F,0x2E,0x24,
		0xCC,0x2B,0xC4,0x79,0xD0,0x9D,0x86,0xDC,0x9A,0xBC,
		0xFD,0xE7,0x05,0x6A,0x8C,0x26,0x6F,0x9E,0xF9,0x7E,
		0xD0,0x85,0x41,0xDB,0xD2,0xE1,0xFF,0xA1),
	.pub_i = chunk_from_chars(
		0x53,0x79,0x11,0x95,0x7C,0x12,0x51,0x48,0xA8,0x7F,
		0x41,0x58,0x9C,0xB2,0x22,0xD0,0xD1,0x92,0x29,0xE2,
		0xCB,0x55,0xE1,0xA0,0x44,0x79,0x1E,0x7C,0xA6,0x11,
		0x92,0xA4,0x64,0x60,0xC3,0x18,0x3D,0x2B,0xCD,0x6D,
		0xE0,0x8A,0x5E,0x76,0x51,0x60,0x3A,0xCC,0x34,0x9C,
		0xA1,0x6C,0xBA,0x18,0xAB,0xB2,0x3A,0x3E,0x8C,0x33,
		0x0D,0x74,0x21,0x59,0x8A,0x62,0x78,0xEC,0x7E,0xBF,
		0xAB,0xCA,0x0E,0xF4,0x88,0xB2,0x29,0x05,0x54,0x75,
		0x34,0x99,0xC0,0x45,0x2E,0x45,0x38,0x15,0x30,0x99,
		0x55,0xB8,0x15,0x0F,0xA1,0xA1,0xE3,0x93,0x38,0x6D, /* 100 */

		0xC1,0x2F,0xDB,0x27,0xB3,0x8C,0x67,0x45,0xF2,0x94,
		0x40,0x16,0xEC,0x45,0x7F,0x39,0xB1,0x8D,0x60,0x4A,
		0x07,0xA1,0xAB,0xE0,0x7B,0xC8,0x44,0x05,0x0F,0xFA,
		0x8A,0x06,0xFA,0x15,0x4A,0x49,0xD8,0x8F,0xAC,0x77,
		0x54,0x52,0xD6,0xA7,0xC0,0xE5,0x89,0xBF,0xB5,0xC3,
		0x70,0xC2,0xC4,0xB6,0x20,0x1D,0xDA,0x80,0xC9,0xAB,
		0x20,0x76,0xEC,0xC0,0x8B,0x44,0x52,0x2F,0xDA,0x33,
		0x26,0xF0,0x33,0x80,0x6D,0xD2,0x69,0x3F,0x31,0x97,
		0x39,0xF4,0x0C,0x4F,0x42,0xB2,0x4A,0xCA,0x70,0x98,
		0xFB,0x8F,0xF5,0xF9,0xAC,0x20,0x29,0x2D,0x02,0xB5, /* 200 */

		0x6A,0xC7,0x46,0x80,0x1A,0xCC,0xCC,0x84,0x86,0x3D,
		0xEE,0x32,0x87,0x84,0x97,0xB6,0x94,0x38,0xBF,0x99,
		0x17,0x76,0x28,0x66,0x50,0x48,0x2C,0x8D,0x9D,0x95,
		0x87,0xBC,0x6A,0x55,0xB8,0x5C,0x4D,0x7F,0xA7,0x4D,
		0x02,0x65,0x6B,0x42,0x1C,0x9E,0x23,0xE0,0x3A,0x48,
		0xD4,0xB7,0x44,0x25,0xC2,0x6E,0x4A,0x20,0xDD,0x95,
		0x62,0xA4,0xDA,0x07,0x93,0xF3,0xA3,0x52,0xCC,0xC0,
		0xF1,0x82,0x17,0xD8,0x68,0xC7,0xF5,0x00,0x2A,0xBE,
		0x76,0x8B,0x1F,0xC7,0x3F,0x05,0x74,0x4E,0x7C,0xC2,
		0x8F,0x10,0x34,0x40,0x62,0xC1,0x0E,0x08,0xEC,0xCC, /* 300 */

		0xED,0x3C,0x1F,0x7D,0x39,0x2C,0x01,0xD9,0x79,0xDD,
		0x71,0x8D,0x83,0x98,0x37,0x46,0x65,0xA1,0x6A,0x98,
		0x70,0x58,0x5C,0x39,0xD5,0x58,0x9A,0x50,0xE1,0x33,
		0x38,0x9C,0x9B,0x9A,0x27,0x6C,0x02,0x42,0x60,0xD9,
		0xFC,0x77,0x11,0xC8,0x1B,0x63,0x37,0xB5,0x7D,0xA3,
		0xC3,0x76,0xD0,0xCD,0x74,0xE1,0x4C,0x73,0x72,0x7B,
		0x27,0x66,0x56,0xB9,0xD8,0xA4,0xEB,0x71,0x89,0x6F,
		0xF5,0x89,0xD4,0xB8,0x93,0xE7,0x11,0x0F,0x3B,0xB9,
		0x48,0xEC,0xE2,0x91,0xDD,0x86,0xC0,0xB7,0x46,0x8A,
		0x67,0x8C,0x74,0x69,0x80,0xC1,0x2A,0xA6,0xB9,0x5E, /* 400 */

		0x2B,0x0C,0xBE,0x43,0x31,0xBB,0x24,0xA3,0x3A,0x27,
		0x01,0x53,0xAA,0x47,0x2C,0x47,0x31,0x23,0x82,0xCA,
		0x36,0x5C,0x5F,0x35,0x25,0x9D,0x02,0x57,0x46,0xFC,
		0x65,0x95,0xFE,0x63,0x6C,0x76,0x75,0x10,0xA6,0x9C,
		0x1E,0x8A,0x17,0x6B,0x79,0x49,0x95,0x8F,0x26,0x97,
		0x39,0x94,0x97,0xA2,0xFC,0x73,0x64,0xA1,0x2C,0x81,
		0x98,0x29,0x52,0x39,0xC8,0x26,0xCB,0x50,0x82,0x08,
		0x60,0x77,0x28,0x2E,0xD6,0x28,0x65,0x1F,0xC0,0x4C,
		0x63,0x9B,0x43,0x85,0x22,0xA9,0xDE,0x30,0x9B,0x14,
		0xB0,0x86,0xD6,0xE9,0x23,0xC5,0x51,0x62,0x3B,0xD7, /* 500 */

		0x2A,0x73,0x3C,0xB0,0xDA,0xBC,0x54,0xA9,0x41,0x6A,
		0x99,0xE7,0x2C,0x9F,0xDA,0x1C,0xB3,0xFB,0x9B,0xA0,
		0x6B,0x8A,0xDB,0x24,0x22,0xD6,0x8C,0xAD,0xC5,0x53,
		0xC9,0x82,0x02,0xA1,0x76,0x56,0x47,0x8A,0xC0,0x44,
		0xEF,0x34,0x56,0x37,0x8A,0xBC,0xE9,0x99,0x1E,0x01,
		0x41,0xBA,0x79,0x09,0x4F,0xA8,0xF7,0x7A,0x30,0x08,
		0x05,0xD2,0xD3,0x2F,0xFC,0x62,0xBF,0x0C,0xA4,0x55,
		0x4C,0x33,0x0C,0x2B,0xB7,0x04,0x2D,0xB3,0x51,0x02,
		0xF6,0x8B,0x1A,0x00,0x62,0x58,0x38,0x65,0x38,0x1C,
		0x74,0xDD,0x91,0x3A,0xF7,0x0B,0x26,0xCF,0x09,0x23, /* 600 */

		0xD0,0xC4,0xCB,0x97,0x16,0x92,0x22,0x25,0x52,0xA8,
		0xF4,0xB7,0x88,0xB4,0xAF,0xD1,0x34,0x1A,0x9D,0xF4,
		0x15,0xCF,0x20,0x39,0x00,0xF5,0xCC,0xF7,0xF6,0x59,
		0x88,0x94,0x9A,0x75,0x58,0x0D,0x04,0x96,0x39,0x85,
		0x31,0x00,0x85,0x4B,0x21,0xF4,0x01,0x80,0x03,0x50,
		0x2B,0xB1,0xBA,0x95,0xF5,0x56,0xA5,0xD6,0x7C,0x7E,
		0xB5,0x24,0x10,0xEB,0xA2,0x88,0xA6,0xD0,0x63,0x5C,
		0xA8,0xA4,0xF6,0xD6,0x96,0xD0,0xA0,0x20,0xC8,0x26,
		0x93,0x8D,0x34,0x94,0x3C,0x38,0x08,0xC7,0x9C,0xC0,
		0x07,0x76,0x85,0x33,0x21,0x6B,0xC1,0xB2,0x9D,0xA6, /* 700 */

		0xC8,0x12,0xEF,0xF3,0x34,0x0B,0xAA,0x8D,0x2E,0x65,
		0x34,0x4F,0x09,0xBD,0x47,0x89,0x4F,0x5A,0x3A,0x41,
		0x18,0x71,0x5B,0x3C,0x50,0x20,0x67,0x93,0x27,0xF9,
		0x18,0x9F,0x7E,0x10,0x85,0x6B,0x23,0x8B,0xB9,0xB0,
		0xAB,0x4C,0xA8,0x5A,0xBF,0x4B,0x21,0xF5,0xC7,0x6B,
		0xCC,0xD7,0x18,0x50,0xB2,0x2E,0x04,0x59,0x28,0x27,
		0x6A,0x0F,0x2E,0x95,0x1D,0xB0,0x70,0x7C,0x6A,0x11,
		0x6D,0xC1,0x91,0x13,0xFA,0x76,0x2D,0xC5,0xF2,0x0B,
		0xD5,0xD2,0xAB,0x5B,0xE7,0x17,0x44,0xDC,0x9C,0xBD,
		0xB5,0x1E,0xA7,0x57,0x96,0x3A,0xAC,0x56,0xA9,0x0A, /* 800 */

		0x0D,0x80,0x23,0xBE,0xD1,0xF5,0xCA,0xE8,0xA6,0x4D,
		0xA0,0x47,0x27,0x9B,0x35,0x3A,0x09,0x6A,0x83,0x5B,
		0x0B,0x2B,0x02,0x3B,0x6A,0xA0,0x48,0x98,0x92,0x33,
		0x07,0x9A,0xEB,0x46,0x7E,0x52,0x2F,0xA2,0x7A,0x58,
		0x22,0x92,0x1E,0x5C,0x55,0x1B,0x4F,0x53,0x75,0x36,
		0xE4,0x6F,0x3A,0x6A,0x97,0xE7,0x2C,0x3B,0x06,0x31,
		0x04,0xE0,0x9A,0x04,0x05,0x98,0x94,0x0D,0x87,0x2F,
		0x6D,0x87,0x1F,0x5E,0xF9,0xB4,0x35,0x50,0x73,0xB5,
		0x47,0x69,0xE4,0x54,0x54,0xE6,0xA0,0x81,0x95,0x99,
		0x40,0x86,0x21,0xAB,0x44,0x13,0xB3,0x55,0x07,0xB0, /* 900 */

		0xDF,0x57,0x8C,0xE2,0xD5,0x11,0xD5,0x20,0x58,0xD5,
		0x74,0x9D,0xF3,0x8B,0x29,0xD6,0xCC,0x58,0x87,0x0C,
		0xAF,0x92,0xF6,0x9A,0x75,0x16,0x14,0x06,0xE7,0x1C,
		0x5F,0xF9,0x24,0x51,0xA7,0x75,0x22,0xB8,0xB2,0x96,
		0x7A,0x2D,0x58,0xA4,0x9A,0x81,0x66,0x1A,0xA6,0x5A,
		0xC0,0x9B,0x08,0xC9,0xFE,0x45,0xAB,0xC3,0x85,0x1F,
		0x99,0xC7,0x30,0xC4,0x50,0x03,0xAC,0xA2,0xBF,0x0F,
		0x84,0x24,0xA1,0x9B,0x74,0x08,0xA5,0x37,0xD5,0x41,
		0xC1,0x6F,0x56,0x82,0xBF,0xE3,0xA7,0xFA,0xEA,0x56,
		0x4F,0x12,0x98,0x61,0x1A,0x7F,0x5F,0x60,0x92,0x2B, /* 1000 */

		0xA1,0x9D,0xE7,0x3B,0x19,0x17,0xF1,0x85,0x32,0x73,
		0x55,0x51,0x99,0xA6,0x49,0x31,0x8B,0x50,0x77,0x33,
		0x45,0xC9,0x97,0x46,0x08,0x56,0x97,0x2A,0xCB,0x43,
		0xFC,0x81,0xAB,0x63,0x21,0xB1,0xC3,0x3C,0x2B,0xB5,
		0x09,0x8B,0xD4,0x89,0xD6,0x96,0xA0,0xF7,0x06,0x79,
		0xC1,0x21,0x38,0x73,0xD0,0x8B,0xDA,0xD4,0x28,0x44,
		0x92,0x72,0x16,0x04,0x72,0x05,0x63,0x32,0x12,0x31,
		0x0E,0xE9,0xA0,0x6C,0xB1,0x00,0x16,0xC8,0x05,0x50,
		0x3C,0x34,0x1A,0x36,0xD8,0x7E,0x56,0x07,0x2E,0xAB,
		0xE2,0x37,0x31,0xE3,0x4A,0xF7,0xE2,0x32,0x8F,0x85, /* 1100 */

		0xCD,0xB3,0x70,0xCC,0xAF,0x00,0x51,0x5B,0x64,0xC9,
		0xC5,0x4B,0xC8,0x37,0x57,0x84,0x47,0xAA,0xCF,0xAE,
		0xD5,0x96,0x9A,0xA3,0x51,0xE7,0xDA,0x4E,0xFA,0x7B,
		0x11,0x5C,0x4C,0x51,0xF4,0xA6,0x99,0x77,0x98,0x50,
		0x29,0x5C,0xA7,0x2D,0x78,0x1A,0xD4,0x1B,0xC6,0x80,
		0x53,0x2B,0x89,0xE7,0x10,0xE2,0x18,0x9E,0xB3,0xC5,
		0x08,0x17,0xBA,0x25,0x5C,0x74,0x74,0xC9,0x5C,0xA9,
		0x11,0x0C,0xC4,0x3B,0x8B,0xA8,0xE6,0x82,0xC7,0xFB,
		0x7B,0x0F,0xDC,0x26,0x5C,0x04,0x83,0xA6,0x5C,0xA4,
		0x51,0x4E,0xE4,0xB8,0x32,0xAA,0xC5,0x80,0x0C,0x3B, /* 1200 */

		0x08,0xE7,0x4F,0x56,0x39,0x51,0xC1,0xFB,0xB2,0x10,
		0x35,0x3E,0xFA,0x1A,0xA8,0x66,0x85,0x6B,0xC1,0xE0,
		0x34,0x73,0x3B,0x04,0x85,0xDA,0xB1,0xD0,0x20,0xC6,
		0xBF,0x76,0x5F,0xF6,0x0B,0x3B,0x80,0x19,0x84,0xA9,
		0x0C,0x2F,0xE9,0x70,0xBF,0x1D,0xE9,0x70,0x04,0xA6,
		0xCF,0x44,0xB4,0x98,0x4A,0xB5,0x82,0x58,0xB4,0xAF,
		0x71,0x22,0x1C,0xD1,0x75,0x30,0xA7,0x00,0xC3,0x29,
		0x59,0xC9,0x43,0x63,0x44,0xB5,0x31,0x6F,0x09,0xCC,
		0xCA,0x70,0x29,0xA2,0x30,0xD6,0x39,0xDC,0xB0,0x22,
		0xD8,0xBA,0x79,0xBA,0x91,0xCD,0x6A,0xB1,0x2A,0xE1, /* 1300 */

		0x57,0x9C,0x50,0xC7,0xBB,0x10,0xE3,0x03,0x01,0xA6,
		0x5C,0xAE,0x31,0x01,0xD4,0x0C,0x7B,0xA9,0x27,0xBB,
		0x55,0x31,0x48,0xD1,0x64,0x70,0x24,0xD4,0xA0,0x6C,
		0x81,0x66,0xD0,0xB0,0xB8,0x12,0x69,0xB7,0xD5,0xF4,
		0xB3,0x4F,0xB0,0x22,0xF6,0x91,0x52,0xF5,0x14,0x00,
		0x4A,0x7C,0x68,0x53,0x68,0x55,0x23,0x43,0xBB,0x60,
		0x36,0x0F,0xBB,0x99,0x45,0xED,0xF4,0x46,0xD3,0x45,
		0xBD,0xCA,0xA7,0x45,0x5C,0x74,0xBA,0x0A,0x55,0x1E,
		0x18,0x46,0x20,0xFE,0xF9,0x76,0x88,0x77,0x3D,0x50,
		0xB6,0x43,0x3C,0xA7,0xA7,0xAC,0x5C,0xB6,0xB7,0xF6, /* 1400 */

		0x71,0xA1,0x53,0x76,0xE5,0xA6,0x74,0x7A,0x62,0x3F,
		0xA7,0xBC,0x66,0x30,0x37,0x3F,0x5B,0x1B,0x51,0x26,
		0x90,0xA6,0x61,0x37,0x78,0x70,0xA6,0x0A,0x7A,0x18,
		0x96,0x83,0xF9,0xB0,0xCF,0x04,0x66,0xE1,0xF7,0x50,
		0x76,0x26,0x31,0xC4,0xAB,0x09,0xF5,0x05,0xC4,0x2D,
		0xD2,0x86,0x33,0x56,0x94,0x72,0x73,0x54,0x42,0x85,
		0x1E,0x32,0x16,0x16,0xD4,0x00,0x98,0x10,0x77,0x7B,
		0x6B,0xD4,0x6F,0xA7,0x22,0x44,0x61,0xA5,0xCC,0x27,
		0x40,0x5D,0xFB,0xAC,0x0D,0x39,0xB0,0x02,0xCA,0xB3,
		0x34,0x33,0xF2,0xA8,0x6E,0xB8,0xCE,0x91,0xC1,0x34, /* 1500 */

		0xA6,0x38,0x6F,0x86,0x0A,0x19,0x94,0xEB,0x4B,0x68,
		0x75,0xA4,0x6D,0x19,0x55,0x81,0xD1,0x73,0x85,0x4B,
		0x53,0xD2,0x29,0x3D,0xF3,0xE9,0xA8,0x22,0x75,0x6C,
		0xD8,0xF2,0x12,0xB3,0x25,0xCA,0x29,0xB4,0xF9,0xF8,
		0xCF,0xBA,0xDF,0x2E,0x41,0x86,0x9A,0xBF,0xBA,0xD1,
		0x07,0x38,0xAD,0x04,0xCC,0x75,0x2B,0xC2,0x0C,0x39,
		0x47,0x46,0x85,0x0E,0x0C,0x48,0x47,0xDB),
	.pub_r = chunk_from_chars(
		0x3C,0xA7,0xA7,0x83,0x8B,0x26,0xFF,0x0E,0x59,0x8F,
		0x1D,0x4C,0xD6,0x51,0x6F,0xD8,0xD2,0x8B,0x7C,0x3A,
		0x61,0x60,0x72,0x04,0xC7,0xFD,0xB3,0x90,0x09,0xD0,
		0x49,0x11,0xC1,0x1F,0x91,0x87,0xDB,0x0E,0x6D,0xC0,
		0x5D,0xED,0xEA,0x64,0x62,0xAA,0x00,0xFF,0x67,0xBB,
		0x40,0x28,0x5C,0xAC,0x75,0x01,0xFD,0x5B,0x7D,0x9E,
		0x2C,0xFC,0x8B,0x91,0x77,0xA1,0x26,0xB6,0x25,0x67,
		0xCF,0xF1,0xF6,0x65,0xEE,0x05,0x70,0x54,0x95,0x01,
		0x7C,0x5C,0x40,0x00,0x1F,0x6A,0x7A,0xBC,0x47,0xD3,
		0x4F,0x36,0xD1,0x83,0xB6,0x24,0xE4,0xBB,0x75,0xF9, /* 100 */

		0x26,0x00,0xF2,0xEE,0xA7,0x0A,0x20,0x52,0xFF,0xEA,
		0x79,0x19,0x87,0x1E,0xC2,0x7F,0x96,0x0E,0x9E,0xED,
		0x46,0xBC,0xF8,0x71,0x3C,0x39,0x6C,0x6F,0x2F,0x3C,
		0xBD,0x0B,0x1E,0xB6,0xCF,0x13,0x6A,0x97,0xFF,0x24,
		0x35,0xF1,0xB3,0x10,0xDB,0x70,0x20,0x6F,0x52,0xB2,
		0x68,0xBB,0x76,0x84,0x07,0xA2,0x7B,0x31,0xED,0xC8,
		0xDE,0x55,0xEF,0x53,0x19,0x2D,0xE1,0x30,0x4D,0x15,
		0xE6,0xC5,0x52,0x3E,0x5B,0x1B,0xB9,0x6D,0x9F,0x28,
		0x8D,0xDB,0x9E,0xD6,0x5E,0x8E,0x32,0x70,0x1D,0x38,
		0x58,0x83,0x2D,0x9D,0x7C,0xDE,0x72,0xE7,0x16,0x56, /* 200 */

		0x5F,0x57,0x88,0x03,0x5A,0x08,0x71,0x21,0xF6,0x0B,
		0xCD,0xC7,0x2A,0xA3,0x86,0xA1,0xBC,0xDA,0x97,0x8E,
		0x15,0xF4,0xAA,0x73,0x6B,0xD2,0xF8,0x84,0xE0,0xA6,
		0x67,0x75,0xD6,0xE2,0xCE,0x78,0xB7,0x3D,0x10,0x92,
		0x67,0xC4,0x80,0x80,0x39,0x6E,0x22,0xB4,0x2D,0x4D,
		0xD0,0xC2,0xD8,0xE7,0x50,0x65,0xF1,0xEE,0xAE,0x86,
		0x54,0xFF,0x9C,0x25,0x94,0x65,0xFF,0x4A,0xB2,0xC2,
		0x53,0xF5,0x17,0xB8,0xDB,0x48,0x1C,0xD6,0xC0,0x0D,
		0x57,0x3D,0x0B,0x46,0xCC,0x7E,0x46,0x44,0xDF,0xE0,
		0xE1,0xE2,0xF9,0x97,0xB5,0x86,0xB2,0xCC,0xCC,0x75, /* 300 */

		0xF5,0x2D,0xEC,0x78,0x8A,0xC2,0x14,0xF8,0x01,0x37,
		0x82,0x20,0x60,0x23,0xEB,0xE8,0xF7,0x22,0x89,0x66,
		0x4E,0xC6,0x37,0xA6,0xF9,0x88,0xBE,0xE8,0xCB,0xA2,
		0x52,0x26,0xAE,0xAB,0xBF,0x09,0x17,0x9E,0x41,0xE7,
		0xE6,0x16,0x8E,0x30,0x81,0x9A,0xF0,0xA8,0x9A,0x74,
		0x56,0x71,0xF3,0xC5,0xA1,0x05,0x60,0xDB,0x93,0xD5,
		0xED,0xAF,0x63,0xAC,0x75,0x39,0xA8,0x61,0x6D,0x84,
		0xB3,0x7B,0xF4,0x24,0x5B,0x09,0xE5,0x24,0x8D,0x7E,
		0x40,0x42,0xC4,0xC0,0xD5,0x58,0x55,0x04,0xBB,0x82,
		0x5E,0x0E,0xD0,0x5E,0xC0,0x8F,0x3F,0x3B,0x43,0x65, /* 400 */

		0x61,0x12,0x99,0xF8,0xF5,0xD2,0xBE,0x0B,0x2E,0x45,
		0x38,0xA2,0xBD,0x39,0x38,0x19,0x4D,0x7F,0x5C,0x79,
		0xFF,0x8A,0xF9,0x62,0x2B,0x33,0x6D,0xBC,0x31,0xBC,
		0xB7,0xD6,0x0F,0x6A,0x19,0x08,0x83,0xF4,0x98,0xB4,
		0xDE,0x46,0x88,0xB3,0xF5,0x47,0x5A,0x20,0x04,0x93,
		0xD9,0xC8,0x21,0x8F,0x85,0x25,0x6E,0x87,0xCC,0xB3,
		0x01,0x37,0x23,0xC7,0xA2,0x64,0xA3,0xAB,0x76,0x4A,
		0xCF,0xC8,0xDC,0xFE,0x2C,0xE3,0x59,0xA7,0xBD,0x53,
		0xFF,0x7F,0x68,0xA8,0x42,0x75,0x1C,0xFC,0xBD,0xD3,
		0x9E,0xBB,0x1C,0x44,0xF6,0x57,0x39,0xC0,0x71,0xAF, /* 500 */

		0x6C,0x24,0x22,0x05,0x75,0x29,0x6E,0x85,0xAE,0x44,
		0x58,0xE0,0xF6,0x76,0x34,0x8A,0x2D,0x7C,0xDB,0x64,
		0x37,0x8E,0x75,0xAB,0xC7,0x4E,0x86,0xD3,0x46,0x48,
		0x09,0x49,0xC6,0xA6,0xD3,0x62,0x76,0x11,0xCB,0x2C,
		0xDF,0x77,0x14,0xEC,0x77,0x9B,0xF0,0x8A,0x47,0x61,
		0x67,0x60,0xA4,0x9F,0x30,0xA1,0xA7,0x12,0x49,0x3F,
		0x19,0x0D,0x43,0x3E,0x78,0x28,0xCA,0xB4,0x50,0x37,
		0xCE,0xF6,0xC8,0x63,0xC5,0xA3,0xCA,0xCD,0x5C,0x14,
		0xBC,0x2E,0x36,0xBB,0x32,0x96,0xE1,0x87,0xE1,0x79,
		0x6D,0x27,0x38,0xB3,0x28,0x35,0x62,0x0E,0x14,0x2E, /* 600 */

		0x1A,0xDF,0x09,0x6B,0xF3,0x09,0xCD,0xC8,0x3A,0x15,
		0x45,0xE4,0x14,0x23,0x2B,0x99,0x05,0xBB,0xAB,0x1F,
		0xD2,0x3B,0xAB,0x0D,0x93,0xD4,0xDF,0x0A,0xD8,0x31,
		0x48,0x09,0xFE,0xDF,0x7A,0x97,0x99,0x28,0x71,0xF5,
		0xBD,0x23,0x8F,0xD7,0x2F,0xF5,0x2B,0xAD,0x31,0xF9,
		0x34,0xEE,0x54,0x35,0xE0,0xE3,0x2A,0x2B,0xE4,0x51,
		0x9E,0x2B,0xA6,0x70,0xBB,0x44,0x3A,0x0E,0x22,0x7A,
		0x3D,0x44,0xB4,0x00,0xCC,0x48,0xA3,0x14,0x75,0xA9,
		0x82,0x33,0x93,0x0C,0xAB,0x93,0x0B,0xF4,0x98,0xCC,
		0x30,0xCC,0xB0,0x42,0x1D,0xC4,0x12,0xC3,0x19,0x0A, /* 700 */

		0x98,0xA0,0x3D,0x8E,0xCE,0x18,0x16,0xDE,0xF4,0xD3,
		0x58,0xE3,0xC3,0xC1,0xD4,0x24,0x92,0x64,0x25,0x2D,
		0x02,0xDE,0x48,0xE2,0xB2,0x21,0x6A,0x2B,0x24,0x6A,
		0x61,0xEA,0xC8,0xDE,0xC3,0xF8,0xA4,0xBD,0xC2,0xA6,
		0x90,0x85,0xDF,0xEC,0x3A,0xD5,0xC8,0x08,0x52,0xB0,
		0x96,0x0D,0xA2,0x01,0x82,0x8E,0xF0,0x4A,0xAE,0x93,
		0xE0,0x26,0xDE,0xB3,0xC6,0x70,0x49,0x04,0x80,0x1B,
		0x41,0x01,0xAA,0xE3,0x05,0x01,0x35,0x81,0x29,0x4F,
		0x1F,0x34,0x87,0x73,0x76,0xA1,0x8D,0xB8,0xEE,0xC0,
		0xF1,0xBF,0xE9,0xB5,0x96,0xBE,0x86,0x38,0x32,0x44, /* 800 */

		0xF9,0x99,0xE2,0x98,0xAC,0xB0,0x42,0x87,0xAC,0xE0,
		0x35,0xD5,0xB8,0x9B,0x2F,0x9D,0xA1,0xCB,0xDB,0x57,
		0x79,0x99,0x4E,0x57,0x33,0xDC,0xC4,0xDE,0x22,0xAD,
		0x2F,0x3F,0x9F,0x1E,0xF4,0x9E,0x5D,0xEB,0x80,0xEC,
		0x9A,0xC1,0x33,0xDF,0x3E,0x0B,0xB9,0xBE,0x7C,0xD8,
		0xDB,0x47,0x8E,0xB3,0x11,0x1B,0xB6,0xD7,0x16,0x84,
		0x9F,0x4A,0x48,0xCA,0x79,0x5B,0xD0,0x49,0xAC,0x1C,
		0x39,0x39,0x86,0x2B,0xF9,0x09,0x16,0x97,0xF0,0x54,
		0xEA,0x7D,0xDD,0xB4,0x22,0x45,0x79,0xBC,0x3E,0x3D,
		0x94,0x6E,0x7D,0x88,0x16,0x69,0xA2,0x90,0xC2,0x3A, /* 900 */

		0x4A,0x22,0x2B,0x3C,0x90,0x07,0x13,0x33,0xE4,0x04,
		0xBC,0x10,0x84,0x27,0x49,0xF7,0xD3,0x05,0x11,0x75,
		0xCC,0xC6,0x45,0x14,0x6A,0x99,0x1C,0xF0,0x7A,0xF0,
		0x50,0x20,0xA2,0xFD,0xF9,0x4F,0x47,0x4D,0xF6,0xBF,
		0xE2,0x47,0x90,0x04,0x74,0x97,0x25,0x79,0x30,0xFA,
		0xFC,0x7B,0x9A,0x56,0xD1,0xC0,0x6E,0xC9,0x58,0x59,
		0xB9,0x57,0x6A,0xCF,0x45,0xC9,0x8C,0xC8,0xD7,0x43,
		0x4C,0x03,0xC9,0x0B,0x01,0x17,0xDA,0x33,0x33,0x88,
		0xAA,0x87,0xDE,0x75,0xA5,0x44,0x1E,0xA6,0x3F,0x27,
		0x23,0x5B,0xE4,0xA7,0xD2,0x5B,0x1D,0x1A,0xFE,0x29, /* 1000 */

		0x1C,0x85,0x79,0x55,0xE3,0x8A,0xE3,0x9D,0x3C,0xF4,
		0x69,0xB2,0xFB,0xBE,0xB3,0x27,0xAA,0xE7,0xC5,0x6E,
		0x90,0x57,0xF5,0xCC,0x59,0x2B,0x37,0xD9,0xD9,0xD6,
		0x71,0xF7,0xC9,0x1C,0xCF,0xE0,0xC2,0x82,0xC3,0xF2,
		0x85,0xB6,0xCF,0x2F,0xDD,0x6F,0xB1,0x10,0xB8,0x98,
		0xCD,0xDC,0xBD,0x37,0x78,0x7C,0xE8,0xE9,0x54,0xB7,
		0x08,0x90,0xF0,0xDA,0xA9,0x0F,0x4A,0x57,0x91,0x36,
		0x0F,0x4F,0x1E,0x68,0xD6,0x4F,0xA7,0xA4,0x8A,0x0F,
		0xD2,0xE3,0x84,0x90,0x48,0x2E,0xFA,0x90,0xA8,0x69,
		0x53,0x5D,0x32,0x0C,0x6B,0x65,0x75,0xFE,0x8E,0x0E, /* 1100 */

		0x55,0x18,0xE9,0xDE,0x40,0xD9,0xF1,0x0B,0xF3,0x5F,
		0x49,0x9A,0x30,0xAD,0xB9,0x75,0xCF,0xAB,0xBA,0x56,
		0x8E,0x28,0xCA,0x3A,0xB8,0x1E,0x21,0x67,0x9E,0x67,
		0x68,0xEE,0x5C,0x89,0x0F,0x4C,0x5E,0x93,0x49,0xF0,
		0x23,0x37,0xED,0xFA,0xFE,0x4E,0xFE,0x10,0x23,0xE9,
		0x3C,0xE3,0x2E,0x84,0x03,0x9B,0xFC,0xA5,0x16,0xE8,
		0x69,0x86,0x64,0x9A,0xEE,0x3E,0x47,0x5B,0xE7,0xB3,
		0x64,0x47,0x74,0xDA,0xBB,0xB6,0xCA,0x8B,0x5C,0x77,
		0xB4,0xCA,0x18,0x65,0xD7,0x76,0xA1,0x00,0x43,0xCD,
		0xCB,0xC6,0x12,0xDF,0x0F,0x43,0x6A,0x82,0x52,0xDE, /* 1200 */

		0xA1,0x72,0xA9,0xF2,0x68,0x6B,0x03,0xFF,0x96,0xF9,
		0xB2,0x90,0x52,0x40,0xAF,0xA2,0x45,0x51,0x91,0x6F,
		0x8F,0x82,0x94,0x4F,0x3F,0xF7,0x17,0x3F,0x6E,0xA4,
		0x87,0x15,0x9E,0xC8,0x3C,0x29,0x0F,0xAE,0x14,0x40,
		0xCC,0x54,0xCE,0x6F,0x78,0x5B,0xBC,0x3C,0xB3,0x76,
		0x3F,0xA9,0x8C,0xF9,0x17,0xD3,0x3B,0xCA,0x00,0xF5,
		0x6D,0x90,0x49,0x16,0x68,0xA1,0xEE,0x89,0xD7,0x61,
		0x16,0xF7,0xF1,0x9E,0xDC,0xDF,0x71,0x60,0x24,0x0B,
		0x98,0x82,0x26,0x13,0x87,0xE1,0x90,0xB1,0x16,0xA9,
		0x5F,0x2D,0xDF,0xD2,0xD8,0xE5,0xB0,0x3D,0xCB,0x85, /* 1300 */

		0x0B,0x89,0x58,0x4E,0xE9,0xFF,0xEC,0xED,0xAB,0xAF,
		0x7F,0x15,0x03,0x0E,0x16,0x67,0x97,0x57,0xD4,0xA2,
		0x7E,0x20,0x14,0xA0,0x4D,0xD8,0xFE,0xEF,0xFF,0xCA,
		0x2E,0xC7,0x95,0xFA,0x9A,0x6F,0xD3,0x1A,0x3F,0x3A,
		0x74,0xE4,0x40,0xDA,0xE0,0x18,0xE5,0xF1,0x74,0x14,
		0x1E,0x20,0x0A,0xDE,0x65,0x6E,0x3C,0x96,0x89,0xBF,
		0xFA,0x04,0xD3,0xFE,0xAF,0xAA,0xB9,0x5D,0x7E,0xD5,
		0xD5,0xE7,0x54,0x61,0xB2,0x36,0x55,0x98,0x26,0x34,
		0x7D,0xAF,0x9E,0x8D,0xF8,0xFA,0x27,0x90,0x4B,0xAB,
		0x70,0x78,0x43,0x3E,0xD9,0xBD,0x45,0x57,0xC7,0x84, /* 1400 */

		0x68,0xDA,0x6B,0x74,0x7E,0x13,0xD6,0xE5,0xC9,0x7D,
		0x92,0x7A,0x5B,0x09,0x15,0x09,0x73,0x16,0xA6,0xFE,
		0x21,0xED,0xA2,0x6D,0x23,0x0C,0xF1,0x9B,0xA8,0x26,
		0x48,0x5C,0x93,0x36,0x58,0x14,0xBF,0x50,0xFC,0x8E,
		0x4A,0xF9,0x03,0x92,0xCB,0x0B,0xB7,0x95,0x70,0x21,
		0x6F,0xD9,0x54,0x32,0x71,0xB6,0x20,0x5D,0xC4,0x44,
		0x16,0xBB,0xF4,0x14,0x48,0x24,0x0D,0x28,0x3E,0x36,
		0x7C,0x4C,0xAB,0x21,0xF0,0x90,0x94,0x1D,0x2E,0x40,
		0x33,0xB7,0xCF,0x02,0xA3,0x45,0xAE,0x9E,0xE8,0x55,
		0x0A,0x4C,0xF1,0x97,0x75,0xCF,0xD4,0x40,0xE7,0x0C, /* 1500 */

		0xE8,0x38,0x55,0x2A,0x31,0x71,0x9D,0x87,0x89,0x40,
		0x1D,0x33,0xF0,0x1F,0x3E,0xE5,0x58,0xA9,0x92,0xD7,
		0x1F,0xD3,0x09,0xCC,0xBE,0x96,0x89,0xC4,0x8B,0x36,
		0x67,0x93,0x0F,0xD8,0xC7,0xDD,0xCA,0x71,0x7E,0x7C,
		0x77,0xDA,0xC2,0x73,0xD7,0xF4,0xCA,0x77,0x75,0x7A,
		0xCA,0x23,0xFF,0x2E,0x55,0x8C,0xEE,0xF1,0x52,0x07,
		0x5A,0xDD,0x70,0xBA,0xA7,0x63,0xC2,0x9F),
	.shared = chunk_from_chars(
		0xEA,0x63,0x6C,0xE3,0x1B,0x73,0xF4,0x02,0x29,0x57,
		0x21,0x46,0xB9,0x7E,0x59,0x0F,0x16,0x05,0xFD,0xAD,
		0xD1,0xC3,0x78,0x18,0x61,0x53,0x0E,0xFF,0xCF,0x2B,
		0x1E,0x18),
};

ke_test_vector_t mlkem1024_1 = {
	.method = ML_KEM_1024,
	.seed = chunk_from_chars(
		0xD8,0x1C,0x4D,0x8D,0x73,0x4F,0xCB,0xFB,0xEA,0xDE,
		0x3D,0x3F,0x8A,0x03,0x9F,0xAA,0x2A,0x2C,0x99,0x57,
		0xE8,0x35,0xAD,0x55,0xB2,0x2E,0x75,0xBF,0x57,0xBB,
		0x55,0x6A,0xC8,0x1A,0xDD,0xE6,0xAE,0xEB,0x4A,0x5A,
		0x87,0x5C,0x3B,0xFC,0xAD,0xFA,0x95,0x8F),
	.pub_i = chunk_from_chars(
		0x93,0x8A,0x45,0x43,0x64,0xCF,0x10,0xA4,0xC7,0x19,
		0x11,0x3A,0x23,0xB2,0x42,0xBC,0x01,0x39,0x62,0xF1,
		0x34,0x21,0xEC,0x06,0x86,0xE3,0x2C,0xCB,0x80,0x84,
		0x07,0x49,0x64,0x3E,0xB4,0xB5,0xCC,0x41,0x82,0xCE,
		0xE2,0x36,0x67,0x17,0xCF,0x77,0xF9,0x7D,0xA2,0x96,
		0xA1,0x85,0x44,0x01,0x13,0x77,0x0B,0x6F,0x75,0x5B,
		0xC5,0x96,0xCB,0xBC,0xE0,0x21,0xE9,0x43,0x06,0xB1,
		0xE4,0xAE,0x43,0x7A,0xB7,0xDB,0xC2,0x95,0x27,0x14,
		0x2A,0x92,0x23,0xA8,0xA7,0xCB,0x26,0x93,0x91,0xCA,
		0xD8,0xF7,0x0C,0x6C,0xF6,0x66,0x11,0x88,0x30,0x48, /* 100 */

		0x5C,0xCA,0xEB,0xBA,0x1B,0xDA,0x24,0x21,0x76,0x55,
		0x99,0x04,0x71,0x3C,0x67,0x22,0xC4,0x13,0xD5,0xC2,
		0xB9,0xC2,0x66,0x9D,0x58,0x13,0x8E,0xD5,0xB2,0x35,
		0x06,0xA7,0x1C,0x61,0x82,0xAF,0x69,0x39,0xDD,0x40,
		0xC5,0x1D,0x67,0x86,0x39,0x23,0x8C,0x8C,0x07,0x13,
		0x84,0xC1,0x25,0x6E,0xE6,0xA3,0x07,0x47,0x5B,0xB7,
		0x2B,0xC9,0x39,0x76,0xC8,0x97,0x6B,0xB8,0x9C,0x73,
		0x6D,0x8F,0xB1,0x95,0x80,0xCA,0xCB,0x1F,0x33,0x5B,
		0x97,0xD1,0x69,0x43,0xC0,0x70,0x6F,0x78,0x09,0x54,
		0x08,0xCD,0xAD,0x77,0xA9,0xFE,0x96,0x7C,0x4B,0xCA, /* 200 */

		0x73,0x89,0x60,0xB3,0xA3,0xC1,0x71,0x68,0x04,0x26,
		0x52,0x8A,0x21,0x57,0x8B,0x79,0xB2,0x26,0xBD,0xA1,
		0x9A,0x9B,0xF1,0x32,0x99,0x8A,0x21,0x49,0xC4,0x63,
		0x93,0x2D,0x63,0x0B,0x0F,0x13,0x1E,0x80,0x96,0xBE,
		0x46,0x20,0x80,0xE6,0xD1,0x71,0x07,0x70,0x4F,0x3B,
		0x0C,0xC9,0x0C,0x39,0x08,0xCD,0x64,0x52,0x68,0x37,
		0x50,0x56,0xE6,0x07,0x24,0x25,0xB5,0x1A,0x46,0xC2,
		0x7A,0x48,0xB9,0xC0,0xDB,0x67,0xB0,0x25,0xA5,0x8E,
		0xE3,0x7B,0x44,0x79,0x44,0x30,0xF3,0xAE,0x6C,0xC7,
		0x4E,0x16,0x73,0xC2,0xB1,0xD4,0xB2,0x9D,0xB5,0x37, /* 300 */

		0x0F,0x59,0x0E,0xAF,0x61,0x65,0xE5,0x80,0x63,0x76,
		0xF6,0xA7,0xD4,0x82,0x5C,0x74,0x03,0x87,0x92,0x68,
		0x3F,0x38,0x89,0x34,0x64,0x10,0x3D,0x82,0x42,0x65,
		0x93,0xF3,0x66,0xF9,0x6B,0x02,0x30,0x08,0xBE,0x21,
		0x95,0x6E,0x6B,0xE8,0xA4,0xBD,0x61,0x2D,0x4F,0x85,
		0x23,0x9D,0x62,0xAB,0x6C,0x39,0x06,0x9A,0xA2,0x10,
		0xB4,0x7B,0x37,0xCA,0x85,0x39,0x4C,0x19,0x32,0x7E,
		0x53,0x03,0xCC,0x9C,0x54,0xA5,0xC6,0x3A,0xEB,0x10,
		0x4D,0xAC,0xF6,0x7A,0x1C,0x1C,0x31,0x42,0x71,0x0D,
		0xCF,0x68,0x2E,0xDA,0xC8,0x24,0x71,0xBA,0x98,0xC8, /* 400 */

		0x1A,0x55,0x2A,0xD2,0x54,0x1B,0x73,0x38,0xB2,0xA0,
		0xCE,0x41,0x35,0x4C,0xCA,0xB8,0x79,0xF5,0x49,0xCF,
		0xA5,0x89,0x95,0xEB,0x8C,0x64,0xA8,0xB3,0x0B,0x6F,
		0x38,0x4E,0xC2,0xE0,0x4D,0xC1,0xA1,0x1A,0xE0,0x4C,
		0x96,0x7F,0x74,0x40,0x24,0xD7,0x0D,0xA7,0x80,0x51,
		0x67,0x77,0xA1,0xB1,0x71,0x5F,0x22,0xCB,0x1E,0xB3,
		0x51,0x1F,0x5C,0x4C,0x99,0xCE,0x57,0x78,0x36,0x48,
		0x9A,0x0F,0x81,0x3A,0x99,0xA6,0x22,0x39,0x0B,0x52,
		0xE4,0x75,0x78,0x43,0xC7,0x72,0xF6,0xE7,0x5F,0x47,
		0x03,0x05,0x25,0x98,0x33,0xB3,0x99,0x39,0x19,0xF3, /* 500 */

		0xAC,0xA6,0x65,0x63,0xF3,0xEB,0x73,0x75,0xF5,0x88,
		0x3C,0xFC,0xA7,0x23,0x98,0xB0,0x49,0x17,0x0E,0xBF,
		0x09,0x97,0xFF,0x27,0x7E,0x44,0x50,0x1F,0xB0,0xE8,
		0x3D,0x0C,0x5C,0xA6,0xC7,0x01,0x04,0x81,0x80,0xAC,
		0x68,0xD8,0x57,0xE0,0xDC,0x5D,0x24,0xB1,0x62,0x84,
		0x33,0xC7,0x37,0xB8,0x80,0x8C,0x29,0x08,0x39,0xD4,
		0x2C,0x6A,0x32,0x22,0x50,0xE0,0xC2,0x2B,0x81,0xB4,
		0xC0,0x18,0x26,0x0D,0x28,0x8C,0x47,0x97,0x6B,0xCB,
		0x84,0xBC,0xD0,0x85,0x6C,0x40,0x44,0x8C,0xA4,0x0B,
		0x72,0x3B,0x81,0x44,0x83,0x79,0x65,0xAD,0x46,0xB5, /* 600 */

		0x21,0xEC,0x4B,0x40,0x5C,0x28,0x58,0x9B,0x85,0x36,
		0x62,0x1A,0x10,0x67,0x8A,0x75,0x23,0x3E,0x2C,0xD5,
		0x9B,0xC6,0xE0,0x3F,0x49,0xD9,0x39,0x25,0xC5,0xBC,
		0xC1,0x33,0xBE,0xA3,0x9A,0x7E,0x21,0x90,0x7A,0x19,
		0x84,0x0C,0x80,0x22,0x5F,0x1F,0x15,0x4C,0x27,0x0A,
		0x4B,0xF0,0x20,0x6A,0x16,0xC2,0x41,0xBF,0x42,0xCB,
		0xD3,0x51,0x05,0x5A,0xDC,0x85,0x19,0xD3,0x8F,0xC9,
		0xD3,0xB9,0x76,0xE2,0x14,0x49,0x09,0x05,0x3C,0x58,
		0x43,0xC3,0x6B,0x7B,0x24,0x7A,0x6E,0xCD,0x04,0xC9,
		0xD7,0x92,0xA6,0xFA,0xE9,0x68,0xCB,0xE1,0x17,0x13, /* 700 */

		0x59,0x2B,0x28,0x7C,0xC8,0xB0,0x7B,0x73,0xC3,0x34,
		0xBD,0xBC,0xB9,0x1D,0x05,0xC7,0xAE,0x4E,0x86,0xAA,
		0x75,0x47,0xB4,0xCC,0x64,0x03,0x50,0x4A,0xCA,0xEF,
		0x4C,0x6E,0x5B,0x75,0x8F,0x4E,0x26,0x1C,0x76,0x24,
		0x33,0x70,0x00,0xC1,0x87,0x64,0x83,0x25,0xF3,0x6A,
		0xC8,0x85,0x4D,0x4A,0x8A,0x1C,0x2C,0x73,0x2A,0x08,
		0x11,0xB3,0x94,0x39,0x34,0x0E,0x23,0xC3,0x3A,0xF0,
		0xCE,0x50,0x90,0xCE,0x80,0x00,0x70,0x30,0x68,0x34,
		0x32,0x1B,0x5A,0xF9,0xE4,0xBE,0x13,0xB2,0x2E,0xA6,
		0x65,0x5D,0xA1,0x14,0x27,0xF9,0x95,0xA1,0x15,0xDC, /* 800 */

		0x86,0xCD,0xA7,0x37,0xAA,0x99,0x07,0x84,0x07,0xB9,
		0xF7,0x8B,0x91,0x4E,0x83,0x1A,0x6B,0xEC,0x2C,0x85,
		0xF9,0x22,0xF9,0x72,0xBD,0xDA,0x97,0x1B,0x66,0x76,
		0x10,0x45,0x81,0x3A,0x85,0x2A,0x0F,0xD9,0x90,0x13,
		0x68,0x40,0x83,0x5F,0x8B,0x5A,0x72,0xC5,0xC8,0xE8,
		0xA6,0x5C,0xE8,0xD3,0xCE,0x95,0x7B,0x06,0x69,0x82,
		0x61,0x16,0x00,0x31,0x59,0x7A,0x29,0xE3,0x38,0x8A,
		0x4F,0x5A,0x5F,0x0B,0x96,0xB7,0x04,0x81,0xA2,0xE5,
		0x73,0x8D,0x3A,0x50,0x18,0x94,0x96,0xC7,0x5A,0x85,
		0x66,0x43,0x89,0x99,0x47,0xC9,0x55,0xBE,0x88,0x71, /* 900 */

		0xE8,0x08,0x03,0x0A,0x57,0x57,0xD6,0xB4,0x60,0x79,
		0x45,0x4B,0x54,0xD5,0x15,0x57,0xCC,0xA9,0x64,0x37,
		0x9B,0xCA,0x5A,0x81,0x91,0x69,0x82,0x75,0xDC,0x51,
		0xCB,0x16,0x4F,0x8E,0x50,0x8F,0xDA,0xF4,0x80,0x03,
		0x1A,0x28,0xAF,0xB1,0xC9,0xCC,0xC6,0x5D,0xA9,0xAC,
		0x1C,0xDE,0xC6,0x25,0x13,0x1C,0x66,0x30,0x18,0x8F,
		0xFF,0xFB,0xCB,0x72,0x61,0x33,0xCA,0x56,0x3D,0xA6,
		0x40,0xC7,0x6A,0x6C,0x24,0x10,0x55,0x02,0xEA,0x3C,
		0x24,0xB9,0x18,0xC5,0xB8,0x18,0x12,0x1C,0xC6,0x8E,
		0x2F,0x1B,0x62,0x11,0x2A,0x41,0x59,0x4A,0x63,0x59, /* 1000 */

		0x87,0x6B,0x91,0xD0,0x58,0xFD,0x66,0x86,0x1B,0x76,
		0x33,0xD7,0x3C,0xC7,0x22,0x6B,0x8D,0xEB,0xEB,0xCF,
		0x90,0xFC,0xC3,0x90,0xF7,0x61,0x36,0x05,0x54,0xAC,
		0xF0,0x13,0xAD,0x03,0x91,0x1B,0x49,0xB6,0x41,0xC7,
		0x36,0x87,0xF4,0xA7,0x3F,0x8C,0x29,0x0A,0x28,0x90,
		0xC0,0x2C,0x56,0xE2,0xA6,0x56,0x7F,0xB0,0x38,0x26,
		0x35,0x3D,0x87,0xEA,0x41,0x36,0x96,0x73,0xE1,0xF9,
		0xB9,0x20,0xB3,0x62,0x03,0xCC,0x9A,0xA4,0x64,0x19,
		0x2D,0x01,0xB6,0xD8,0x59,0x30,0x79,0x27,0xA6,0x29,
		0xA2,0x4B,0x34,0x13,0x2C,0xBC,0xEC,0xA4,0x87,0xA0, /* 1100 */

		0xCB,0x3C,0x74,0x42,0xAB,0xA4,0x7C,0x4A,0x07,0x7F,
		0xBE,0x74,0x42,0xC2,0xA4,0x22,0x1A,0x20,0x0A,0x93,
		0xE7,0x26,0x51,0x88,0x30,0xB1,0x1B,0x69,0x0F,0x04,
		0x97,0xAC,0x08,0x51,0xA5,0xB2,0x7E,0xA2,0x02,0x53,
		0x06,0x5A,0x9A,0xE7,0xE6,0x56,0x7F,0xC2,0x65,0x00,
		0xED,0xB1,0x13,0x46,0x95,0x5A,0xEB,0x2D,0x9C,0x6C,
		0x22,0x8F,0x16,0x44,0xF2,0x1B,0x34,0xE3,0x69,0x4C,
		0x05,0x01,0x62,0x84,0xBC,0xAF,0x74,0x6A,0xB6,0x34,
		0x4A,0x10,0xEE,0xC8,0x7C,0xD0,0xF7,0x6A,0x61,0x1A,
		0xB6,0x4E,0xC8,0x62,0x88,0xC3,0x39,0x29,0x4A,0xA1, /* 1200 */

		0x52,0x33,0xAB,0xFB,0x68,0x00,0x36,0xD2,0x15,0x5F,
		0xE7,0x73,0x63,0x53,0x0C,0x3E,0xE0,0xC4,0xB8,0xB7,
		0x4A,0x54,0xE9,0xA2,0x45,0x3C,0xA0,0x81,0xD0,0x1F,
		0x96,0x26,0x86,0x76,0x32,0x87,0x0B,0x17,0x8D,0x25,
		0xB8,0x31,0xFD,0x13,0x3A,0x2A,0xC0,0x59,0x6E,0x68,
		0x86,0x52,0xC1,0x77,0x2E,0x29,0x79,0x30,0x5C,0x5E,
		0x2A,0x10,0x94,0x9B,0xF3,0x5B,0xDC,0x08,0xCE,0xCD,
		0x67,0x71,0xB7,0xF6,0x59,0xB6,0x19,0x60,0xA4,0x50,
		0x54,0x99,0x84,0x72,0x16,0x3C,0x21,0xD3,0x10,0x24,
		0xC0,0xC2,0x49,0x72,0xD5,0xB1,0x45,0x60,0x8C,0x10, /* 1300 */

		0x44,0x71,0x8F,0x60,0x89,0x4B,0xF5,0x87,0x5A,0x45,
		0x3D,0x4A,0x7C,0x45,0x69,0xC6,0x3A,0x2B,0x38,0xC2,
		0xDB,0x6A,0x4F,0xB8,0xDC,0x92,0x89,0x61,0x0E,0xF1,
		0x27,0x54,0xF3,0x7C,0xB4,0xDA,0xD3,0xB6,0x6E,0x13,
		0x5D,0x1F,0x75,0x75,0x96,0x22,0xCD,0x7F,0xA0,0x3C,
		0x4B,0xA8,0x1D,0x71,0xAC,0x99,0x38,0xC9,0xA9,0x78,
		0x39,0xA1,0x2D,0x7A,0x46,0x22,0x94,0x3C,0x38,0x1A,
		0x04,0x8E,0xDB,0x26,0xC6,0x49,0x26,0xB3,0xA8,0x07,
		0xB6,0x44,0xAE,0xBC,0x91,0x10,0xC5,0x4B,0x06,0xD4,
		0xEA,0x1C,0xBF,0x64,0x62,0xA5,0x5B,0x31,0xC7,0xF5, /* 1400 */

		0x20,0x2C,0xCA,0x8B,0x7C,0x69,0x8D,0x95,0x59,0x27,
		0x6A,0x1A,0x39,0x63,0xCC,0x56,0xDD,0x26,0xCE,0x3C,
		0x05,0x11,0x67,0xF2,0x99,0xD8,0x0C,0x3A,0x8E,0xE8,
		0x94,0xF1,0x34,0x56,0x7C,0xEA,0x10,0x32,0xC8,0x45,
		0xD7,0x9C,0xCD,0x95,0x31,0x3F,0x04,0xD6,0x45,0x9D,
		0x1B,0xA7,0x2E,0x00,0xCC,0x60,0x43,0x65,0xA9,0xC5,
		0xC5,0x18,0x50,0x05,0xF1,0x7A,0x03,0x97,0xA5,0x59,
		0xA0,0xB2,0x8F,0x31,0xB4,0x1A,0x5E,0x09,0x48,0xC8,
		0x15,0x01,0xE9,0x02,0xB9,0xCC,0x25,0x3F,0x6E,0xBA,
		0x88,0x17,0x2B,0xBB,0x92,0x02,0x49,0x84,0xFA,0x99, /* 1500 */

		0xC5,0xBA,0xB0,0x8D,0xC0,0x15,0x04,0x55,0x6A,0xAB,
		0xE8,0xC8,0x1B,0xC9,0x9C,0x56,0xFC,0x77,0x1D,0xDC,
		0x85,0xD3,0x7B,0x7C,0x50,0x18,0x23,0x45,0xE1,0x89,
		0x26,0x79,0x34,0x56,0x4A,0xA9,0xDD,0x80,0xBC,0xDC,
		0x25,0xB1,0x27,0xF1,0x24,0x47,0x63,0x64,0x40,0x04,
		0xE5,0xEA,0xF0,0x6B,0x09,0x5B,0x89,0x2B,0xAE,0x09,
		0x5E,0x0B,0xA7,0xF9,0xD2,0x70,0x0A,0x50),
	.pub_r = chunk_from_chars(
		0xCE,0x2F,0xA3,0xE8,0x9C,0xD1,0xD0,0xC1,0x3C,0x47,
		0x70,0x59,0x8D,0x67,0x15,0x5B,0x43,0x84,0x41,0x90,
		0xD8,0xFA,0x83,0x65,0x15,0x07,0xB4,0xEF,0x68,0xF6,
		0x84,0x70,0xF6,0xE9,0x82,0xE7,0xF7,0x83,0x4A,0xD6,
		0x7A,0x1A,0xF8,0x13,0x37,0x98,0xD3,0xDD,0x57,0x36,
		0x95,0x75,0x22,0x6C,0x94,0xCE,0xB6,0x9F,0x60,0xCE,
		0x2D,0x0C,0x06,0x4F,0xD4,0x7D,0x8E,0x4D,0x8D,0x18,
		0xA1,0xE3,0xFE,0xC6,0x80,0x06,0xE3,0xD2,0xB9,0xD7,
		0x6C,0x65,0x43,0xD9,0x76,0x7D,0xA1,0x21,0xF5,0x2C,
		0x08,0x61,0x53,0xAB,0x91,0x49,0x2D,0x6B,0xAC,0xEC, /* 100 */

		0xA4,0x44,0xC0,0x15,0xA0,0x94,0xA2,0x78,0xD9,0xD2,
		0xD3,0xBD,0x7F,0x05,0xC9,0x8D,0x5C,0xDF,0xFD,0x1F,
		0x02,0x66,0xC4,0x64,0xE4,0x92,0xF4,0xDA,0xC8,0x1E,
		0x05,0xC5,0x67,0x50,0x0E,0xC9,0xC2,0x2C,0x16,0xB6,
		0x1B,0xE0,0xA6,0xED,0x43,0x6F,0x2D,0x34,0x33,0xAE,
		0x7D,0x7F,0x9C,0x21,0xC2,0xA7,0xCE,0x3F,0xDE,0xEF,
		0x39,0x41,0x9B,0x90,0x17,0xA3,0x98,0x71,0x05,0xE1,
		0x89,0xD5,0xFE,0x81,0xCB,0xFF,0x12,0x74,0x1D,0x63,
		0x1F,0xB3,0x09,0xE3,0xB9,0x80,0x82,0x1B,0x96,0x8D,
		0x66,0x4A,0x1F,0x89,0x28,0xC9,0x70,0xE7,0x89,0xC1, /* 200 */

		0x96,0x9A,0x62,0xA3,0xB7,0xBA,0x8B,0x29,0x80,0xFF,
		0x5E,0x8D,0x0B,0xA9,0x28,0x7C,0xC9,0x5B,0xD8,0xC7,
		0x3D,0x3A,0xDE,0xC8,0xC0,0x28,0xE4,0x19,0x51,0xF8,
		0x17,0xB8,0xFD,0xEA,0xFD,0x0F,0x1D,0x91,0xE2,0xAF,
		0x6B,0x25,0x14,0x9A,0xA5,0xB1,0x9B,0xE5,0xA3,0x42,
		0x46,0xC4,0xE5,0x4C,0x3D,0x0A,0x0F,0xC0,0x33,0x38,
		0xCE,0x2D,0x75,0x39,0xC2,0x4F,0xC4,0xED,0x09,0xDC,
		0x94,0xBD,0xCF,0x96,0xC4,0xCD,0x2E,0x11,0xDF,0x4E,
		0xFA,0x97,0x28,0x20,0x46,0x82,0xB4,0x41,0xCE,0xD8,
		0x6A,0xF3,0x3A,0x8A,0x90,0x89,0xDC,0x94,0x63,0x07, /* 300 */

		0x72,0xCD,0xBA,0x6E,0xD8,0xBA,0x77,0xF9,0xBA,0xA8,
		0xCE,0x32,0xA2,0x2D,0x28,0x6C,0x3C,0x5D,0x5F,0x16,
		0xBC,0xBF,0x54,0x46,0x66,0x77,0x0C,0xA3,0xB5,0x74,
		0x09,0xB6,0x5E,0x3F,0xD7,0x59,0x24,0x59,0x9E,0x9D,
		0x26,0xC0,0x5C,0xAA,0xF6,0x5E,0xA6,0x75,0xC7,0x91,
		0x95,0xF3,0x9F,0x39,0xB5,0x41,0xD9,0x2B,0x3B,0xFC,
		0x04,0x3B,0xBE,0xDC,0x89,0xE5,0xC5,0x3D,0x47,0x70,
		0xCE,0x11,0x09,0xD5,0x0B,0x45,0xA1,0x41,0x9B,0x68,
		0xC9,0xD1,0xC1,0x7A,0x84,0x21,0xE1,0xBC,0x55,0x8F,
		0xAD,0x9C,0xEA,0x3D,0x46,0xF1,0xED,0x6C,0x5B,0x46, /* 400 */

		0x32,0x1E,0x19,0xF4,0x1B,0xD0,0x04,0x9C,0x5E,0xF8,
		0xD7,0xCB,0xAD,0x8E,0x7B,0x0E,0xB3,0x26,0x7F,0x94,
		0x87,0x3D,0x01,0x88,0xE4,0x35,0xEE,0xD7,0x59,0x62,
		0xFB,0x1E,0x4C,0x2B,0x16,0x03,0xF8,0x4E,0xBA,0x69,
		0x6D,0x64,0x6C,0x18,0x80,0x31,0x36,0x5C,0x31,0xB1,
		0xC2,0x98,0xB6,0x60,0xC7,0xF6,0xBC,0x03,0x79,0xE1,
		0xF5,0x6E,0xE5,0xC0,0x4C,0x13,0x9E,0xE2,0x11,0xA6,
		0x50,0x7E,0x94,0x6E,0x4C,0xA1,0xD0,0x96,0xC4,0x82,
		0x49,0x4E,0x74,0xE1,0x54,0xD3,0x7D,0x4E,0x71,0xE8,
		0x02,0x4A,0x19,0xCE,0x81,0xA0,0xDD,0xF0,0x41,0xF6, /* 500 */

		0xBE,0x65,0xE7,0xD0,0x40,0xF8,0xDC,0xDD,0x3B,0xB9,
		0x13,0x45,0xA7,0x2F,0xB1,0x1A,0x30,0xE4,0x4F,0x27,
		0xF6,0x82,0xCC,0xF5,0x6E,0x00,0x73,0x59,0xD6,0xC4,
		0xEB,0xF5,0x51,0x90,0x13,0x32,0x41,0xA5,0x10,0xE8,
		0x11,0x6E,0x24,0x97,0x39,0x1D,0xEC,0x14,0x67,0x9A,
		0x9F,0xF1,0x45,0x9D,0x12,0x52,0x32,0x65,0x33,0x02,
		0x62,0x21,0x0D,0xEA,0x3A,0x9D,0x72,0xA2,0x09,0x25,
		0x35,0xC6,0xA9,0x6B,0x8C,0xE5,0xD1,0x65,0x59,0x8F,
		0x1F,0x3B,0x29,0x13,0x81,0x97,0x13,0xD1,0xFC,0x8D,
		0x5E,0xD5,0xA2,0xA8,0x47,0x51,0x06,0x6F,0x59,0x7E, /* 600 */

		0x60,0x69,0x40,0x79,0x4D,0xD7,0x83,0x9E,0x10,0x61,
		0x50,0xC4,0x93,0x54,0xEA,0xE9,0x3D,0x23,0x4D,0x90,
		0x0F,0x40,0x57,0x4A,0xC7,0xD3,0x50,0x27,0x0C,0x9B,
		0xA4,0xFC,0xC3,0x10,0xD3,0x19,0x59,0x04,0xCC,0x80,
		0x99,0xCF,0x7B,0x01,0xAD,0xFC,0x83,0xB4,0x76,0x33,
		0xB7,0x57,0x52,0xFD,0xE5,0xB8,0x5A,0xA9,0xA9,0x93,
		0xAD,0x2F,0xCC,0xB3,0x44,0xE1,0x73,0xAF,0xE9,0xCD,
		0x9F,0xC5,0xC8,0x6F,0x96,0x87,0x29,0xDF,0x19,0xEC,
		0x5E,0x01,0xF3,0x59,0xCD,0x4E,0x9D,0x6F,0xB8,0x20,
		0x0D,0x66,0x94,0x88,0x88,0x57,0x1D,0x73,0x59,0x98, /* 700 */

		0x3A,0x45,0x4E,0x48,0xB4,0xA6,0xEA,0x0B,0x25,0xA1,
		0xA9,0x70,0xE0,0xE6,0x88,0xF9,0x73,0x4E,0x17,0x83,
		0xE0,0x9F,0x9A,0x7C,0x38,0x9B,0xA0,0xC3,0xF3,0x93,
		0xFC,0x8E,0x55,0x88,0xF2,0xDB,0x00,0xAE,0x8E,0x14,
		0xDF,0xF8,0x02,0x1D,0xB6,0xCF,0xBA,0x71,0xDB,0x5E,
		0xE0,0xC7,0x7E,0x90,0xEA,0x44,0x60,0x46,0x91,0xAE,
		0xD7,0x6E,0xDE,0x15,0x79,0x23,0xDE,0x9A,0x7E,0x8E,
		0x29,0x62,0xFA,0xFF,0xE9,0xD4,0x23,0xAF,0xC2,0xD4,
		0x5B,0x51,0x7D,0x17,0x9B,0x6C,0xD8,0xA6,0x92,0x4E,
		0x20,0x44,0xBA,0x6A,0xE7,0x8C,0xB9,0xC1,0x21,0x9F, /* 800 */

		0x65,0xA2,0x08,0x01,0x45,0xC6,0x12,0x62,0x86,0xF8,
		0x9D,0xD2,0x07,0xD2,0x16,0xF2,0xD6,0xA9,0x7F,0x48,
		0x0F,0x3E,0x66,0x99,0x00,0x86,0xE4,0x5A,0x8E,0x4D,
		0x09,0x48,0xEA,0x9F,0xC3,0x84,0x7B,0x6B,0x24,0x3F,
		0xFC,0x0D,0x16,0xCA,0x2F,0x8A,0x38,0x4F,0x7A,0xE5,
		0x55,0x85,0x7E,0x55,0x00,0xE1,0x36,0xF8,0xC2,0x22,
		0x01,0xA9,0xD3,0x38,0xAB,0xC9,0x99,0x12,0x71,0x77,
		0x95,0xF9,0x57,0xBA,0x94,0x5A,0xAE,0xF2,0x67,0xC2,
		0x8E,0xB1,0xF5,0xC6,0xC8,0x3A,0x58,0xAA,0x1F,0x4C,
		0xF8,0xA4,0xBE,0xBB,0x12,0x64,0xE2,0xAC,0xF0,0x6B, /* 900 */

		0xB0,0x70,0x51,0x18,0x14,0xA4,0x7A,0x29,0x3D,0xEC,
		0xDD,0xE8,0x29,0xA8,0x75,0xF1,0xAC,0xC5,0x1F,0x6C,
		0x96,0x9E,0x86,0xC5,0xD6,0xAD,0x66,0x0D,0x03,0xF7,
		0xD0,0x9A,0x91,0xDC,0xD1,0x6A,0xAB,0x62,0xD0,0x3C,
		0x58,0xCF,0xC1,0x90,0xB5,0x5D,0xAA,0x9B,0x38,0xB6,
		0x31,0x8C,0xC8,0x39,0x0B,0x99,0xD9,0x8F,0x32,0xFB,
		0x13,0xFC,0xC8,0x08,0x21,0x5E,0x10,0x6D,0x6C,0x32,
		0x5D,0xE5,0xAF,0xB1,0xC2,0x54,0x93,0x3C,0x2F,0x42,
		0x97,0x7F,0xD8,0xCD,0x9F,0x07,0xE7,0x37,0xE4,0x4D,
		0xF4,0xC0,0x89,0xC6,0x08,0xBF,0xFD,0x3B,0x81,0x6D, /* 1000 */

		0x9E,0x24,0xA7,0xE9,0x44,0xDB,0x69,0x3B,0x1F,0xA4,
		0x14,0x5F,0xFF,0xFE,0xD8,0x80,0x4B,0xC4,0x11,0xE6,
		0x33,0x96,0xFC,0x1E,0x9A,0x69,0x24,0xEF,0x00,0x70,
		0xC9,0xF8,0x43,0x01,0x40,0xC4,0x03,0x33,0x3C,0xC5,
		0x09,0x32,0x44,0x38,0xFA,0xF9,0xDC,0xB4,0x84,0x1A,
		0x8A,0xBA,0xDE,0x35,0xF3,0x4F,0x07,0x38,0xF5,0x07,
		0x34,0xEB,0xD5,0x23,0x2C,0xDB,0x90,0x50,0x61,0x22,
		0x5A,0xFD,0xDB,0x1E,0xD4,0x1F,0x00,0xB8,0x37,0x22,
		0x22,0x5C,0x65,0x7D,0x7A,0x89,0xDC,0x49,0xA0,0x7A,
		0x29,0x40,0x53,0xD4,0xAC,0x48,0x38,0xF7,0x9A,0x74, /* 1100 */

		0xCD,0x50,0xD3,0x60,0x46,0xC6,0xA7,0x31,0x03,0x0E,
		0x24,0xC2,0xD0,0x93,0x6C,0x5F,0x92,0xE3,0xD8,0x20,
		0x4D,0x3C,0x32,0x2C,0x83,0x13,0xA6,0xFB,0x88,0x3A,
		0x15,0x37,0x97,0x13,0xD2,0x98,0x85,0xCF,0x60,0x0B,
		0x10,0x51,0xDF,0x9D,0x36,0x14,0x8D,0xC6,0x99,0x8F,
		0x3E,0xFA,0x9B,0x54,0xE5,0x9F,0x65,0x91,0x39,0xC2,
		0x3D,0xEF,0xD6,0xD0,0x31,0x16,0xC3,0xB7,0x7A,0x7A,
		0x5B,0x72,0x19,0xFA,0x3E,0x9C,0x3A,0x4F,0xCD,0x82,
		0x42,0x68,0x5C,0xA2,0x6E,0xBC,0xA8,0x8F,0x5B,0x3B,
		0x09,0x4A,0xF8,0xED,0xAC,0xDB,0x74,0xEA,0x00,0x3D, /* 1200 */

		0xEB,0xDF,0xCB,0x8D,0xBA,0xA3,0xD1,0x7A,0x39,0x92,
		0xE1,0x04,0xD3,0x44,0x09,0x2D,0x36,0x80,0x4A,0xFE,
		0xF0,0x86,0x0E,0xC5,0xB6,0x86,0xA6,0x3A,0xFC,0xAB,
		0x4F,0xD4,0x2D,0xDD,0xF7,0xE1,0x27,0x69,0x46,0x4D,
		0xD5,0x2D,0x93,0xC4,0x21,0x03,0xB2,0xEA,0x23,0x5F,
		0xEC,0x59,0x79,0xDE,0xCB,0xEE,0x9C,0x14,0x92,0x29,
		0x1F,0xB7,0x02,0x80,0xC7,0xE3,0x7C,0x33,0xDF,0xD4,
		0xA3,0x10,0x9D,0x5E,0xDB,0x5D,0x24,0x36,0x35,0x88,
		0x59,0x62,0xBF,0x8A,0x5E,0x0A,0x66,0xE8,0xD3,0x59,
		0x3A,0x56,0x0B,0x5F,0xA5,0x4F,0x90,0x6F,0x7C,0xF8, /* 1300 */

		0x56,0x66,0x9F,0xB0,0x81,0xF4,0x5E,0x49,0x84,0x60,
		0xAD,0x05,0xA5,0xBB,0x23,0x67,0xEE,0x66,0x94,0x00,
		0xA2,0x88,0x68,0x16,0x8A,0x6A,0x3B,0xA8,0x89,0xC7,
		0xF3,0xE4,0x06,0x9E,0x29,0x78,0x28,0x79,0xC1,0x30,
		0xF6,0xF3,0x44,0xE2,0xD4,0xB8,0x04,0xDF,0xD7,0xD7,
		0xA3,0x11,0x98,0x18,0x8E,0xD5,0x2A,0x16,0xC5,0x0E,
		0xC3,0x92,0xBE,0x7D,0xAE,0xED,0x5C,0x2F,0xA0,0x4B,
		0x9B,0x5D,0x08,0xCB,0xC1,0x59,0xB3,0xBF,0xF2,0x40,
		0x97,0x08,0x14,0xB4,0xF8,0x4D,0xAD,0xD0,0xDB,0x9B,
		0xC3,0xBD,0x6D,0x08,0xFD,0x23,0xA8,0x1D,0xE7,0xC8, /* 1400 */

		0xCF,0xC5,0x03,0xC2,0x3E,0x48,0xE6,0x1D,0xD8,0xE7,
		0x40,0x43,0xD6,0xFB,0xAC,0x87,0xAC,0xFE,0xF1,0xF6,
		0x6D,0xC1,0x56,0xD1,0xE6,0x90,0x96,0x9A,0x7D,0x93,
		0xA9,0xCD,0x62,0x97,0x81,0xAB,0xD4,0xC9,0x22,0x6B,
		0x34,0xCA,0xE4,0x77,0xB5,0x47,0x60,0x21,0x91,0xCF,
		0xA1,0xEB,0x1F,0xC4,0xD6,0x69,0xD3,0x8D,0x1B,0xA7,
		0xD2,0xF6,0xD6,0xAF,0x4F,0xDC,0xF0,0x9F,0x3E,0x29,
		0x8C,0xB8,0x94,0x46,0x69,0x33,0x2F,0x03,0x5C,0xB0,
		0x79,0x8A,0x1C,0x7F,0x36,0x66,0x35,0x56,0xF0,0xC3,
		0xA3,0x87,0x9D,0x4D,0x55,0xA1,0x00,0x56,0x07,0x7E, /* 1500 */

		0x29,0xE8,0xA4,0x4D,0x8F,0x61,0x27,0x56,0xD0,0x35,
		0xE2,0xF0,0xB9,0x42,0xF6,0xAE,0xF0,0xFE,0xC5,0x89,
		0x71,0x4C,0x94,0x81,0xCC,0x3B,0xD4,0xB0,0xD7,0xCF,
		0xB0,0x66,0x37,0x36,0x3F,0x39,0xD8,0x33,0x46,0x6B,
		0xDA,0x93,0x79,0x5E,0x91,0x9C,0xC8,0x6F,0xE1,0x3E,
		0x62,0x0C,0xD3,0x72,0x84,0x4E,0x16,0x88,0x11,0x97,
		0x51,0xCE,0x65,0xD0,0xD7,0x9D,0x11,0xF8),
	.shared = chunk_from_chars(
		0x8B,0xDB,0x8B,0x7D,0xA6,0xAF,0x99,0xA6,0x86,0x47,
		0x98,0x3D,0x18,0xEF,0x82,0xD0,0x27,0x8B,0xA1,0xED,
		0xB9,0x64,0x7E,0x3B,0xB1,0x5D,0x30,0xFE,0xC2,0xEE,
		0x82,0x6C),
};

ke_test_vector_t mlkem1024_2 = {
	.method = ML_KEM_1024,
	.seed = chunk_from_chars(
		0x64,0x33,0x5B,0xF2,0x9E,0x5D,0xE6,0x28,0x42,0xC9,
		0x41,0x76,0x6B,0xA1,0x29,0xB0,0x64,0x3B,0x5E,0x71,
		0x21,0xCA,0x26,0xCF,0xC1,0x90,0xEC,0x7D,0xC3,0x54,
		0x38,0x30,0x55,0x7F,0xDD,0x5C,0x03,0xCF,0x12,0x3A,
		0x45,0x6D,0x48,0xEF,0xEA,0x43,0xC8,0x68),
	.pub_i = chunk_from_chars(
		0xA1,0xA3,0x41,0xB5,0x78,0xB4,0x76,0x5C,0x46,0x49,
		0xE6,0xBF,0xAF,0x5C,0x8B,0x2A,0xD8,0x0D,0xE5,0x20,
		0x0E,0x4D,0xD3,0x0D,0xA0,0xB6,0x93,0xF5,0xEB,0xBF,
		0xCF,0xBA,0x9F,0x0A,0xD2,0xAD,0x46,0x57,0x6A,0x7A,
		0x06,0xB6,0x5D,0xE6,0xBC,0xDC,0x43,0x2B,0x6A,0x75,
		0x3F,0x3A,0x63,0x26,0xA2,0x70,0x73,0x43,0x64,0x83,
		0xAE,0xF1,0x70,0xB8,0x64,0x9E,0x2C,0xA0,0x22,0x49,
		0x73,0x29,0xD0,0x71,0xAF,0x4C,0x25,0x96,0x8C,0xB0,
		0x52,0x1A,0x11,0x8A,0xEC,0x41,0xCB,0xEA,0xB0,0x13,
		0x78,0x6A,0x1C,0x29,0x84,0xCB,0x2C,0x64,0x9D,0xA0, /* 100 */

		0xFC,0x94,0xFD,0xD8,0x80,0xFD,0x3A,0x05,0x23,0x9B,
		0x84,0xAC,0x05,0x3C,0xA0,0xF8,0x22,0xE4,0xC9,0x4E,
		0xBF,0xE1,0xB6,0xD6,0x89,0x8D,0xA0,0x3B,0x02,0xE4,
		0x55,0x46,0x7A,0xF6,0xC8,0x9D,0xB7,0x22,0x5A,0xD6,
		0x53,0xA5,0xAC,0x0C,0x8E,0x16,0x5F,0x3B,0x9B,0x68,
		0xB0,0x12,0x70,0xEF,0x0B,0xA7,0xCC,0x56,0x32,0x2D,
		0x5C,0x67,0x18,0x05,0xB7,0xD3,0x33,0xB8,0x99,0xAB,
		0xBA,0xC1,0x20,0xC0,0x10,0x03,0x3F,0x30,0x64,0x64,
		0x7E,0x4A,0x38,0x76,0x49,0xC2,0x72,0xA3,0x2C,0x2D,
		0xCC,0x2F,0xD6,0x27,0x93,0xD8,0x68,0x9C,0xF5,0x96, /* 200 */

		0x19,0xF4,0x3C,0x87,0xFD,0xEA,0xAE,0xBC,0xD4,0xC3,
		0xF2,0x14,0x83,0x86,0x35,0x63,0xCE,0x84,0x42,0xA7,
		0xF5,0x81,0x00,0x94,0x10,0xB3,0xC5,0x06,0x14,0x63,
		0x90,0xFC,0xD4,0x4D,0x0F,0x91,0x8B,0x29,0x03,0x94,
		0x0A,0x32,0x1C,0x66,0x70,0x0C,0xA4,0xE2,0x05,0xD4,
		0x50,0x1E,0xCE,0xC2,0x3A,0x4D,0x17,0x15,0x7C,0xE4,
		0xA0,0xBA,0xC5,0x62,0x53,0x32,0x73,0xFB,0x80,0x71,
		0xDD,0xF9,0x60,0x21,0x76,0x96,0xCC,0xD9,0x8A,0x59,
		0xE6,0xC3,0x9D,0xAC,0x5C,0xD1,0x85,0xA0,0x79,0x3A,
		0x91,0x90,0xE2,0x5A,0x04,0x60,0x83,0x7D,0x07,0xD0, /* 300 */

		0x20,0xD7,0x2B,0x5C,0x5B,0xA9,0x84,0x0B,0x09,0x40,
		0x83,0x2F,0xA3,0xA4,0x83,0xAE,0xC9,0x9D,0x0D,0x62,
		0x9C,0xFD,0x67,0xA1,0xCE,0x77,0x76,0x6A,0x23,0xB2,
		0x34,0x99,0x47,0x96,0x8C,0x23,0x8D,0x01,0x26,0xF6,
		0xAB,0x94,0x22,0xB9,0x7D,0x6D,0xEC,0x56,0x57,0xA2,
		0x0A,0x68,0x35,0x44,0xBA,0x07,0x97,0x1B,0xC7,0x73,
		0xA5,0xDA,0x90,0xBF,0x47,0x1E,0xAF,0x12,0x31,0x02,
		0x83,0x94,0x4C,0x72,0x3E,0xB1,0xE0,0x60,0x0E,0xEB,
		0x24,0x64,0xBC,0x71,0xDE,0x72,0x33,0xEB,0xFB,0x7B,
		0xE6,0x42,0x02,0xEE,0x21,0x08,0x82,0x22,0xA4,0x8B, /* 400 */

		0xA2,0x00,0xF0,0x4B,0x6B,0xF6,0xA1,0xC7,0xF6,0x40,
		0x42,0x95,0x76,0x4E,0x6F,0xD8,0x57,0xD7,0x47,0xCD,
		0x98,0x5B,0x07,0xF4,0xD8,0x3F,0x8B,0x13,0xA9,0x01,
		0xCC,0xBF,0xF7,0xB0,0x1E,0x21,0xB8,0x78,0x9C,0x05,
		0xB6,0x35,0xBC,0xC1,0x4C,0x81,0x28,0xC3,0x79,0xBD,
		0xD5,0x20,0x29,0xEF,0xD6,0x60,0x46,0x8A,0x34,0x51,
		0x2B,0xB7,0x8D,0xA2,0x24,0xFF,0xF9,0xA2,0x23,0x37,
		0x1E,0x45,0x73,0x33,0xAD,0x46,0x4C,0x6F,0xDB,0xA4,
		0xF6,0xC5,0x45,0x1B,0x58,0x55,0x83,0x65,0x51,0x82,
		0x69,0x1F,0x6E,0xE8,0x17,0xCF,0x81,0xCE,0x6E,0x7A, /* 500 */

		0x5C,0xC6,0xB9,0xCA,0xCE,0xE2,0x75,0xC4,0x73,0xAE,
		0x1C,0x58,0xAA,0x38,0xE4,0x3D,0xE5,0x07,0x96,0xF2,
		0x24,0x75,0xFC,0xA4,0x42,0x70,0xE3,0xA4,0x6F,0xB5,
		0xB8,0xBD,0xDA,0x74,0xE3,0xE0,0x5B,0x88,0x21,0x0F,
		0x54,0xF3,0x6E,0x79,0xFB,0x05,0xBE,0xCC,0x76,0x80,
		0x5B,0x71,0x61,0x54,0xA6,0xE2,0x34,0x6B,0x0F,0x4C,
		0xAC,0x24,0x17,0x48,0xFD,0xE9,0x3F,0x32,0x5B,0x4F,
		0x82,0xF3,0xB6,0x81,0x13,0x6A,0xFF,0x80,0x56,0xBC,
		0x73,0x04,0xEF,0x22,0x65,0x2F,0x16,0x79,0xC8,0x16,
		0x03,0x2B,0xC3,0x5D,0xB6,0x8B,0x64,0xB2,0x84,0x83, /* 600 */

		0x51,0xDA,0x88,0xDF,0x89,0x16,0x67,0x07,0xB1,0x94,
		0xCB,0x47,0x59,0xE6,0x76,0x10,0x32,0xAC,0xA4,0xB7,
		0xA5,0x4E,0x80,0x64,0xF5,0x75,0x5D,0x36,0x28,0x8F,
		0xC5,0x06,0x10,0x72,0xDA,0x1E,0xC3,0x5A,0x84,0x91,
		0x39,0x6C,0x94,0x06,0x44,0x60,0x60,0x34,0x73,0x61,
		0xBE,0xD8,0x60,0xB0,0x28,0xB5,0x72,0x50,0xF0,0x18,
		0x7F,0x45,0xA2,0x04,0xFB,0xA6,0x78,0xF5,0x87,0xAE,
		0x03,0x1A,0xC3,0x84,0x80,0xCF,0x4C,0x3F,0xFF,0x90,
		0x18,0x8D,0x92,0x5B,0xD4,0xC7,0xC6,0xBD,0x0B,0x75,
		0x86,0x98,0xBC,0xDE,0x97,0xCE,0x39,0x58,0x69,0x76, /* 700 */

		0x10,0xCE,0x05,0xE4,0xAB,0xFC,0xA6,0x0E,0xAC,0x13,
		0xB6,0x66,0x73,0x2E,0x82,0x62,0xB3,0x6B,0x74,0x47,
		0xC0,0xC9,0x3F,0x91,0x36,0x2C,0x51,0x93,0x37,0x05,
		0x28,0x65,0x92,0x7A,0x98,0x62,0xD2,0xC1,0xED,0x18,
		0x6B,0x41,0x15,0x54,0x27,0x5A,0x24,0xDC,0x97,0x38,
		0x75,0x1A,0x4B,0xA2,0x43,0xBC,0xFE,0x46,0x10,0x92,
		0x12,0xB7,0x11,0xAA,0x45,0x3A,0x35,0xB0,0x9B,0xA2,
		0x0A,0xB4,0x05,0xB6,0xF0,0xF3,0x45,0x73,0x21,0xB2,
		0x40,0x17,0x26,0x1B,0x49,0x92,0x32,0x25,0xB3,0x62,
		0x06,0x78,0xC4,0xF4,0x74,0x51,0x53,0x96,0x11,0x38, /* 800 */

		0x80,0xDF,0x46,0x31,0x38,0xAB,0xC3,0xE8,0x20,0xC6,
		0x54,0x15,0xC5,0x93,0xD7,0x4C,0x01,0xE5,0x13,0x61,
		0x50,0xB0,0x2F,0x1C,0xCC,0x55,0xA6,0xC1,0xE7,0xE9,
		0xB1,0x21,0x58,0x83,0x62,0xD7,0x84,0xC3,0xE8,0x56,
		0x58,0x53,0x23,0x31,0xBB,0x59,0x66,0x82,0x37,0x7C,
		0x75,0x2E,0x7C,0x25,0x96,0x08,0xB6,0x29,0xE7,0x4A,
		0x5A,0x47,0x0B,0xAD,0x77,0xD3,0xC0,0x67,0x2B,0x23,
		0x83,0x1A,0x21,0x7C,0x1A,0xC8,0x2B,0x94,0xA8,0x24,
		0x37,0x64,0x38,0x07,0x68,0x1C,0xC1,0x68,0x40,0x98,
		0xB8,0x6D,0xE0,0x31,0x44,0x0A,0x79,0x37,0x8A,0x22, /* 900 */

		0x9B,0xD9,0x02,0xF2,0x99,0x6F,0x32,0x6A,0x0D,0xF1,
		0x63,0x54,0x77,0x8A,0x4C,0xD5,0x37,0x5C,0x75,0x6A,
		0x34,0x89,0x0C,0xCF,0x0C,0x92,0xBA,0x07,0xA6,0x15,
		0x46,0x38,0x7B,0xF5,0x4C,0x2C,0xB0,0xC8,0x54,0x8D,
		0x47,0x80,0xAA,0x72,0xB6,0x67,0xC0,0xA1,0xFE,0x40,
		0x0B,0x30,0xF1,0x94,0x46,0x5C,0xC9,0x7F,0xF4,0x25,
		0x11,0xE7,0x54,0x57,0xD1,0xA8,0xA9,0x71,0x2A,0x6E,
		0xA9,0x71,0xD7,0x97,0xBA,0xA8,0x94,0xB0,0x34,0x85,
		0x61,0xDF,0x06,0x8A,0x34,0x55,0xA8,0x9A,0xFC,0x36,
		0x84,0x08,0x1D,0xDF,0xC0,0xA9,0x0F,0xD8,0xAC,0xBD, /* 1000 */

		0xC2,0x17,0xC8,0x33,0x98,0x38,0x02,0x67,0x77,0x60,
		0x51,0xCF,0x34,0x91,0x54,0x25,0x34,0xB1,0xA7,0x4B,
		0x40,0x9B,0x69,0xA8,0xA1,0x42,0xC1,0xB4,0x37,0xB5,
		0xA8,0x56,0x1F,0xB0,0x64,0xEC,0xB5,0x91,0xB7,0x33,
		0x4A,0xFC,0x48,0x2A,0xF6,0x78,0xCF,0xA4,0xE1,0x1E,
		0x97,0x0A,0x48,0xBD,0xF8,0x22,0xA0,0xB4,0x5E,0x1C,
		0x59,0x1D,0x07,0x79,0x28,0x84,0xE4,0xBD,0x05,0x7A,
		0x15,0xEE,0xCC,0x5C,0xDF,0x66,0xC9,0x94,0xCC,0xA7,
		0x65,0xDA,0xAB,0x1A,0xF3,0x77,0xA1,0x4B,0x98,0xBC,
		0xD8,0x91,0x8F,0xE3,0x54,0x6F,0xA4,0x43,0x8B,0xA5, /* 1100 */

		0x32,0x2D,0xA1,0xBA,0x75,0x54,0x3B,0xE2,0x45,0xB1,
		0x2E,0x22,0x8D,0x07,0xC1,0xCF,0x10,0x91,0x49,0x9A,
		0x2C,0x57,0xDD,0x42,0xC9,0xF6,0xBA,0x32,0x40,0xC0,
		0x53,0x16,0x5C,0xA3,0xFD,0x66,0x9B,0x4C,0x39,0x44,
		0xA7,0xD0,0x03,0xD0,0x16,0xA0,0xFC,0xA8,0x26,0xB5,
		0xE6,0x15,0x84,0x6C,0x48,0x87,0x3C,0xB7,0xF2,0x22,
		0xCC,0x12,0x46,0x20,0x23,0xE3,0x8F,0x14,0xD9,0x8F,
		0xA0,0x48,0x6E,0xAE,0x48,0x0F,0x81,0xA7,0x37,0x6E,
		0xB6,0x38,0x0A,0x42,0xA1,0xA2,0xA6,0x12,0x3D,0x8A,
		0x4E,0xFB,0x10,0x2E,0xE7,0xB9,0x00,0xCB,0xE7,0xA9, /* 1200 */

		0x23,0x8B,0x59,0x00,0x6A,0xBD,0xDF,0x2A,0xB2,0x55,
		0x65,0x81,0x59,0x47,0xB3,0xBF,0x6B,0x14,0xEE,0xD4,
		0x3B,0xE9,0x1A,0x56,0x87,0xB3,0x39,0xE6,0xBB,0x66,
		0x14,0x38,0x75,0x83,0xC5,0x85,0xF1,0x45,0x63,0x72,
		0xAA,0x68,0xB1,0x74,0x49,0xC2,0xA1,0xC6,0x26,0x9A,
		0x05,0xF8,0x65,0x9D,0xA9,0x75,0x85,0xF5,0x24,0x21,
		0xAB,0x74,0xB9,0xD7,0x05,0xBA,0x13,0xAC,0x13,0x9F,
		0x15,0xBE,0x9F,0x41,0x97,0x1E,0x0A,0x6E,0x04,0xBA,
		0x16,0x7B,0xE9,0xB6,0x81,0xD2,0x94,0xC7,0x06,0xD0,
		0x47,0x7A,0x22,0xD6,0x45,0x65,0xE7,0x19,0x79,0x00, /* 1300 */

		0x9A,0x18,0xC8,0x08,0x60,0x59,0xF3,0x96,0x4E,0xD3,
		0x27,0x28,0x8A,0xAA,0x0C,0x8A,0x2B,0x2D,0x97,0x6D,
		0x5F,0xBB,0x50,0xB8,0x42,0xBC,0xDF,0x3C,0x9D,0x25,
		0x65,0x6E,0xD7,0x22,0x4F,0x34,0x35,0xC9,0xBC,0x9B,
		0x59,0xB1,0x77,0x27,0xB4,0x6A,0x54,0x6E,0xB8,0x55,
		0x3D,0xAB,0x54,0xEC,0xC6,0x84,0x2C,0xBA,0x94,0xFB,
		0x24,0x59,0x2E,0xD0,0x77,0xFC,0x56,0xBD,0x5D,0xD2,
		0xCD,0x79,0x87,0x15,0xFE,0x0C,0x51,0x33,0x77,0x36,
		0x1D,0x26,0x3C,0x63,0x48,0x34,0x0A,0x50,0x18,0xB0,
		0xE9,0xB4,0x3E,0x11,0x2B,0xB6,0xBA,0x60,0x50,0x3C, /* 1400 */

		0xA9,0x54,0xF2,0xBB,0x25,0xFC,0x58,0x5C,0x02,0x1B,
		0x57,0xFB,0x3D,0x1A,0xCC,0x80,0xA3,0xB2,0x0F,0xF4,
		0x55,0x5E,0xBF,0xA2,0x23,0xF9,0x07,0x62,0xE2,0x00,
		0x8C,0xE9,0x20,0xA4,0x20,0xB4,0x8C,0xD8,0xFA,0x96,
		0x06,0xEC,0x58,0x34,0x05,0x1D,0x09,0x04,0x84,0x74,
		0x66,0x56,0x42,0x94,0xCD,0xAD,0xDB,0x25,0xCA,0xB5,
		0xCC,0x68,0x00,0x8F,0x96,0x97,0x1B,0x27,0x11,0xAE,
		0x8D,0x73,0xB7,0x54,0xCC,0x8A,0xB2,0xBC,0x13,0xE0,
		0x77,0x08,0xE7,0xB5,0x55,0xAC,0x68,0x1C,0x0A,0xF0,
		0x96,0x1A,0x19,0x10,0x5B,0xF6,0x2B,0x93,0x26,0x7B, /* 1500 */

		0x8C,0x47,0xC7,0x35,0x26,0xB8,0xC2,0xD0,0x8B,0xB1,
		0x38,0x67,0xBE,0x10,0x21,0x13,0x47,0x65,0x6C,0x01,
		0x08,0xAA,0x38,0x25,0x23,0x48,0x47,0x51,0x9A,0x52,
		0xB6,0xA4,0x59,0x9C,0xCA,0xC5,0x9E,0x10,0xE9,0x45,
		0xEB,0x7B,0xD5,0x79,0xE0,0xC9,0x8D,0x12,0xAA,0x99,
		0x05,0xED,0x85,0x27,0x18,0x87,0xA9,0x74,0xAB,0x8F,
		0x04,0xA7,0x83,0x74,0x7C,0xAA,0xB6,0x9D),
	.pub_r = chunk_from_chars(
		0x0D,0x6F,0x09,0x75,0x71,0x4A,0x79,0x4C,0x4E,0x31,
		0x11,0x47,0xC5,0xC8,0x28,0x51,0xC8,0xDF,0xB1,0x79,
		0x0F,0x78,0x0C,0xEC,0x27,0xC7,0x61,0xA9,0xEA,0xBB,
		0xB5,0x2B,0xD3,0xB9,0x08,0xFB,0x8B,0xFE,0x1A,0xD4,
		0x0F,0x4A,0x1F,0x24,0x83,0x4D,0xF0,0xDC,0xCB,0x43,
		0xFD,0x1A,0x9A,0xF0,0x36,0x7B,0x5E,0x71,0xEC,0x9A,
		0x72,0x23,0x61,0xD5,0x64,0x3D,0xD5,0x94,0x16,0x6B,
		0x99,0xED,0x7A,0x64,0xC1,0x3A,0x8B,0x8E,0x93,0x8A,
		0x01,0x5A,0x96,0xC7,0xD9,0x10,0xA4,0x9F,0x7E,0x65,
		0x34,0x5B,0x74,0x2C,0x71,0x01,0x1F,0x2F,0xFE,0x81, /* 100 */

		0xF7,0x95,0x31,0x3E,0x0E,0x5C,0x77,0x95,0xF1,0xDF,
		0x65,0x7F,0xAE,0xCF,0xD9,0x2C,0x39,0xEB,0x0D,0xDC,
		0xD3,0xDC,0xF9,0xA7,0x53,0x09,0x62,0xD4,0x74,0x38,
		0x25,0xA6,0x8C,0xB7,0xEE,0x30,0x3F,0x76,0xB4,0x43,
		0xB5,0x66,0x23,0x42,0x55,0xBD,0xD5,0x89,0xA3,0xE7,
		0xB0,0x26,0xFB,0xE2,0x3A,0x66,0x0E,0x00,0xDE,0x61,
		0x54,0x03,0x44,0x63,0xC7,0xC8,0xD0,0xD2,0x86,0x7F,
		0x0E,0xF2,0xAD,0xF3,0x39,0xC9,0x4E,0x71,0xD9,0xB6,
		0xD8,0x32,0x54,0xC7,0x22,0x62,0x85,0x46,0xAB,0xCD,
		0x02,0xAF,0xB3,0xB3,0x71,0x8E,0x2E,0xCC,0xD9,0x0D, /* 200 */

		0xFF,0x86,0x36,0x79,0xBC,0x9F,0x8F,0x5E,0xA7,0xC7,
		0xEA,0xF0,0xB5,0xB7,0xB6,0x81,0x3B,0x23,0x39,0x94,
		0x9E,0x0F,0x96,0x56,0xA2,0x30,0x2B,0x09,0xB8,0x7A,
		0xDF,0x25,0xFD,0x32,0x8D,0x99,0xE5,0xE0,0xAE,0xA3,
		0xB4,0xE0,0xA2,0xDA,0x6A,0x25,0x10,0x2C,0xC7,0x87,
		0xDE,0x08,0x1C,0xD2,0x5F,0x9B,0x96,0x20,0xE1,0x46,
		0xB4,0x02,0xD4,0xCC,0x76,0xC7,0xB9,0x63,0x16,0xA2,
		0xA2,0x17,0xF8,0x7E,0xDB,0x7A,0xF8,0x95,0xDA,0x47,
		0x09,0x7D,0x41,0x2D,0xAE,0xB1,0xB3,0xAE,0x26,0xEF,
		0x7F,0xFC,0x47,0xD2,0x26,0x94,0x23,0xF8,0xCE,0x33, /* 300 */

		0xEE,0x20,0x6A,0x9A,0x18,0x41,0x06,0x4F,0x54,0x34,
		0x75,0x07,0xC5,0x74,0x25,0xCC,0x8A,0x7A,0x62,0x51,
		0xD4,0x4A,0xD2,0x1D,0xB6,0xE3,0xB0,0x7B,0xA5,0x1E,
		0xE5,0x21,0x72,0xE8,0xB3,0x5F,0xA6,0x2F,0x02,0xF4,
		0x84,0x83,0x0D,0x24,0x3E,0x99,0x01,0x48,0x89,0xE9,
		0x2E,0x70,0x85,0x40,0x42,0x87,0xE0,0x36,0xB5,0x3C,
		0x90,0x39,0x84,0x83,0xDA,0xE3,0xA0,0x81,0x55,0x42,
		0x41,0x10,0x7C,0x5F,0x3B,0x2D,0xD4,0x5E,0x25,0x96,
		0x58,0x86,0x54,0x61,0xB5,0x8B,0x21,0xDA,0x16,0xF7,
		0x8D,0xD6,0xE8,0x92,0x06,0x3B,0x93,0x30,0xB3,0xCC, /* 400 */

		0x79,0xD1,0x82,0x75,0xBE,0x1E,0x58,0xF2,0x7B,0x5F,
		0x44,0x2D,0xF7,0xF5,0x2F,0x1E,0x86,0xE7,0x9E,0xFB,
		0xF5,0xB8,0x34,0xA9,0x10,0x02,0x79,0x43,0x64,0xA7,
		0x10,0x4E,0x36,0xCF,0x5D,0xAF,0xC8,0x01,0xFB,0xEB,
		0x71,0x40,0x61,0xB5,0xEA,0xC9,0xEE,0xBA,0xAA,0xFF,
		0x64,0x82,0xBF,0x0B,0xAE,0x53,0x19,0x59,0x52,0x89,
		0x25,0x4F,0x3B,0xD3,0x01,0x63,0x3A,0x99,0x95,0x24,
		0xDC,0xD3,0xA0,0x2B,0x4E,0x83,0x6C,0xBD,0xF2,0x10,
		0x72,0xF2,0x3C,0xE4,0x14,0xFC,0x87,0x1A,0x81,0x99,
		0xBC,0x60,0xB0,0xF2,0xD8,0x76,0xB0,0xBC,0x18,0x90, /* 500 */

		0x5A,0x58,0xE8,0xD4,0xB3,0x87,0x1E,0x2B,0x93,0x00,
		0xF7,0xBE,0xF6,0xDE,0x50,0x00,0x31,0xCB,0xA9,0x20,
		0x25,0xDA,0xDB,0x96,0xED,0x73,0x15,0xC7,0x3A,0x5A,
		0xBF,0xDB,0x72,0x58,0x3F,0x53,0xA1,0x9B,0x21,0x82,
		0x55,0xDE,0x6B,0x4E,0x07,0x57,0xEE,0x24,0x9A,0x37,
		0x4B,0x5B,0xE2,0x0B,0x4E,0x03,0x66,0xC6,0x45,0xEF,
		0x68,0x63,0x0E,0xCA,0x9A,0x8B,0x29,0x07,0xAD,0x00,
		0xDF,0xBF,0xD2,0xA1,0x83,0x9D,0x79,0x0E,0xC3,0xC4,
		0x2F,0xC6,0x87,0xEB,0x93,0x5B,0x2E,0x2B,0x0C,0xE4,
		0x38,0xF2,0x2C,0xD5,0x47,0xA3,0x93,0xD9,0x56,0xAC, /* 600 */

		0x22,0x4D,0x4C,0x53,0x0A,0x51,0xC4,0xBB,0x2F,0xE0,
		0x08,0x6F,0xAB,0xFE,0x31,0x05,0x1E,0xFB,0x02,0x74,
		0x22,0x7C,0x94,0xEC,0x11,0x13,0x81,0x79,0x3E,0x22,
		0x8F,0x0D,0xF9,0xA0,0x78,0x46,0xB0,0x9D,0x2B,0xF4,
		0xC8,0x09,0x18,0xE4,0x6E,0x90,0x22,0x0A,0x6F,0x3C,
		0x27,0xD0,0x56,0x0D,0x66,0x33,0x5F,0x2F,0xED,0xF7,
		0x17,0xF6,0x65,0x3B,0xED,0xA1,0xFF,0x38,0x8C,0x26,
		0xCC,0xDC,0x3F,0x43,0xB3,0x3F,0xB6,0xBF,0x09,0x66,
		0xC4,0xCD,0x30,0x94,0x4D,0xBB,0xC2,0x29,0x8D,0xE3,
		0x57,0xBA,0x1F,0xD3,0xE6,0xAD,0x71,0x35,0xD5,0xB0, /* 700 */

		0x9A,0x30,0xB5,0x58,0xB9,0x9B,0x18,0x7A,0x3C,0x5A,
		0x44,0x11,0x1D,0x7F,0x79,0xB0,0x2A,0x31,0x1A,0x2F,
		0x0D,0x3F,0x96,0xDE,0x2A,0xB9,0x50,0x77,0xAC,0x4D,
		0x4F,0x13,0x85,0x95,0xD1,0xC4,0xA4,0xBC,0x6B,0xDB,
		0x9C,0x35,0x1E,0x87,0xDF,0x7E,0xA0,0xA9,0xA4,0x13,
		0x8A,0x23,0x8C,0x27,0x62,0x14,0x98,0x24,0xB0,0xB7,
		0x4A,0xA9,0x6A,0xBF,0xFB,0x29,0xB8,0x4C,0xB7,0x50,
		0x86,0x6F,0x93,0xC0,0x12,0xFC,0xFD,0xD4,0xAB,0x77,
		0x50,0x50,0xCC,0xB1,0x15,0x51,0x09,0xD4,0x00,0xF4,
		0x8F,0x18,0x6E,0x0F,0xB5,0xFD,0x50,0x59,0x46,0x94, /* 800 */

		0x9A,0x96,0x31,0x89,0x26,0x46,0xE4,0xCC,0xCE,0x2F,
		0xFF,0x69,0xF3,0x36,0x2A,0xFF,0xFF,0xC0,0x73,0x7F,
		0xE9,0xB2,0xE4,0x82,0xAB,0x55,0x83,0x0C,0x16,0xDF,
		0x63,0x47,0xCA,0xBC,0x23,0x53,0x9E,0x6A,0xE4,0x3F,
		0xC6,0xCC,0x60,0x99,0x09,0x8D,0x6B,0x6F,0x37,0x68,
		0x77,0x4B,0x84,0x28,0xE6,0x84,0x5C,0x79,0x15,0xBA,
		0x0B,0x01,0x80,0x9B,0x07,0x01,0x4E,0x76,0x99,0xA2,
		0x52,0xC8,0xED,0xD6,0x0E,0x77,0x87,0x5F,0x3E,0xB9,
		0x9D,0x84,0xB2,0x15,0xD4,0x7F,0x5B,0x4D,0x36,0x9C,
		0x85,0xB5,0x23,0x7E,0xFF,0x62,0x99,0x05,0x68,0x96, /* 900 */

		0x0A,0xA9,0x10,0x4C,0xCF,0xFE,0x51,0x81,0x5A,0x06,
		0x95,0xC3,0x53,0x5F,0x88,0x44,0x06,0x41,0xB4,0x12,
		0x81,0x51,0x17,0x65,0xAF,0x6D,0x4E,0x31,0x4A,0xD8,
		0x3F,0xC6,0x24,0xC3,0x81,0xDD,0x8C,0x7D,0xC6,0x61,
		0xD7,0x91,0x11,0x62,0x1D,0x12,0x3F,0x5C,0xA1,0x81,
		0x15,0xDF,0xEF,0x05,0x04,0xF8,0x60,0xCB,0x63,0xFA,
		0x83,0x38,0xF7,0x26,0xCA,0x29,0x27,0xC1,0x07,0x3E,
		0xFC,0x3F,0x55,0x4A,0x57,0xA1,0x88,0x41,0x87,0x17,
		0xE9,0x90,0xE9,0x97,0xF8,0x43,0xDE,0xEC,0x21,0xB5,
		0xC3,0x79,0x72,0x88,0xC5,0xB6,0x9E,0xF9,0xEE,0xF6, /* 1000 */

		0x70,0xFB,0x51,0x1A,0x60,0x06,0xFD,0x27,0xE4,0x82,
		0x82,0x30,0x11,0xEE,0x70,0x22,0xBE,0x4E,0xDF,0x5D,
		0xB3,0x9F,0x39,0xEB,0x9F,0x39,0xB9,0xC7,0xA7,0x16,
		0x13,0x99,0x78,0xAF,0xD5,0x79,0x15,0x99,0x76,0x47,
		0xED,0x7B,0xFC,0xD8,0xD3,0x9A,0x39,0xB6,0x8A,0x17,
		0x2C,0x90,0x4E,0xCE,0x15,0x77,0xB4,0x77,0xDE,0x81,
		0xDE,0x4B,0x61,0x45,0x25,0xF0,0x47,0x11,0x88,0x05,
		0x6C,0xD1,0xAC,0x7B,0x32,0xC9,0x1E,0x5D,0x3E,0xB2,
		0x7C,0x61,0x71,0x19,0xC4,0x89,0x93,0x44,0xAC,0x99,
		0xCF,0x06,0x1E,0x58,0x2F,0x48,0x86,0x8B,0x95,0x1F, /* 1100 */

		0x51,0x03,0x17,0x73,0x00,0x23,0xA9,0x10,0xE3,0xB4,
		0xCB,0xFF,0x43,0xDA,0x5B,0xDA,0xC5,0x5D,0xB3,0xDC,
		0x67,0xAA,0xB0,0xDD,0xF5,0xE3,0x57,0x37,0x4A,0xFF,
		0x81,0x7B,0xA3,0x7C,0x50,0x80,0x00,0x21,0xBC,0x65,
		0x9B,0x6E,0x2B,0xFA,0xCF,0x6A,0xC0,0xB4,0x68,0xE9,
		0x59,0x68,0x81,0x21,0x23,0xDC,0xF0,0x02,0x66,0xB3,
		0x8D,0x98,0x23,0x17,0x1E,0x2D,0xF8,0x6B,0x9B,0x6A,
		0x2B,0x76,0x4F,0x0C,0xBC,0x55,0x7E,0x4C,0x42,0xCC,
		0x6E,0x49,0x65,0x40,0xF6,0xB9,0x61,0x8B,0x90,0x49,
		0xAB,0xF0,0xD7,0x0A,0xBD,0xBB,0x00,0x63,0x54,0xA0, /* 1200 */

		0xC5,0x0F,0x2F,0x77,0xEA,0x7E,0x09,0x01,0xC8,0x60,
		0x6E,0x30,0x37,0x8B,0x5B,0x06,0x11,0xE1,0x1E,0xA4,
		0x70,0x10,0xF4,0xFB,0x04,0xB1,0xE6,0xD6,0x6E,0xC3,
		0xD5,0x8B,0xA1,0x7F,0x90,0x7F,0x08,0x87,0x2C,0x5A,
		0x65,0x60,0xB3,0x00,0xA4,0xC7,0xB2,0xD4,0x32,0x78,
		0x01,0xC2,0x31,0x16,0xAE,0x08,0x2C,0x12,0x81,0x14,
		0xD0,0x63,0x5B,0x4F,0xEE,0x92,0x6B,0xC6,0xC0,0xD3,
		0x8F,0x68,0xE9,0x5C,0x47,0x18,0xA8,0xCB,0x3F,0xCC,
		0xEF,0x05,0x53,0xD3,0xC4,0x98,0x4A,0xD6,0xC5,0x71,
		0x3A,0x4E,0x70,0xE9,0x21,0xFB,0xB5,0xB0,0xAF,0xE7, /* 1300 */

		0x6E,0xE7,0x47,0x30,0xB9,0x3E,0x92,0x86,0xB0,0x94,
		0x66,0x32,0xD8,0xDA,0x98,0xF4,0xE7,0x28,0xE6,0x2D,
		0x11,0x51,0x8D,0xDB,0x66,0xB9,0x71,0xD4,0x73,0xB3,
		0xAF,0x15,0xD1,0x84,0xEE,0xBD,0x0B,0xF1,0xDC,0x0F,
		0xCC,0x3E,0xA9,0xDE,0xE2,0x89,0x47,0xEA,0x5A,0xD9,
		0xFB,0xFD,0x42,0x1F,0x9E,0x5E,0xA4,0x63,0xFC,0xBD,
		0x31,0x74,0x2B,0x18,0x7C,0x89,0xCC,0x06,0x6E,0x88,
		0x37,0x67,0x5E,0x47,0x43,0x27,0xD6,0x5B,0x5B,0x74,
		0xD6,0x1E,0x33,0x6F,0x47,0xF9,0x70,0x2C,0xC7,0xD0,
		0x18,0xAA,0x55,0x3D,0x1D,0xFF,0x91,0x5D,0x9D,0x79, /* 1400 */

		0x08,0x4D,0x84,0x34,0xFD,0x75,0xCC,0x74,0x86,0x49,
		0x22,0xC7,0x69,0xA9,0xD7,0x51,0x94,0xF6,0x0B,0xD0,
		0x75,0x8C,0x3D,0xA6,0x92,0x7D,0x28,0x55,0xE9,0x2F,
		0xEC,0xA1,0x4D,0x85,0xC4,0x0F,0x05,0xC2,0xA3,0x3E,
		0x9E,0xE9,0x57,0xB6,0x1C,0x6D,0x57,0xCE,0xEB,0xE9,
		0x46,0x03,0x6B,0xAB,0xFC,0x82,0x5B,0x28,0x2F,0xA5,
		0x7E,0x6E,0xF9,0xED,0x80,0x21,0xF8,0x36,0x8D,0xBD,
		0xA5,0xBA,0xF4,0xA0,0x1F,0xA2,0xA9,0xD0,0x2B,0x9D,
		0x4F,0x84,0xC2,0xA3,0x6C,0x17,0x91,0x4C,0xED,0x60,
		0x0C,0x8B,0xAD,0xA8,0x4C,0x49,0xB0,0x62,0xFF,0xF3, /* 1500 */

		0xBB,0xD9,0xD7,0x35,0x2D,0x3B,0xA9,0x55,0x16,0xF6,
		0xA8,0xFB,0x0F,0xB6,0x4F,0x8A,0xC3,0x87,0xFF,0x7D,
		0xBC,0x5C,0x52,0xAB,0x9B,0xB5,0x73,0x19,0xE4,0xB4,
		0x50,0x26,0x30,0xF0,0xDB,0x18,0x3C,0x72,0x13,0xB8,
		0x38,0xEE,0xA3,0xB0,0xDE,0xD9,0x48,0x65,0xDB,0xF4,
		0x1A,0x03,0x92,0x71,0x97,0xAF,0xBD,0x6B,0xAA,0xBD,
		0x03,0x3C,0x6F,0x2A,0x00,0x41,0xAC,0xF6),
	.shared = chunk_from_chars(
		0x71,0x33,0xAF,0xF8,0xFC,0x9E,0x3B,0x14,0xE4,0x76,
		0x97,0x1D,0x96,0x51,0x97,0x6A,0x1B,0x41,0xA2,0x89,
		0xB5,0x4F,0xA6,0x04,0x0D,0xCC,0x82,0x0C,0x96,0xD5,
		0x55,0x00),
};

ke_test_vector_t mlkem1024_3 = {
	.method = ML_KEM_1024,
	.seed = chunk_from_chars(
		0x22,0x5D,0x5C,0xE2,0xCE,0xAC,0x61,0x93,0x0A,0x07,
		0x50,0x3F,0xB5,0x9F,0x7C,0x2F,0x93,0x6A,0x3E,0x07,
		0x54,0x81,0xDA,0x3C,0xA2,0x99,0xA8,0x0F,0x8C,0x5D,
		0xF9,0x22,0x3A,0x07,0x3E,0x7B,0x90,0xE0,0x2E,0xBF,
		0x98,0xCA,0x22,0x27,0xEB,0xA3,0x8C,0x1A),
	.pub_i = chunk_from_chars(
		0xC8,0x01,0x6E,0xEA,0xE4,0x6E,0x26,0x10,0xA5,0xFD,
		0x25,0x21,0x02,0xA4,0x80,0xCA,0xC2,0xBC,0x6E,0x78,
		0x60,0x62,0x02,0x59,0xE4,0x1C,0x64,0x59,0x15,0x84,
		0xF1,0xE6,0xCE,0x8E,0x33,0x12,0xDC,0x88,0x62,0xDD,
		0x80,0x6C,0xC2,0x18,0x1A,0x18,0xBB,0x91,0xFB,0x18,
		0x4D,0xCF,0x04,0x5B,0x03,0x08,0x07,0xA5,0x02,0x3C,
		0x4B,0x37,0x79,0xF1,0x4A,0x16,0x2E,0x09,0x38,0x74,
		0x3B,0x78,0xC7,0x14,0x93,0xBE,0xDA,0x78,0x26,0x02,
		0x2E,0x78,0xAC,0x10,0x0E,0x78,0x92,0xFC,0x10,0x4D,
		0x48,0xE2,0x5C,0xE9,0x26,0x10,0xC0,0x72,0xA3,0xE9, /* 100 */

		0xE8,0x20,0x8C,0x93,0xCC,0x22,0x5C,0x91,0x6A,0x29,
		0x41,0xC1,0x38,0x38,0xED,0xA3,0x82,0xF3,0x11,0xA8,
		0x09,0xC1,0x9B,0x9C,0xA6,0x0F,0x18,0x89,0x10,0xA1,
		0xA1,0xBE,0xDE,0x47,0x10,0xB6,0xB7,0x21,0x27,0xE4,
		0x8A,0x54,0x54,0xA4,0x57,0xC8,0x03,0x74,0x39,0x57,
		0x2C,0x55,0x83,0x78,0xCC,0x78,0x45,0x54,0x17,0x1F,
		0x6A,0x0C,0x78,0xC6,0x76,0xDB,0x8B,0xB8,0xAA,0x38,
		0x03,0x57,0xFC,0xC1,0xF6,0x18,0x59,0xB0,0xEC,0x72,
		0x73,0x0C,0x10,0xF8,0xC3,0x93,0x29,0x75,0xC7,0x0E,
		0x63,0x29,0xAA,0xD3,0x4A,0x56,0x79,0x7A,0x03,0xC0, /* 200 */

		0x29,0x7F,0xB4,0xAA,0x6B,0x97,0xA6,0x3B,0xE1,0x9F,
		0x5C,0xA6,0x31,0x8D,0x0B,0x2C,0x8B,0x4C,0x6D,0x9E,
		0xF8,0x08,0xEB,0xE3,0x64,0xD0,0xA4,0x1A,0x82,0xDC,
		0x76,0x8B,0xE4,0x7F,0x44,0xF3,0x00,0xBC,0x9B,0x2E,
		0x4C,0xCA,0x10,0x50,0xE2,0x1A,0x99,0x19,0xA9,0xB6,
		0x5A,0x07,0xA6,0xC8,0xBC,0x1C,0xA5,0x62,0x2A,0x44,
		0x11,0x17,0xA1,0x74,0x92,0x11,0x57,0x1C,0x46,0x2A,
		0x40,0xC9,0x4C,0x33,0x01,0x08,0x18,0x67,0xB3,0xA8,
		0x95,0x7F,0xF4,0x5C,0x1C,0x3F,0x7C,0x00,0x09,0x7A,
		0x35,0xAF,0x93,0x43,0x8D,0x74,0xC7,0x8C,0x06,0x1C, /* 300 */

		0x2D,0x64,0x5E,0x7F,0x6C,0x3B,0xE5,0xFA,0x6D,0xA7,
		0x49,0x08,0x5D,0xD1,0x24,0x61,0xFC,0x51,0x7E,0x0C,
		0x28,0xBA,0xF9,0x34,0x18,0x97,0x87,0x72,0xE4,0xC7,
		0x23,0xC0,0x71,0x90,0x89,0x15,0xA1,0xA5,0x80,0x70,
		0x50,0xC5,0x7D,0x2A,0x3D,0x56,0x59,0x7B,0x1E,0x1B,
		0xAE,0x4B,0x43,0x76,0x32,0xA2,0x52,0x2F,0x31,0x4B,
		0x39,0xFA,0x6A,0xB7,0xA0,0x47,0xE1,0x2B,0x65,0x02,
		0x02,0x7E,0x33,0x4B,0x71,0x5B,0x5B,0x0B,0x17,0xBB,
		0x76,0xAE,0x60,0x8A,0xCC,0x02,0x50,0xD9,0x1A,0x85,
		0x9C,0x8C,0x44,0x8B,0x27,0xBF,0x4C,0x44,0xB0,0x92, /* 400 */

		0x35,0x0E,0x26,0x0B,0x65,0xC2,0x0C,0x82,0x58,0x8C,
		0x8C,0x1E,0xF2,0x5A,0xFF,0xAA,0xAF,0x15,0x04,0x79,
		0xE8,0x08,0x01,0xC8,0x2C,0xBE,0xA7,0x3A,0x37,0xDD,
		0x1A,0x7D,0xF6,0xD2,0x68,0xA9,0xB0,0xC4,0xFC,0x75,
		0x31,0x58,0xC3,0x9E,0xA4,0xE9,0x09,0x60,0xC2,0x04,
		0x60,0x80,0x43,0x29,0xF5,0x8B,0xD4,0x60,0xA3,0x73,
		0x57,0xAE,0x65,0x15,0xBC,0xD5,0x05,0xC7,0xCD,0xA5,
		0x5B,0x26,0x69,0x01,0x7B,0xEC,0x3D,0x51,0x81,0x6E,
		0x70,0x01,0xB0,0xA6,0x48,0x37,0xF9,0xE0,0x7E,0xA2,
		0x67,0x35,0xE8,0x30,0x09,0x2A,0x64,0x8F,0x82,0x8C, /* 500 */

		0xB3,0x09,0x32,0x28,0xB9,0x8A,0x0C,0x92,0x65,0x6C,
		0x1A,0x14,0xCB,0x3A,0x30,0x69,0x29,0xC6,0x20,0x77,
		0x51,0xCE,0xBE,0x31,0xAE,0xA4,0xC0,0x37,0x2C,0xD5,
		0xC0,0x6C,0xC6,0x37,0xDD,0x9B,0x68,0x94,0xF0,0x0D,
		0xCB,0x37,0x96,0xD0,0x9C,0xA8,0x78,0x84,0x41,0x27,
		0x26,0xA2,0x5B,0xFA,0x9B,0xF6,0xEB,0x61,0x3E,0x1A,
		0x2A,0x19,0xA1,0x96,0xE1,0xE1,0x63,0x4E,0x1C,0xA0,
		0x34,0x7A,0x9B,0xCC,0x96,0x81,0x69,0x23,0x09,0x29,
		0x61,0xB6,0x9D,0xC0,0xC0,0x13,0xC1,0x91,0xF7,0xBA,
		0x4F,0xB0,0xB1,0x3C,0xE5,0x55,0x5E,0x8C,0x1B,0x07, /* 600 */

		0xD8,0x56,0x87,0x22,0x15,0x52,0x87,0x26,0x4B,0x5F,
		0xA9,0x90,0xE9,0x26,0x03,0xAE,0xD7,0x1D,0xF7,0x78,
		0x83,0x2E,0xE3,0xBF,0x9A,0x76,0x7F,0xA9,0xE6,0x50,
		0x42,0x0A,0x14,0xF5,0xAC,0x22,0x7D,0xF4,0x97,0xEF,
		0xAB,0x89,0x51,0x13,0x02,0xD2,0x31,0xA5,0x21,0xC5,
		0x35,0x3B,0xD4,0x25,0x30,0x6A,0x14,0xB7,0xD2,0xA5,
		0x66,0xAB,0x3B,0x32,0xAA,0xCF,0x35,0x3C,0x2B,0x04,
		0xA1,0x48,0xE4,0xC9,0x3C,0x3E,0x47,0x06,0x05,0x06,
		0x7D,0xC7,0x59,0x07,0x22,0xC4,0x52,0x24,0x5B,0x3D,
		0x75,0x02,0x05,0xD2,0xC0,0xAA,0xD0,0x3B,0x0E,0x2E, /* 700 */

		0xC5,0x35,0xBC,0x6C,0xC3,0x95,0x61,0x32,0xE3,0x29,
		0xA9,0x80,0xC8,0x49,0x5D,0x76,0x5E,0xD1,0xEB,0x07,
		0x1C,0x58,0x68,0x0A,0x11,0x85,0x9C,0x3C,0xC0,0xB1,
		0xE9,0x86,0x48,0x6A,0x5B,0x54,0x39,0x49,0x64,0x3B,
		0x83,0x45,0x2C,0x30,0x7D,0xF4,0x77,0x68,0x5A,0x4D,
		0x0A,0x70,0xA7,0x52,0x8A,0x55,0xA9,0x97,0xBE,0xA4,
		0x67,0x80,0xB2,0x62,0x52,0xAF,0x15,0x7F,0xE5,0xD8,
		0x74,0xCF,0x5C,0x42,0x0B,0x5B,0x1A,0x32,0x4A,0x93,
		0x2B,0xA5,0x22,0x91,0x42,0x8A,0x39,0x67,0x2B,0xB3,
		0x59,0xAB,0xC4,0xB5,0x4B,0xBF,0xAC,0x60,0x8A,0x01, /* 800 */

		0x35,0xCC,0x99,0xA3,0x8E,0x10,0x2D,0xA3,0xE0,0x88,
		0xE4,0xEC,0x5C,0x47,0x98,0x4A,0x63,0x6C,0x7C,0xB9,
		0x9A,0x6B,0x63,0x04,0xCB,0x03,0x50,0x05,0x26,0x18,
		0x47,0x1F,0x71,0x6B,0xE0,0x6B,0x89,0x90,0x87,0xAA,
		0x38,0x05,0x31,0x11,0xE3,0x5A,0x95,0x96,0x5C,0x04,
		0xA7,0xC4,0x2B,0xDA,0x23,0x19,0x59,0x88,0x19,0xAA,
		0x0A,0xEB,0x5C,0x39,0x18,0x10,0x79,0xF4,0x19,0x82,
		0xE0,0x52,0x23,0xC8,0xE8,0x9A,0xBB,0xC3,0x2F,0xC1,
		0x48,0x7F,0x3C,0xB9,0x9A,0x16,0x7B,0xA6,0x94,0xB0,
		0x66,0x8A,0x55,0x25,0xA8,0x84,0x72,0x43,0x9C,0x89, /* 900 */

		0x17,0x91,0x47,0xEF,0x83,0x10,0x22,0xB4,0x5F,0x6B,
		0x85,0x53,0xC8,0x15,0x0E,0x32,0xB5,0x42,0x1F,0x75,
		0x29,0x1C,0x83,0x0A,0x86,0x98,0xA9,0x16,0x40,0x68,
		0x07,0xF2,0x01,0x14,0x1A,0x0E,0xA1,0xCB,0x28,0x25,
		0x5A,0x9E,0xD7,0x09,0x50,0x2E,0xC7,0x85,0x1A,0x56,
		0xA8,0x6C,0xD3,0xB7,0xF0,0x57,0x26,0xED,0x6A,0x6E,
		0x50,0xFC,0x52,0x70,0x61,0x04,0xE8,0x29,0xB0,0x2F,
		0x7B,0x61,0xA1,0x85,0x6C,0x9B,0x44,0xB8,0xD1,0x21,
		0x47,0x68,0x85,0x1A,0xC1,0x90,0xB4,0x6B,0x61,0xC0,
		0x53,0xC3,0x71,0xA3,0x33,0xA9,0x5B,0x78,0x06,0xEE, /* 1000 */

		0xD1,0x6C,0x03,0x37,0x90,0xC1,0x56,0x3F,0xB8,0x23,
		0x20,0x4E,0x26,0xCE,0x3A,0xF8,0x77,0x50,0xCA,0x63,
		0x9E,0x60,0x5C,0xD4,0x35,0x02,0x43,0x36,0x75,0x1C,
		0xA0,0xBD,0xAB,0x99,0xCE,0x5B,0x45,0xCE,0xFA,0x04,
		0xCA,0xA2,0x9B,0xA3,0x3B,0x49,0x98,0xDA,0x0A,0xA8,
		0x32,0xEC,0x22,0x55,0x18,0x21,0xBA,0xC0,0x55,0x96,
		0x1A,0xCE,0xFA,0x07,0x7D,0xE5,0x44,0xBB,0xB8,0x02,
		0x11,0x87,0x8C,0x61,0x09,0xB5,0x12,0xB6,0xB4,0x85,
		0x0B,0x5A,0x8D,0x0B,0xD9,0xBE,0xA6,0x97,0x8A,0x0F,
		0x92,0x0C,0x1F,0xE8,0x60,0x1E,0x11,0x23,0xA9,0x8A, /* 1100 */

		0xA2,0x3C,0xFA,0x30,0x09,0x08,0x17,0xEE,0x84,0x9E,
		0x65,0x4A,0x03,0x87,0xA1,0x85,0x17,0x1A,0x43,0x04,
		0x77,0x58,0xA5,0x78,0xA0,0x2E,0xBB,0x55,0x79,0x04,
		0x10,0x6E,0xB3,0x09,0x6D,0x3C,0x1F,0x92,0x09,0x2F,
		0x02,0x89,0x1F,0x1D,0x3B,0x06,0x59,0xF2,0x84,0x18,
		0x8B,0x71,0x19,0xC1,0x48,0x96,0x68,0x8E,0xF2,0xBC,
		0xC8,0x08,0x6C,0x60,0x4F,0xC1,0x40,0xA8,0xFA,0x24,
		0xC9,0x59,0x5D,0x02,0x02,0x69,0xB0,0x35,0x5A,0xD9,
		0xA4,0x4C,0xBD,0x80,0x3B,0x59,0x01,0x0A,0x1B,0x85,
		0xB1,0x2F,0xC7,0x85,0xEF,0x0A,0x07,0x89,0xC1,0x0B, /* 1200 */

		0x3A,0x30,0x92,0x5C,0xF1,0x55,0xAE,0x86,0x82,0x85,
		0x9A,0x8F,0x81,0x64,0xCF,0x62,0xFC,0x08,0xDF,0x57,
		0x7E,0xAB,0x20,0x71,0xDB,0x87,0xB0,0x29,0xA1,0x6E,
		0xC9,0x03,0x03,0xF5,0xB9,0x71,0x17,0x76,0x1C,0x5F,
		0x20,0x03,0x31,0x10,0xB1,0xBD,0xF7,0x38,0x88,0x95,
		0xB3,0x0A,0xCA,0x86,0x2F,0x22,0xAD,0xDE,0x69,0x30,
		0x43,0x08,0x21,0xDC,0xCB,0x2B,0x1B,0xA4,0xCB,0x92,
		0x36,0x52,0x30,0x9B,0x4F,0x16,0x60,0x0F,0xC9,0xFB,
		0xCF,0xAA,0x85,0x53,0xCB,0x45,0xBB,0xB5,0x30,0x35,
		0xCD,0x28,0x52,0x6E,0xA8,0x44,0x98,0x44,0xAC,0x68, /* 1300 */

		0xB8,0x7A,0xA6,0x10,0x47,0xB9,0x42,0x44,0xF1,0x26,
		0xBF,0xE2,0x32,0x3A,0xF5,0xF3,0x0B,0xBA,0x9C,0x1B,
		0x96,0xA4,0x26,0xA1,0xD8,0x4C,0x2D,0xA3,0x5F,0xBC,
		0x00,0x1E,0x20,0xA5,0x17,0xD4,0x58,0xB4,0x29,0x1C,
		0xB3,0x63,0x94,0xCD,0x9A,0x07,0x4E,0x51,0xD8,0xA4,
		0x5C,0xE6,0x23,0x48,0x4A,0x3A,0xA7,0xD5,0x75,0x97,
		0x72,0x42,0xAE,0xD5,0x35,0xD4,0x6B,0x74,0xD8,0x64,
		0x5A,0xED,0x24,0x66,0x92,0xAC,0x1B,0x8B,0x05,0x5D,
		0x71,0x7C,0xCB,0xD5,0x97,0x69,0xFF,0x96,0x17,0xBC,
		0xC3,0xAA,0x7B,0xB9,0x36,0x6B,0xA5,0x1A,0x73,0x6B, /* 1400 */

		0x2C,0xE7,0x9B,0x97,0x1E,0x0C,0x20,0x4A,0xC6,0x2F,
		0xD0,0x97,0x55,0xB4,0xE4,0xB2,0x33,0xE9,0x4A,0x9C,
		0xD3,0x49,0x0F,0x40,0x49,0xCE,0x6C,0xC9,0x6C,0x7C,
		0x0F,0xA1,0x25,0x4C,0x4E,0xB6,0x77,0x48,0xB3,0x37,
		0xC4,0x1C,0x47,0x3F,0xAB,0x4A,0x94,0x43,0x64,0x53,
		0x1A,0x6F,0xE1,0x86,0x52,0x85,0x67,0x97,0x78,0x7C,
		0x80,0x4C,0xE1,0xC5,0x94,0x62,0xB8,0xD1,0x45,0xCA,
		0xE4,0xB9,0x72,0x32,0x44,0x0B,0xA9,0x9C,0xC3,0xA6,
		0x6C,0x1E,0x01,0x9B,0x53,0xFD,0xB2,0x72,0xD0,0x00,
		0xC8,0x5A,0xD6,0x3D,0x68,0x6A,0x90,0x7A,0x31,0x70, /* 1500 */

		0x0F,0xD4,0xA2,0x9E,0x59,0x36,0xA5,0x3A,0x4F,0x2D,
		0x24,0x86,0x75,0x18,0x51,0xED,0xD1,0xB7,0x01,0x15,
		0xAD,0xAB,0x07,0xA2,0x9B,0x15,0x59,0x45,0x66,0x22,
		0x1B,0x32,0x90,0x76,0xA5,0xA1,0x16,0x67,0x6D,0x35,
		0x34,0x04,0x49,0x65,0x01,0x4D,0xAA,0x0D,0x13,0xDA,
		0x17,0x85,0x81,0x61,0x9D,0xC0,0x6D,0xF4,0x0A,0x02,
		0x75,0x0B,0xEB,0x99,0x54,0x80,0xEE,0xE9),
	.pub_r = chunk_from_chars(
		0x78,0x22,0x77,0x04,0x50,0xAE,0xF9,0x7A,0x43,0x0E,
		0xBE,0xC3,0x9C,0x17,0x99,0x83,0xDB,0x49,0xB9,0x94,
		0xF1,0x9E,0x55,0xE9,0xB6,0xDF,0xC7,0x1C,0xD2,0x91,
		0xC6,0xFB,0x4E,0x47,0xAA,0x5F,0xF4,0x64,0x44,0x29,
		0x37,0x7E,0x4D,0xAD,0x4C,0x09,0xB7,0x31,0xAC,0x31,
		0x88,0x1B,0xD5,0x09,0xEF,0x0C,0xD2,0x97,0x71,0x57,
		0x59,0x16,0xCF,0x1C,0x4D,0xA9,0x45,0x79,0xD0,0x35,
		0x1A,0x8A,0x4C,0x11,0x19,0x22,0x00,0x6E,0x79,0xEF,
		0x37,0x26,0xD7,0x81,0xC2,0x17,0x1B,0x71,0x43,0x12,
		0x02,0x78,0x5D,0x05,0x31,0x92,0x3A,0xDA,0xC3,0xE2, /* 100 */

		0x84,0xDC,0x90,0x11,0x72,0xF4,0x8D,0x6D,0x15,0x11,
		0x81,0x8D,0x8B,0x14,0xEC,0x1E,0x71,0x43,0x6D,0x58,
		0x65,0x5B,0x76,0xF6,0x04,0x1B,0x37,0x59,0x92,0xA2,
		0x35,0xB3,0x47,0x6F,0x57,0xDF,0x3B,0x37,0xEB,0x1F,
		0xB2,0x3E,0xCA,0xE5,0x72,0x6E,0xE3,0x5C,0xBD,0x4B,
		0xE6,0xFD,0x40,0x76,0x55,0xB5,0xCD,0x50,0xFF,0x0F,
		0x2E,0x7B,0xE0,0xCC,0xFC,0x21,0xB2,0xB5,0x62,0x68,
		0xF4,0x83,0xFC,0x6F,0x53,0x22,0x85,0x65,0x1F,0x9D,
		0x76,0xAD,0x97,0x12,0x6C,0x6C,0x8D,0x14,0x03,0xFB,
		0xD7,0xA7,0x3C,0x77,0x97,0xEE,0x4D,0xF6,0x0E,0x3B, /* 200 */

		0xC3,0xE5,0xC9,0xC6,0xE2,0x01,0x93,0x22,0x0B,0x3D,
		0x51,0x61,0x23,0x94,0x46,0x49,0x86,0x9A,0x8E,0x3E,
		0x35,0x93,0xA2,0xF7,0xD2,0x36,0xE2,0x8F,0x7C,0x87,
		0x69,0x15,0x42,0x97,0x50,0xCE,0x23,0xA2,0xFE,0x30,
		0x8B,0xAD,0xDC,0x6F,0x8E,0x84,0xD6,0xBA,0x43,0x9B,
		0x0D,0x68,0x71,0x11,0x3E,0xE5,0xD1,0x42,0xF3,0xD9,
		0xDF,0x37,0x31,0xCB,0xF5,0xC2,0x4D,0x8C,0x39,0x25,
		0x0E,0x5E,0x8B,0x33,0x53,0x2E,0x28,0xDB,0x54,0xAC,
		0x36,0x32,0xF7,0x11,0x40,0xF1,0xB9,0x60,0x39,0xCE,
		0xE1,0xDA,0x91,0x5C,0x0B,0x48,0xF0,0x55,0x31,0xF6, /* 300 */

		0x6D,0xD8,0x19,0xDE,0x6F,0x61,0x77,0x3B,0x6C,0x09,
		0xB6,0xD4,0xD3,0x0F,0xEF,0x53,0x8E,0x0F,0xBA,0xC9,
		0x90,0xB2,0x09,0x65,0xDF,0x16,0xE2,0xB2,0xB7,0xA0,
		0xC1,0xAF,0xD4,0xB6,0x50,0x67,0x35,0x35,0x67,0x84,
		0xD7,0xB7,0xC3,0x69,0x60,0x3E,0xF0,0x91,0x9C,0xBF,
		0xD4,0x79,0xA5,0x38,0x7C,0xF3,0x17,0x06,0x3B,0xD3,
		0x51,0xC6,0x56,0x2D,0xA5,0x8B,0x8D,0x75,0x6E,0x4E,
		0x08,0x9F,0xE1,0x5B,0x92,0xE3,0xD2,0x4F,0x49,0xFC,
		0xA7,0x14,0xDC,0xBA,0x7B,0xAB,0xC3,0x0E,0x95,0x96,
		0xFD,0x75,0x68,0x05,0x73,0xA4,0x5D,0x05,0x66,0x89, /* 400 */

		0x56,0x6D,0x45,0x90,0xD0,0xA0,0x87,0x61,0xA7,0x4B,
		0xB2,0xDA,0x4D,0x5C,0x52,0x0C,0x0A,0xAC,0x7B,0xE6,
		0x7F,0xC3,0x46,0xEE,0x5D,0x22,0xAF,0xCE,0x7A,0xBF,
		0xE0,0xCB,0xD1,0x60,0x58,0x98,0x2F,0xE8,0x31,0xC1,
		0xE4,0x7F,0x6D,0xA0,0x46,0x26,0xC3,0x84,0x40,0xCF,
		0x92,0x2E,0x02,0x4F,0xDB,0x08,0x71,0xC8,0xCB,0x1A,
		0x32,0x71,0xBA,0x98,0x36,0x09,0x8D,0xE8,0xBD,0x65,
		0x16,0xAB,0x2C,0xA1,0xB7,0xF1,0xDB,0x09,0x1B,0xD3,
		0xAC,0xCF,0x0B,0x9D,0x21,0xD0,0x10,0xC8,0x1D,0x3F,
		0x5C,0xD6,0x8A,0x00,0x7C,0x3D,0xD4,0x67,0x43,0xF1, /* 500 */

		0x78,0xAC,0xAC,0x59,0x9A,0xBB,0x40,0x41,0x26,0xFB,
		0x23,0xD0,0xE4,0x43,0x99,0xEE,0xF1,0x46,0x85,0x3A,
		0x28,0x51,0x1B,0x07,0x43,0x57,0xC3,0x7C,0x35,0x99,
		0x5A,0x95,0x40,0x35,0x08,0xEB,0x20,0xE8,0x0B,0x80,
		0x40,0x21,0x71,0x12,0xBA,0x46,0xD2,0xCE,0x37,0x7B,
		0x63,0x8F,0xA4,0x02,0xCE,0x60,0x45,0x93,0x48,0x12,
		0x65,0xD9,0x30,0x66,0xFC,0x2A,0x11,0x93,0xF6,0x44,
		0x96,0x53,0x82,0xB3,0x37,0x39,0x6E,0x67,0x58,0x35,
		0xAE,0x50,0x79,0x44,0xF9,0x56,0xA3,0xB6,0xC6,0xE8,
		0x66,0x8D,0x1B,0xBF,0x0F,0xDA,0x07,0xC8,0x58,0x6C, /* 600 */

		0xDE,0xFF,0x4F,0x37,0xA3,0xD3,0xCD,0x81,0xDE,0x1C,
		0x99,0xDA,0xDE,0x64,0x98,0xDD,0xAA,0x8A,0xEE,0x4A,
		0x22,0x8B,0xA9,0xA5,0x5E,0x2B,0xF1,0x6F,0x33,0xC2,
		0x94,0x19,0x11,0xEA,0x19,0x83,0x2D,0x75,0x7C,0x1C,
		0x43,0x07,0xAA,0x88,0xA0,0xF5,0x05,0xB3,0x56,0x05,
		0x97,0xE2,0x8A,0x72,0xE9,0x56,0x62,0xC4,0xC9,0x5D,
		0x07,0x70,0x71,0x7B,0x4B,0xC7,0x98,0x9B,0x6D,0x75,
		0x8B,0xFE,0x6F,0xA9,0x2E,0x0E,0x49,0x42,0x01,0xDC,
		0x59,0x3B,0xB4,0xBD,0xB6,0x73,0xCF,0x25,0x85,0xC4,
		0xC4,0xB5,0xA8,0xB5,0xE3,0x74,0xC2,0x70,0xF7,0x02, /* 700 */

		0x2F,0x53,0x9E,0x7B,0xCD,0xF0,0xC8,0x3C,0xD7,0x45,
		0x6A,0x7C,0xFF,0x24,0x71,0xD7,0x69,0xAA,0x5E,0xAB,
		0x88,0xCA,0x99,0x84,0x42,0x16,0xBA,0x9C,0xD8,0x32,
		0xFD,0xB8,0xC1,0x2F,0xC7,0xA1,0xFE,0x3A,0x6A,0xE6,
		0x84,0x57,0x15,0x72,0x9C,0xE2,0x26,0xA3,0x01,0xE8,
		0x58,0xAD,0x4E,0xAF,0x11,0x4F,0xF4,0xE5,0x5E,0x8F,
		0xE1,0xDE,0x7F,0xA3,0x2D,0x2D,0xEC,0x81,0x09,0xC1,
		0xB9,0xDE,0x79,0xC3,0x85,0x25,0xC5,0x4F,0x35,0x10,
		0x95,0x89,0xEA,0xEB,0x64,0xCC,0x41,0x03,0xFB,0x56,
		0xAB,0x71,0x79,0xF7,0x69,0xD4,0x3B,0x56,0xD5,0xBD, /* 800 */

		0x33,0x44,0xCC,0x28,0xE4,0x89,0x6F,0x83,0x3D,0xAF,
		0xFC,0xCA,0x9A,0xB3,0x0A,0xD3,0x82,0x23,0x0B,0x21,
		0x1A,0x0E,0x61,0xCE,0xC0,0xB2,0x90,0xFD,0xC3,0x9B,
		0xFA,0xDD,0x25,0x86,0x5E,0x82,0xEE,0x4F,0x76,0xC8,
		0x3D,0x50,0x0F,0xC7,0x46,0x77,0x76,0xE3,0xAA,0xE7,
		0x6B,0xD7,0x4E,0xED,0xA2,0x81,0x75,0xD3,0x2E,0x00,
		0x6A,0x6E,0x00,0x6C,0x22,0x16,0xCC,0x7E,0x27,0x33,
		0x43,0xCC,0xF7,0x47,0x86,0x23,0xF9,0xBC,0x96,0x76,
		0x38,0x29,0x65,0xFE,0x17,0xB5,0xF2,0x83,0x21,0xD3,
		0x9B,0xE7,0xAC,0xDF,0x2B,0xB4,0xBF,0xE7,0xD7,0xCF, /* 900 */

		0xD0,0x27,0xB3,0x04,0x60,0xD0,0x1B,0x99,0x88,0xC1,
		0x19,0xEF,0x0B,0x0A,0xF1,0xA8,0x7D,0x61,0x9E,0x96,
		0x42,0xB6,0x2E,0xE8,0xC9,0x8A,0x1D,0xE8,0x8C,0x71,
		0x7D,0x43,0xEB,0x7B,0x43,0x4A,0x62,0x81,0x42,0xDD,
		0x2D,0x58,0x8B,0xAD,0xDC,0x96,0xBD,0x14,0xD2,0x76,
		0x0D,0xA8,0x06,0x20,0x66,0xE8,0x9C,0x87,0xAA,0x3C,
		0xDF,0x2E,0x4F,0xEA,0x1A,0xBF,0x89,0x59,0xCF,0x98,
		0x32,0xDD,0x53,0x0D,0xD0,0x82,0x07,0x35,0xAF,0x43,
		0x57,0xD0,0x03,0x46,0x6E,0xD5,0xA6,0x07,0x6B,0xB8,
		0x5F,0xC1,0x1E,0xE3,0x73,0xD8,0x36,0xD6,0x94,0xED, /* 1000 */

		0x8B,0xB9,0x44,0x4C,0x86,0xC7,0xBB,0x42,0x61,0xD1,
		0xB8,0x74,0x2F,0x9B,0xA3,0xDE,0x00,0x9C,0x3E,0xB4,
		0x6E,0x4C,0x26,0x34,0x40,0x20,0xA5,0x59,0xC4,0x42,
		0x56,0xB8,0xE1,0x41,0xC2,0xD2,0x33,0xC2,0x4F,0x57,
		0x39,0x4D,0xC5,0x7D,0x53,0x73,0x68,0xFF,0xF3,0x70,
		0x33,0x39,0xBB,0xF7,0xA4,0x84,0x56,0x29,0x84,0x2F,
		0x3E,0x5E,0xCB,0xCD,0x1B,0x6B,0xBF,0x5F,0x00,0x43,
		0xFF,0xF7,0xC5,0xD0,0xA1,0x6B,0x3A,0xD5,0x0B,0xA5,
		0x12,0x96,0x19,0x4B,0x94,0x6B,0x63,0xAA,0xD3,0x3B,
		0x04,0x05,0xA6,0x4D,0x98,0x49,0x2E,0xC0,0xF4,0xD3, /* 1100 */

		0x60,0x06,0xE9,0x9A,0xD5,0x3A,0xCB,0x57,0x55,0xF0,
		0xD9,0x40,0xB2,0xA1,0x74,0xE6,0xD8,0x8A,0x77,0x8E,
		0x3D,0xF1,0xB0,0x23,0x3C,0x3F,0x5E,0x20,0xB6,0xA9,
		0x3A,0x75,0x61,0x8B,0x9B,0xDE,0x83,0x91,0xC2,0x1E,
		0x4B,0xA7,0xBA,0x11,0x6C,0x50,0x5C,0x0F,0x79,0x81,
		0xD0,0x05,0x66,0xA2,0x82,0x62,0xD0,0x1A,0xA5,0x1D,
		0x4E,0x9E,0xBA,0xDA,0x5F,0x67,0xA2,0x5F,0x18,0x88,
		0x02,0x91,0x75,0x14,0x48,0xD0,0x38,0xB0,0x06,0x33,
		0x1E,0x35,0xB6,0xAA,0xEA,0x12,0x23,0xDA,0xB8,0x7D,
		0xB9,0x49,0xC1,0x25,0x53,0x78,0x7F,0xBB,0xB0,0xE9, /* 1200 */

		0x8E,0x29,0x85,0x94,0x29,0x0B,0x55,0xD6,0xE4,0x5E,
		0x49,0x01,0xDB,0x0A,0xAE,0x63,0x27,0xCB,0xFB,0xB4,
		0x90,0x16,0x38,0x36,0x2E,0x6F,0x8B,0xA3,0xD3,0xFC,
		0x67,0x3E,0xC4,0x99,0x11,0x05,0x3D,0x6F,0xD2,0x96,
		0x10,0x3C,0x2B,0x67,0xEA,0xCB,0x37,0x38,0x57,0x8B,
		0x5C,0xB9,0x27,0x12,0x3E,0xB7,0xFA,0xE5,0xD9,0x78,
		0xBB,0x93,0xA9,0x49,0x1E,0xD7,0x4F,0xB0,0xA9,0xE9,
		0xC3,0x42,0xD3,0x12,0x53,0x22,0x99,0xBB,0xA2,0xE6,
		0x14,0x31,0xA3,0x80,0x24,0x65,0x70,0x8F,0x6A,0x49,
		0x11,0x81,0xFB,0xFF,0x2C,0x1F,0x3B,0x35,0xD9,0x2D, /* 1300 */

		0x83,0x24,0x1C,0x10,0x3D,0xF7,0x6D,0xC5,0xBE,0x06,
		0xC6,0xD9,0x09,0x5E,0xB9,0xE6,0x7D,0xE6,0x96,0xDA,
		0x8C,0xA2,0xEE,0xD3,0x66,0x4B,0xEF,0x74,0x74,0x35,
		0x0C,0x96,0x07,0x57,0x4B,0x72,0x81,0x47,0xC7,0x27,
		0x97,0x89,0x27,0x52,0x30,0x68,0x50,0x92,0x61,0x48,
		0x68,0x34,0xFA,0xCF,0xCF,0xEE,0x6A,0x6C,0x69,0x96,
		0x33,0x9B,0xA8,0x93,0xE2,0x9D,0x61,0xFC,0x35,0x56,
		0x0D,0x10,0xA4,0x7A,0x6F,0x18,0xB4,0xED,0xF1,0xB7,
		0xAD,0x8B,0x95,0xA3,0x67,0xB1,0xDB,0x0A,0xEC,0x4A,
		0x3B,0x02,0x8F,0x20,0x8F,0xE1,0x72,0xC8,0x66,0xBF, /* 1400 */

		0x47,0xF2,0x74,0x9F,0x94,0xD8,0xFC,0x6D,0x8C,0xE0,
		0x8E,0xA5,0xBA,0x3D,0xF3,0x5E,0xA6,0xBE,0x06,0x32,
		0x1E,0x7B,0xD5,0x1C,0x93,0x65,0x97,0x50,0xF4,0x44,
		0x20,0x4F,0xFB,0x87,0x0D,0x31,0xD0,0xBA,0x53,0x4F,
		0xF5,0xF8,0xE0,0x0D,0x1D,0x9C,0xE5,0xD5,0x38,0xD7,
		0x15,0x11,0x2A,0x63,0x60,0x30,0x5D,0x26,0x37,0xC8,
		0x14,0xD8,0xDE,0x75,0xF4,0x0E,0x9E,0x54,0x2B,0xF3,
		0xAD,0xD2,0x90,0x29,0xAA,0x63,0x2B,0x8C,0xB0,0x11,
		0x34,0x03,0x9D,0xEE,0x78,0xB8,0x37,0x62,0xF1,0x25,
		0x4A,0x2F,0xAF,0x19,0xF3,0x60,0xD4,0xC3,0x35,0xB2, /* 1500 */

		0x52,0x92,0x1C,0x6C,0xF0,0xBE,0xDB,0x4D,0x77,0xD3,
		0x4B,0x01,0x2B,0x4C,0x45,0x42,0xD8,0xC5,0x41,0x1C,
		0x88,0x32,0x84,0x30,0x89,0x23,0xC7,0x67,0x1D,0x6F,
		0xF8,0x7D,0xF8,0xF7,0xB5,0xAE,0x44,0xF6,0x30,0x23,
		0xB0,0x46,0x5A,0x6D,0x9C,0x36,0x44,0x5B,0xB3,0x71,
		0x1D,0x1E,0x10,0xFB,0x47,0xDC,0x8B,0x77,0x2B,0x20,
		0x0E,0x96,0x2D,0x78,0x36,0xF0,0x7D,0x81),
	.shared = chunk_from_chars(
		0xED,0x1B,0xF2,0x17,0xD1,0x40,0x64,0xB5,0x16,0xC6,
		0x9B,0xEA,0x71,0x9B,0xB1,0x5E,0x14,0x5F,0x57,0xB1,
		0x04,0x75,0x07,0xE0,0x05,0xFE,0x93,0x58,0xF7,0x7E,
		0x57,0xA5),
};
