#ifndef AGENTREQUESTHANDLER_H
#define AGENTREQUESTHANDLER_H

#include "agent.h"
#include "mcpclient.h"

#include <QObject>
#include <QString>
#include <QJsonArray>
#include <QJsonObject>
#include <QQueue>
#include <QFuture>
#include <QtConcurrent>

namespace uos_ai {

class ToolParser;
class MCPServer;

class AgentRequestHandler : public Agent
{
    Q_OBJECT
public:
    /**
     * 构造函数
     * @param parent 父对象指针
     */
    explicit AgentRequestHandler(QObject *parent = nullptr);
    
    /**
     * 析构函数
     * 清理资源，关闭连接
     */
    ~AgentRequestHandler() override;

    /**
     * 获取智能体的MCPServer类
     * @returns {MCPServer} MCPServer对象的共享指针
     */
    virtual QSharedPointer<MCPServer> mcpServer() const;

    /**
     * 初始化客户端连接
     * 建立与MCP服务器的连接，加载配置信息
     * @returns {bool} 初始化是否成功，true表示成功，false表示失败
     */
    virtual bool initClient();
    
    /**
     * 获取可用的服务器列表
     * 查询当前智能体可以访问的所有MCP服务器
     * @returns {QStringList} 服务器名称列表，如果没有可用服务器则返回空列表
     */
    virtual QStringList listServers() const;
    
    /**
     * 获取可用的工具列表
     * 从所有连接的MCP服务器获取可用的工具定义
     * @returns {QPair<int, QJsonValue>} 状态码和工具定义数组
     *          状态码：0表示成功，非0表示错误
     *          QJsonValue：包含工具定义的JSON数组
     */
    virtual QPair<int, QJsonValue> listTools() const;

    /**
     * 获取工具信息
     * 从指定的服务器列表中获取详细的工具信息
     * @param {QStringList} servers - 要查询的服务器列表
     * @returns {QPair<int, QString>} 状态码和工具信息的JSON字符串
     *          状态码：0表示成功，非0表示错误
     *          QString：包含工具信息的JSON字符串
     */
    virtual QPair<int, QString> fetchTools(const QStringList &servers);

    /**
     * 处理用户请求
     * 处理来自用户的请求，包括文本对话和工具调用
     * @param {QJsonObject} question - 当前请求内容，包含用户的问题和相关参数
     * @param {QJsonArray} history - 聊天消息记录，包含之前的对话历史
     * @param {QSharedPointer<LLM>} llm - 模型服务的共享指针
     * @param {QJsonObject} params - 扩展参数，包含额外的配置信息
     */
    void processRequest(const QJsonObject &question, const QJsonArray &history, QSharedPointer<LLM> llm, const QVariantHash &params = {}) override;

public Q_SLOTS:
    /**
     * 取消对话
     */
    virtual void cancel();
protected:
    /**
     * 初始化聊天记录
     * 将用户请求和历史记录整合成完整的聊天消息数组
     * @param {QJsonObject} question - 当前请求内容
     * @param {QJsonArray} history - 历史消息记录
     * @returns {QJsonArray} 初始化后的消息数组，包含系统提示、历史记录和当前请求
     */
    virtual QJsonArray initChatMessages(const QJsonObject &question, const QJsonArray &history) const = 0;

    /**
     * 发送文本链式内容
     * 将文本内容以流式方式发送给客户端
     * @param {QString} content - 要发送的文本内容
     */
    void textChainContent(const QString &content);

    /**
     * 发送工具调用内容
     * 将工具调用信息发送给客户端
     * @param {ToolUse} tool - 要发送的工具调用对象，包含工具名称和参数
     */
    void toolUseContent(const ToolUse &tool);

    /**
     * 处理模型流式输出的内容
     * 解析模型的流式输出，识别文本内容和工具调用
     * @param {QString} content - 模型输出的当前内容块
     * @param {QString} current_output - 当前累积的输出内容（引用传递）
     * @param {QString} tool_results - 工具调用结果的累积内容（引用传递）
     * @param {ToolParser} parser - 工具解析器实例（引用传递）
     * @returns {bool} 处理是否成功，true表示成功处理，false表示处理失败
     */
    virtual bool handleModelOutput(const QString &content, QString &current_output, QString &tool_results, ToolParser &parser);

protected:
    /**
     * 同步调用MCP客户端的方法，避免事件循环嵌套
     * 在单独的线程中执行函数，避免阻塞主事件循环
     * @param {std::function<T()>} func - 要在工作线程中执行的函数对象
     * @returns {T} 函数执行结果
     * @note 此方法会阻塞当前线程直到函数执行完成
     */
    template<typename T>
    T syncCall(std::function<T()> func) const {
        T result;
        auto id = QThread::currentThreadId();
        QThread *thread = QThread::create([&func, &result, id]() {
            result = func();
        });

        thread->start();
        thread->wait();
        delete thread;
        return result;
    }

signals:
    /**
     * 当有新的聊天增量内容可读时发出此信号
     * @param {QString} deltaData - 增量数据内容，包含新的文本或工具调用结果
     */
    void readyReadChatDeltaContent(const QString &deltaData);

protected:
    McpClient *m_mcpClient;                  // MCP客户端实例，用于与MCP服务器通信
    mutable QSet<QString> m_toolList;       // 可用工具列表的缓存，避免重复查询
    QList<ToolUse> m_usedTool;              // 已使用的工具列表，用于跟踪工具调用历史
    bool canceled = false;                  // 是否取消
};

} // namespace uos_ai

#endif // AGENTREQUESTHANDLER_H 
