use crate::buffer::Buffer;

const MACHINE_TRANS_KEYS: &[u8] = &[
    8, 8, 4, 8, 5, 7, 5, 8, 4, 8, 6, 6, 16, 16, 4, 8,
    4, 13, 4, 8, 8, 8, 5, 7, 5, 8, 4, 8, 6, 6, 16, 16,
    4, 8, 4, 13, 4, 13, 4, 13, 8, 8, 5, 7, 5, 8, 4, 8,
    6, 6, 16, 16, 4, 8, 4, 8, 4, 13, 8, 8, 5, 7, 5, 8,
    4, 8, 6, 6, 16, 16, 4, 8, 4, 8, 5, 8, 8, 8, 1, 19,
    3, 17, 3, 17, 4, 17, 1, 16, 5, 10, 5, 10, 10, 10, 5, 10,
    1, 16, 1, 16, 1, 16, 3, 10, 4, 10, 5, 10, 4, 10, 5, 10,
    3, 10, 5, 10, 3, 17, 3, 17, 3, 17, 3, 17, 4, 17, 1, 16,
    3, 17, 3, 17, 4, 17, 1, 16, 5, 10, 10, 10, 5, 10, 1, 16,
    1, 16, 3, 10, 4, 10, 5, 10, 4, 10, 5, 10, 5, 10, 3, 10,
    5, 10, 3, 17, 3, 17, 4, 8, 3, 17, 3, 17, 4, 17, 1, 16,
    3, 17, 1, 16, 5, 10, 10, 10, 5, 10, 1, 16, 1, 16, 3, 10,
    4, 10, 5, 10, 3, 17, 4, 10, 5, 10, 5, 10, 3, 10, 5, 10,
    3, 17, 4, 13, 4, 8, 3, 17, 3, 17, 4, 17, 1, 16, 3, 17,
    1, 16, 5, 10, 10, 10, 5, 10, 1, 16, 1, 16, 3, 10, 4, 10,
    5, 10, 3, 17, 4, 10, 5, 10, 5, 10, 3, 10, 5, 10, 1, 17,
    3, 17, 1, 17, 4, 13, 5, 10, 10, 10, 5, 10, 1, 16, 3, 10,
    5, 10, 5, 10, 10, 10, 5, 10, 1, 16, 0
];

const MACHINE_KEY_SPANS: &[u16] = &[
    1, 5, 3, 4, 5, 1, 1, 5,
    10, 5, 1, 3, 4, 5, 1, 1,
    5, 10, 10, 10, 1, 3, 4, 5,
    1, 1, 5, 5, 10, 1, 3, 4,
    5, 1, 1, 5, 5, 4, 1, 19,
    15, 15, 14, 16, 6, 6, 1, 6,
    16, 16, 16, 8, 7, 6, 7, 6,
    8, 6, 15, 15, 15, 15, 14, 16,
    15, 15, 14, 16, 6, 1, 6, 16,
    16, 8, 7, 6, 7, 6, 6, 8,
    6, 15, 15, 5, 15, 15, 14, 16,
    15, 16, 6, 1, 6, 16, 16, 8,
    7, 6, 15, 7, 6, 6, 8, 6,
    15, 10, 5, 15, 15, 14, 16, 15,
    16, 6, 1, 6, 16, 16, 8, 7,
    6, 15, 7, 6, 6, 8, 6, 17,
    15, 17, 10, 6, 1, 6, 16, 8,
    6, 6, 1, 6, 16
];

const MACHINE_INDEX_OFFSETS: &[u16] = &[
    0, 2, 8, 12, 17, 23, 25, 27,
    33, 44, 50, 52, 56, 61, 67, 69,
    71, 77, 88, 99, 110, 112, 116, 121,
    127, 129, 131, 137, 143, 154, 156, 160,
    165, 171, 173, 175, 181, 187, 192, 194,
    214, 230, 246, 261, 278, 285, 292, 294,
    301, 318, 335, 352, 361, 369, 376, 384,
    391, 400, 407, 423, 439, 455, 471, 486,
    503, 519, 535, 550, 567, 574, 576, 583,
    600, 617, 626, 634, 641, 649, 656, 663,
    672, 679, 695, 711, 717, 733, 749, 764,
    781, 797, 814, 821, 823, 830, 847, 864,
    873, 881, 888, 904, 912, 919, 926, 935,
    942, 958, 969, 975, 991, 1007, 1022, 1039,
    1055, 1072, 1079, 1081, 1088, 1105, 1122, 1131,
    1139, 1146, 1162, 1170, 1177, 1184, 1193, 1200,
    1218, 1234, 1252, 1263, 1270, 1272, 1279, 1296,
    1305, 1312, 1319, 1321, 1328
];

const MACHINE_INDICIES: &[u8] = &[
    1, 0, 2, 3, 3, 4, 1, 0,
    3, 3, 4, 0, 3, 3, 4, 1,
    0, 5, 3, 3, 4, 1, 0, 6,
    0, 7, 0, 8, 3, 3, 4, 1,
    0, 2, 3, 3, 4, 1, 0, 0,
    0, 0, 9, 0, 11, 12, 12, 13,
    14, 10, 14, 10, 12, 12, 13, 10,
    12, 12, 13, 14, 10, 15, 12, 12,
    13, 14, 10, 16, 10, 17, 10, 18,
    12, 12, 13, 14, 10, 11, 12, 12,
    13, 14, 10, 10, 10, 10, 19, 10,
    11, 12, 12, 13, 14, 10, 10, 10,
    10, 20, 10, 22, 23, 23, 24, 25,
    21, 21, 21, 21, 26, 21, 25, 21,
    23, 23, 24, 27, 23, 23, 24, 25,
    21, 28, 23, 23, 24, 25, 21, 29,
    21, 30, 21, 22, 23, 23, 24, 25,
    21, 31, 23, 23, 24, 25, 21, 33,
    34, 34, 35, 36, 32, 32, 32, 32,
    37, 32, 36, 32, 34, 34, 35, 32,
    34, 34, 35, 36, 32, 38, 34, 34,
    35, 36, 32, 39, 32, 40, 32, 33,
    34, 34, 35, 36, 32, 41, 34, 34,
    35, 36, 32, 23, 23, 24, 1, 0,
    43, 42, 45, 46, 47, 48, 49, 50,
    24, 25, 44, 51, 52, 52, 26, 44,
    53, 54, 55, 56, 57, 44, 59, 60,
    61, 62, 4, 1, 58, 63, 58, 58,
    9, 58, 58, 58, 64, 58, 65, 60,
    66, 66, 4, 1, 58, 63, 58, 58,
    58, 58, 58, 58, 64, 58, 60, 66,
    66, 4, 1, 58, 63, 58, 58, 58,
    58, 58, 58, 64, 58, 45, 58, 58,
    58, 67, 68, 58, 1, 58, 63, 58,
    58, 58, 58, 58, 45, 58, 69, 69,
    58, 1, 58, 63, 58, 63, 58, 58,
    70, 58, 63, 58, 63, 58, 63, 58,
    58, 58, 58, 63, 58, 45, 58, 71,
    58, 69, 69, 58, 1, 58, 63, 58,
    58, 58, 58, 58, 45, 58, 45, 58,
    58, 58, 69, 69, 58, 1, 58, 63,
    58, 58, 58, 58, 58, 45, 58, 45,
    58, 58, 58, 69, 68, 58, 1, 58,
    63, 58, 58, 58, 58, 58, 45, 58,
    72, 7, 73, 74, 4, 1, 58, 63,
    58, 7, 73, 74, 4, 1, 58, 63,
    58, 73, 73, 4, 1, 58, 63, 58,
    75, 76, 76, 4, 1, 58, 63, 58,
    67, 77, 58, 1, 58, 63, 58, 67,
    58, 69, 69, 58, 1, 58, 63, 58,
    69, 77, 58, 1, 58, 63, 58, 59,
    60, 66, 66, 4, 1, 58, 63, 58,
    58, 58, 58, 58, 58, 64, 58, 59,
    60, 61, 66, 4, 1, 58, 63, 58,
    58, 9, 58, 58, 58, 64, 58, 79,
    80, 81, 82, 13, 14, 78, 83, 78,
    78, 20, 78, 78, 78, 84, 78, 85,
    80, 86, 82, 13, 14, 78, 83, 78,
    78, 78, 78, 78, 78, 84, 78, 80,
    86, 82, 13, 14, 78, 83, 78, 78,
    78, 78, 78, 78, 84, 78, 87, 78,
    78, 78, 88, 89, 78, 14, 78, 83,
    78, 78, 78, 78, 78, 87, 78, 90,
    80, 91, 92, 13, 14, 78, 83, 78,
    78, 19, 78, 78, 78, 84, 78, 93,
    80, 86, 86, 13, 14, 78, 83, 78,
    78, 78, 78, 78, 78, 84, 78, 80,
    86, 86, 13, 14, 78, 83, 78, 78,
    78, 78, 78, 78, 84, 78, 87, 78,
    78, 78, 94, 89, 78, 14, 78, 83,
    78, 78, 78, 78, 78, 87, 78, 83,
    78, 78, 95, 78, 83, 78, 83, 78,
    83, 78, 78, 78, 78, 83, 78, 87,
    78, 96, 78, 94, 94, 78, 14, 78,
    83, 78, 78, 78, 78, 78, 87, 78,
    87, 78, 78, 78, 94, 94, 78, 14,
    78, 83, 78, 78, 78, 78, 78, 87,
    78, 97, 17, 98, 99, 13, 14, 78,
    83, 78, 17, 98, 99, 13, 14, 78,
    83, 78, 98, 98, 13, 14, 78, 83,
    78, 100, 101, 101, 13, 14, 78, 83,
    78, 88, 102, 78, 14, 78, 83, 78,
    94, 94, 78, 14, 78, 83, 78, 88,
    78, 94, 94, 78, 14, 78, 83, 78,
    94, 102, 78, 14, 78, 83, 78, 90,
    80, 86, 86, 13, 14, 78, 83, 78,
    78, 78, 78, 78, 78, 84, 78, 90,
    80, 91, 86, 13, 14, 78, 83, 78,
    78, 19, 78, 78, 78, 84, 78, 11,
    12, 12, 13, 14, 78, 79, 80, 86,
    82, 13, 14, 78, 83, 78, 78, 78,
    78, 78, 78, 84, 78, 104, 48, 105,
    105, 24, 25, 103, 51, 103, 103, 103,
    103, 103, 103, 55, 103, 48, 105, 105,
    24, 25, 103, 51, 103, 103, 103, 103,
    103, 103, 55, 103, 106, 103, 103, 103,
    107, 108, 103, 25, 103, 51, 103, 103,
    103, 103, 103, 106, 103, 47, 48, 109,
    110, 24, 25, 103, 51, 103, 103, 26,
    103, 103, 103, 55, 103, 106, 103, 103,
    103, 111, 108, 103, 25, 103, 51, 103,
    103, 103, 103, 103, 106, 103, 51, 103,
    103, 112, 103, 51, 103, 51, 103, 51,
    103, 103, 103, 103, 51, 103, 106, 103,
    113, 103, 111, 111, 103, 25, 103, 51,
    103, 103, 103, 103, 103, 106, 103, 106,
    103, 103, 103, 111, 111, 103, 25, 103,
    51, 103, 103, 103, 103, 103, 106, 103,
    114, 30, 115, 116, 24, 25, 103, 51,
    103, 30, 115, 116, 24, 25, 103, 51,
    103, 115, 115, 24, 25, 103, 51, 103,
    47, 48, 105, 105, 24, 25, 103, 51,
    103, 103, 103, 103, 103, 103, 55, 103,
    117, 118, 118, 24, 25, 103, 51, 103,
    107, 119, 103, 25, 103, 51, 103, 111,
    111, 103, 25, 103, 51, 103, 107, 103,
    111, 111, 103, 25, 103, 51, 103, 111,
    119, 103, 25, 103, 51, 103, 47, 48,
    109, 105, 24, 25, 103, 51, 103, 103,
    26, 103, 103, 103, 55, 103, 22, 23,
    23, 24, 25, 120, 120, 120, 120, 26,
    120, 22, 23, 23, 24, 25, 120, 122,
    123, 124, 125, 35, 36, 121, 126, 121,
    121, 37, 121, 121, 121, 127, 121, 128,
    123, 125, 125, 35, 36, 121, 126, 121,
    121, 121, 121, 121, 121, 127, 121, 123,
    125, 125, 35, 36, 121, 126, 121, 121,
    121, 121, 121, 121, 127, 121, 129, 121,
    121, 121, 130, 131, 121, 36, 121, 126,
    121, 121, 121, 121, 121, 129, 121, 122,
    123, 124, 52, 35, 36, 121, 126, 121,
    121, 37, 121, 121, 121, 127, 121, 129,
    121, 121, 121, 132, 131, 121, 36, 121,
    126, 121, 121, 121, 121, 121, 129, 121,
    126, 121, 121, 133, 121, 126, 121, 126,
    121, 126, 121, 121, 121, 121, 126, 121,
    129, 121, 134, 121, 132, 132, 121, 36,
    121, 126, 121, 121, 121, 121, 121, 129,
    121, 129, 121, 121, 121, 132, 132, 121,
    36, 121, 126, 121, 121, 121, 121, 121,
    129, 121, 135, 40, 136, 137, 35, 36,
    121, 126, 121, 40, 136, 137, 35, 36,
    121, 126, 121, 136, 136, 35, 36, 121,
    126, 121, 122, 123, 125, 125, 35, 36,
    121, 126, 121, 121, 121, 121, 121, 121,
    127, 121, 138, 139, 139, 35, 36, 121,
    126, 121, 130, 140, 121, 36, 121, 126,
    121, 132, 132, 121, 36, 121, 126, 121,
    130, 121, 132, 132, 121, 36, 121, 126,
    121, 132, 140, 121, 36, 121, 126, 121,
    45, 46, 47, 48, 109, 105, 24, 25,
    103, 51, 52, 52, 26, 103, 103, 45,
    55, 103, 59, 141, 61, 62, 4, 1,
    58, 63, 58, 58, 9, 58, 58, 58,
    64, 58, 45, 46, 47, 48, 142, 143,
    24, 144, 58, 145, 58, 52, 26, 58,
    58, 45, 55, 58, 22, 146, 146, 24,
    144, 58, 63, 58, 58, 26, 58, 145,
    58, 58, 147, 58, 145, 58, 145, 58,
    145, 58, 58, 58, 58, 145, 58, 45,
    58, 71, 22, 146, 146, 24, 144, 58,
    63, 58, 58, 58, 58, 58, 45, 58,
    149, 148, 150, 150, 148, 43, 148, 151,
    148, 150, 150, 148, 43, 148, 151, 148,
    151, 148, 148, 152, 148, 151, 148, 151,
    148, 151, 148, 148, 148, 148, 151, 148,
    45, 120, 120, 120, 120, 120, 120, 120,
    120, 120, 52, 120, 120, 120, 120, 45,
    120, 0
];

const MACHINE_TRANS_TARGS: &[u8] = &[
    39, 45, 50, 2, 51, 5, 6, 53,
    57, 58, 39, 67, 11, 73, 68, 14,
    15, 75, 80, 81, 84, 39, 89, 21,
    95, 90, 98, 39, 24, 25, 97, 103,
    39, 112, 30, 118, 113, 121, 33, 34,
    120, 126, 39, 137, 39, 40, 60, 85,
    87, 105, 106, 91, 107, 127, 128, 99,
    135, 140, 39, 41, 43, 8, 59, 46,
    54, 42, 1, 44, 48, 0, 47, 49,
    52, 3, 4, 55, 7, 56, 39, 61,
    63, 18, 83, 69, 76, 62, 9, 64,
    78, 71, 65, 17, 82, 66, 10, 70,
    72, 74, 12, 13, 77, 16, 79, 39,
    86, 26, 88, 101, 93, 19, 104, 20,
    92, 94, 96, 22, 23, 100, 27, 102,
    39, 39, 108, 110, 28, 35, 114, 122,
    109, 111, 124, 116, 29, 115, 117, 119,
    31, 32, 123, 36, 125, 129, 130, 134,
    131, 132, 37, 133, 39, 136, 38, 138,
    139
];

const MACHINE_TRANS_ACTIONS: &[u8] = &[
    1, 0, 2, 0, 2, 0, 0, 2,
    2, 2, 3, 2, 0, 2, 0, 0,
    0, 2, 2, 2, 2, 4, 2, 0,
    5, 0, 5, 6, 0, 0, 5, 2,
    7, 2, 0, 2, 0, 2, 0, 0,
    2, 2, 8, 0, 11, 2, 2, 5,
    0, 12, 12, 0, 2, 5, 2, 5,
    2, 0, 13, 2, 0, 0, 2, 0,
    2, 2, 0, 2, 2, 0, 0, 2,
    2, 0, 0, 0, 0, 2, 14, 2,
    0, 0, 2, 0, 2, 2, 0, 2,
    2, 2, 2, 0, 2, 2, 0, 0,
    2, 2, 0, 0, 0, 0, 2, 15,
    5, 0, 5, 2, 2, 0, 5, 0,
    0, 2, 5, 0, 0, 0, 0, 2,
    16, 17, 2, 0, 0, 0, 0, 2,
    2, 2, 2, 2, 0, 0, 2, 2,
    0, 0, 0, 0, 2, 0, 18, 18,
    0, 0, 0, 0, 19, 2, 0, 0,
    0
];

const MACHINE_TO_STATE_ACTIONS: &[u8] = &[
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 9,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0
];

const MACHINE_FROM_STATE_ACTIONS: &[u8] = &[
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 10,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0
];

const MACHINE_EOF_TRANS: &[u8] = &[
    1, 1, 1, 1, 1, 1, 1, 1,
    1, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 22, 22, 28, 22, 22,
    22, 22, 22, 22, 33, 33, 33, 33,
    33, 33, 33, 33, 33, 1, 43, 0,
    59, 59, 59, 59, 59, 59, 59, 59,
    59, 59, 59, 59, 59, 59, 59, 59,
    59, 59, 59, 59, 79, 79, 79, 79,
    79, 79, 79, 79, 79, 79, 79, 79,
    79, 79, 79, 79, 79, 79, 79, 79,
    79, 79, 79, 79, 79, 104, 104, 104,
    104, 104, 104, 104, 104, 104, 104, 104,
    104, 104, 104, 104, 104, 104, 104, 104,
    104, 121, 121, 122, 122, 122, 122, 122,
    122, 122, 122, 122, 122, 122, 122, 122,
    122, 122, 122, 122, 122, 122, 122, 104,
    59, 59, 59, 59, 59, 59, 59, 149,
    149, 149, 149, 149, 121
];

#[derive(Clone, Copy)]
pub enum SyllableType {
    ConsonantSyllable = 0,
    VowelSyllable,
    StandaloneCluster,
    SymbolCluster,
    BrokenCluster,
    NonIndicCluster,
}

pub fn find_syllables_indic(buffer: &mut Buffer) {
    let mut cs = 39usize;
    let mut ts = 0;
    let mut te = 0;
    let mut act = 0;
    let mut p = 0;
    let pe = buffer.len;
    let eof = buffer.len;
    let mut syllable_serial = 1u8;
    let mut reset = true;
    let mut slen;
    let mut trans = 0;
    if p == pe {
        if MACHINE_EOF_TRANS[cs] > 0 {
            trans = (MACHINE_EOF_TRANS[cs] - 1) as usize;
        }
    }

    loop {
        if reset {
            if MACHINE_FROM_STATE_ACTIONS[cs] == 10 {
                ts = p;
            }

            slen = MACHINE_KEY_SPANS[cs] as usize;
            let cs_idx = ((cs as i32) << 1) as usize;
            let i = if slen > 0 &&
                MACHINE_TRANS_KEYS[cs_idx] <= buffer.info[p].indic_category() as u8 &&
                buffer.info[p].indic_category() as u8 <= MACHINE_TRANS_KEYS[cs_idx + 1]
            {
                (buffer.info[p].indic_category() as u8 - MACHINE_TRANS_KEYS[cs_idx]) as usize
            } else {
                slen
            };
            trans = MACHINE_INDICIES[MACHINE_INDEX_OFFSETS[cs] as usize + i] as usize;
        }
        reset = true;

        cs = MACHINE_TRANS_TARGS[trans] as usize;

        if MACHINE_TRANS_ACTIONS[trans] != 0 {
            match MACHINE_TRANS_ACTIONS[trans] {
                2 => te = p + 1,
                11 => {
                    te = p + 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::NonIndicCluster, buffer);
                }
                13 => {
                    te = p;
                    p -= 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::ConsonantSyllable, buffer);
                }
                14 => {
                    te = p;
                    p -= 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::VowelSyllable, buffer);
                }
                17 => {
                    te = p;
                    p -= 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::StandaloneCluster, buffer);
                }
                19 => {
                    te = p;
                    p -= 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::SymbolCluster, buffer);
                }
                15 => {
                    te = p;
                    p -= 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::BrokenCluster, buffer);
                }
                16 => {
                    te = p;
                    p -= 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::NonIndicCluster, buffer);
                }
                1 => {
                    p = te - 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::ConsonantSyllable, buffer);
                }
                3 => {
                    p = te - 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::VowelSyllable, buffer);
                }
                7 => {
                    p = te - 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::StandaloneCluster, buffer);
                }
                8 => {
                    p = te - 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::SymbolCluster, buffer);
                }
                4 => {
                    p = te - 1;
                    found_syllable(ts, te, &mut syllable_serial, SyllableType::BrokenCluster, buffer);
                }
                6 => {
                    match act {
                        1 => {
                            p = te - 1;
                            found_syllable(ts, te, &mut syllable_serial, SyllableType::ConsonantSyllable, buffer);
                        }
                        5 => {
                            p = te - 1;
                            found_syllable(ts, te, &mut syllable_serial, SyllableType::BrokenCluster, buffer);
                        }
                        6 => {
                            p = te - 1;
                            found_syllable(ts, te, &mut syllable_serial, SyllableType::NonIndicCluster, buffer);
                        }
                        _ => {}
                    }
                }
                18 => {
                    te = p + 1;
                    act = 1;
                }
                5 => {
                    te = p + 1;
                    act = 5;
                }
                12 => {
                    te = p + 1;
                    act = 6;
                }
                _ => {}
            }
        }

        if MACHINE_TO_STATE_ACTIONS[cs] == 9 {
            ts = 0;
        }

        p += 1;
        if p != pe {
            continue;
        }

        if p == eof {
            if MACHINE_EOF_TRANS[cs] > 0 {
                trans = (MACHINE_EOF_TRANS[cs] - 1) as usize;
                reset = false;
                continue;
            }
        }

        break;
    }
}

#[inline]
fn found_syllable(
    start: usize,
    end: usize,
    syllable_serial: &mut u8,
    kind: SyllableType,
    buffer: &mut Buffer,
) {
    for i in start..end {
        buffer.info[i].set_syllable((*syllable_serial << 4) | kind as u8);
    }

    *syllable_serial += 1;

    if *syllable_serial == 16 {
        *syllable_serial = 1;
    }
}
