package org.jbpm.gd.jpdl.ui.properties;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jdt.core.Flags;
import org.eclipse.jdt.core.IField;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.jpdl.model.ConfigInfoElement;
import org.jbpm.gd.jpdl.model.Delegation;
import org.jbpm.gd.jpdl.model.JpdlElementFactory;
import org.jbpm.gd.jpdl.ui.JpdlLogger;
import org.jbpm.gd.jpdl.ui.util.AutoResizeTableLayout;

public class DelegationConfigurationTable {
	
	public static final Policy BEAN_POLICY = new BeanPolicy();
	public static final Policy FIELD_POLICY = new FieldPolicy();

	private TabbedPropertySheetWidgetFactory widgetFactory;
	private DelegationConfigurationComposite delegationConfigurationComposite;
	
	
	private Policy policy;
	private Table table;
	private HashMap configInfoElementsMap = new HashMap();

	public static DelegationConfigurationTable create(
			TabbedPropertySheetWidgetFactory widgetFactory, DelegationConfigurationComposite delegationConfigurationComposite, Policy policy) {
		DelegationConfigurationTable result = new DelegationConfigurationTable();
		result.widgetFactory = widgetFactory;
		result.delegationConfigurationComposite = delegationConfigurationComposite;
		result.policy = policy;
		result.create();
		return result;
	}

	private void create() {
		table = widgetFactory.createTable(delegationConfigurationComposite.getParent(), SWT.BORDER | SWT.CHECK | SWT.FULL_SELECTION);
		table.setVisible(false);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		table.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				handleTableItemSelection(e);
			}			
		});
		AutoResizeTableLayout handlerConfigBeanTableLayout = new AutoResizeTableLayout(table);
		handlerConfigBeanTableLayout.addColumnData(new ColumnWeightData(40));
		handlerConfigBeanTableLayout.addColumnData(new ColumnWeightData(60));
		table.setLayout(handlerConfigBeanTableLayout);
		TableColumn handlerConfigBeanTableNameColumn = new TableColumn(table, SWT.NONE);
		handlerConfigBeanTableNameColumn.setText("Name");
		TableColumn handlerConfigBeanTableValueColumn = new TableColumn(table, SWT.NONE);
		handlerConfigBeanTableValueColumn.setText("Value");
		createTableViewer();
	}
	
	private void createTableViewer() {
		CheckboxTableViewer tableViewer = new CheckboxTableViewer(table);
		tableViewer.setColumnProperties(new String[] { "name", "value" });
		tableViewer.setCellEditors(new CellEditor[] { new TextCellEditor(table), new TextCellEditor(table) });
		tableViewer.setCellModifier(new CellModifier(table));
	}
	
	public void setLayoutData(Object layoutData) {
		table.setLayoutData(layoutData);
	}
	
	public void setVisible(boolean visible) {
		if (!visible) {
			table.setVisible(false);
		} else if (table.getItemCount() == 0) {
			table.setVisible(false);
			delegationConfigurationComposite.getConfigInfoLabel().setVisible(false);
			delegationConfigurationComposite.getMessageLabel().setText(policy.getMessage());
			delegationConfigurationComposite.getMessageLabel().setVisible(true);
		} else {
			table.setVisible(true);
			delegationConfigurationComposite.getConfigInfoLabel().setVisible(true);
			delegationConfigurationComposite.getMessageLabel().setVisible(false);
			updateDelegationConfigInfo();
		}
	}
	
	public void updateDelegationConfigInfo() {
		Delegation delegation = delegationConfigurationComposite.getDelegation();
		Iterator iterator = configInfoElementsMap.keySet().iterator();
		while (iterator.hasNext()) {
			delegation.addConfigInfoElement((ConfigInfoElement)configInfoElementsMap.get(iterator.next()));
		}
	}

	public void refreshDelegationInfo(IType type) {
		configInfoElementsMap.clear();
		List items = policy.getItems(type);
		Delegation delegation = delegationConfigurationComposite.getDelegation();
		if (policy.getType().equals(delegation.getConfigType())) {
			ConfigInfoElement[] configInfoElements = delegation.getConfigInfoElements();
			for (int i = 0; i < configInfoElements.length; i++) {
				configInfoElementsMap.put(configInfoElements[i].getName(), configInfoElements[i]);
			}
		}
		table.removeAll();
		for (int i = 0; i < items.size(); i++) {
			TableItem item = new TableItem(table, SWT.NONE);
			item.setText(0, (String)items.get(i));
			ConfigInfoElement configInfoElement = (ConfigInfoElement)configInfoElementsMap.get(items.get(i));
			if (configInfoElement != null) {
				item.setText(1, configInfoElement.getValue());
				item.setChecked(true);
			}
		}
		setVisible(policy.getType().equals(delegation.getConfigType()));
	}
	
	public void removeAll() {
		table.removeAll();
	}
	
	private void handleTableItemSelection(SelectionEvent event) {
		if (event.detail != SWT.CHECK) return;
		if (event.item instanceof TableItem) {
			TableItem item = (TableItem)event.item;
			if (item.getChecked()) {
				addConfigInfoElement(item.getText(0), item.getText(1));
			} else {
				removeConfigInfoElement(item.getText(0));
			}
		}
	}
	
	private void addConfigInfoElement(String name, String value) {
		Delegation delegation = delegationConfigurationComposite.getDelegation();
		ConfigInfoElement configInfoElement = (ConfigInfoElement)JpdlElementFactory.INSTANCE.createById("org.jbpm.gd.jpdl.core.configInfoElement");
		configInfoElement.setName(name);
		configInfoElement.setValue(value);
		configInfoElementsMap.put(name, configInfoElement);
		delegation.addConfigInfoElement(configInfoElement);
	}
	
	private void removeConfigInfoElement(String name) {
		Delegation delegation = delegationConfigurationComposite.getDelegation();
		ConfigInfoElement configInfoElement = (ConfigInfoElement)configInfoElementsMap.get(name);
		if (configInfoElement != null) {
			delegation.removeConfigInfoElement(configInfoElement);
			configInfoElementsMap.remove(configInfoElement.getName());
		}
	}
	
	private static abstract class Policy {
		protected abstract String getType();
		protected abstract List getItems(IType type);
		protected abstract String getMessage();
	}
	
	private static class BeanPolicy extends Policy {
		
		protected String getType() {
			return "bean";
		}
		
		protected List getItems(IType type) {
			return getSetters(type);
		}
		
		protected String getMessage() {
			return "The class does not have any setters";
		}
		
		private List getSetters(IType type) {
			List result = new ArrayList();
			try {
				IMethod[] methods = type.getMethods();
				for (int i = 0; i < methods.length; i++) {
					if (methods[i].getElementName().startsWith("set")) {
						StringBuffer buff = new StringBuffer(methods[i].getElementName().substring(3));
						buff.setCharAt(0, Character.toLowerCase(buff.charAt(0)));
						result.add(buff.toString());
					}
				}
			} catch (JavaModelException  e) {
				JpdlLogger.logError("Error while getting the setters for type " + type + ".", e);
			}
			return result;
		}
	}
	
	private static class FieldPolicy extends Policy {
		
		protected String getType() {
			return "field";
		}
		
		protected List getItems(IType type) {
			return getFields(type);
		}
		
		protected String getMessage() {
			return "The class does not have any fields";
		}
		
		private List getFields(IType type) {
			List result = new ArrayList();
			try {
				IField[] fields = type.getFields();
				for (int i = 0; i < fields.length; i++) {
					if (!Flags.isStatic(fields[i].getFlags())) {
						result.add(fields[i].getElementName());
					}
				}
			} catch (JavaModelException  e) {
				JpdlLogger.logError("Error while getting the fields for type " + type + ".", e);
			}
			return result;
		}
		
	}
		
	private class CellModifier implements ICellModifier {
		
		Table t;
		
		public CellModifier(Table table) {
			this.t = table;
		}
		
		public boolean canModify(Object element, String property) {
			return "value".equals(property);
		}

		public Object getValue(Object element, String property) {
			TableItem selection = t.getSelection()[0];
			if ("name".equals(property)) {
				return selection.getText(0);
			} else {
				return selection.getText(1);
			}
		}

		public void modify(Object element, String property, Object value) {
			TableItem selection = t.getSelection()[0];
			if ("name".equals(property)) {
				selection.setText(0, (String)value);
			} else {
				selection.setText(1, (String)value);
				if (selection.getChecked()) {
					ConfigInfoElement configInfoElement = (ConfigInfoElement)configInfoElementsMap.get(selection.getText(0));
					configInfoElement.setValue((String)value);
				}
			}
		}
		
	}
	
}
