{

    This file is the source for a series of routines to detect the make and
    model of the CPU the program is running on.
    Copyright (C) 1998 by Phil Brutsche

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the
    Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA  02111-1307  USA.

}

{$E-}{$N-} { DO NOT link in FPU support or emulation!  That's what the
             assembly modules are for! }
{$M 65520, 0, 25000}{$G-}{$V-}{$P+}{$R-}{$S-}{$I-}{$Q-}
program compinfo;

uses screen, lists, cursor, strs, pci, pcilist, os, mca,
     video, enhkey, crt, strings, dos, printer, cpu;

const
  compinfo_release_date = 'Dec 26 1998';

const
  esc = #27;
  ctrl_left = #115;
  ctrl_right = #116;
  pgup = #73;
  pgdn = #81;
  space = #32;
  tab = #9;
  nul = #0;
  leftarrow = #75;
  rightarrow = #77;
  endkey = #79;
  home = #71;
  cr = #$0A;
  lf = #$0D;
  crlf = #$0A#$0D;
  backspace = #8;
  delkey = #83;
  uparrow = #72;
  downarrow = #80;
  F1 = #$3b;
{  F3 = #$3d;                   { get the last command }
{  F4 = #$3e;                   { get and execute the last command }
{  F6 = #$40;                   { delete everything after the current position }
{  F7 = #$41;                   { list the command line history }
{  F8 = #$42;                   { select an item from the command line history }
{  F9 = #$43;                   { clear the command line history }

const
  indent = '     ';

type
  TFCB = Record
    Drive: Byte;
    Name: array [0..7] of Char;
    Ext: array [0..2] of Char;
    CurBlock: Word;
    RecSize: Word;
    FileSize: LongInt;
    FileDate: Word;
    FileTime: Word;
    Reserved: array [0..7] of Char;
    CurRec: Byte;
    RandRec: LongInt;
  end;

  TDBP = Record
    Drive: Byte;
    AUnit: Byte;
    SectorSize: Word;
    Rest: array [0..28] of Byte;
  end;
  PDBP = ^TDBP;

var
  fpu_type, cputype, check_fpu : byte;
  running_winnt, verbose, present : boolean;
  xmsdriver : pointer;
  xmserror : byte;
  param : string [90];
  out : ^text;
  outfile : text;
  fp_status, old_off, old_seg : word;
  outputlist : list;
  outputstring : string;
  tempstr : string [80];
  {video_mode : word;}
  outputfile : text;

function cpu_speed (processor : word) : longint; forward;

procedure clear_screen;
var
  current_page : byte;
begin
  asm
    { get current page }
    mov ah, $0f
    int $10
    { clear the screen }
    mov current_page, bh
    mov ax, $0600
    mov bh, $07
    mov ch, 0
    mov cl, 0
    mov dh, 24
    mov dl, 79
    int $10
    { set the cursor position to origin }
    mov ah, $02
    mov bh, current_page
    mov dl, 0
    mov dh, 0
    int $10
  end;
end;

function versionString (major, minor : byte) : string;
var
  tempstr1 : string [5];
  tempstr2 : string [10];
begin
  versionString := numstr (major) + '.' + numstr (minor);
end;

procedure verbose_writeln (str : string);
begin
  if verbose then begin
    writeln (outputfile, str);
    close (outputfile);
    append (outputfile);
  end;
end;

procedure verbose_write (str : string);
begin
  if verbose then begin
    write (outputfile, str);
    close (outputfile);
    append (outputfile);
  end;
end;

var
  Sx: array[0..80] of Char;
  FCBx: TFCB;
  DBP: PDBP;

function ISOK(Drive: Byte): Boolean; assembler;
asm
  push ds
  mov dl, Drive
  mov ah, 32h
  int 21h
  cmp al, $FF
  jz  @error
  mov cx, ds
  mov es, cx
  pop ds
  mov word ptr DBP, bx
  mov word ptr DBP + 2, es
  mov al, 1
  jmp @Ok
@error:
  pop ds
  mov al, 0
@Ok:
end;

function GetInfo: Boolean; assembler;
asm
  push bp
  push ds
  mov si, seg Sx
  mov ds, si
  mov si, offset sx
  mov di, seg FCBx
  mov es, di
  mov di, offset fcbx
  mov al, 1
  mov ah, 29h
  int 21h
  mov bl, al
  mov ax, 1
  cmp bl, $FF
  jnz @Done
  mov ax, 0
@Done:
  pop ds
  pop bp
end;

procedure _GetDrives;
var
  S1: String;
  i: Integer;
  bad: Boolean;
  S: PChar;
  normal, network : string [26];
begin
  verbose_writeln ('Gather fixed disk info:');
  verbose_writeln ('Allocating memory for drive info...');
  GetMem(S, 80);
  S1 := 'c:*.*';
  normal := '';
  network := '';
  FillChar (FCBx, SizeOf(TFCB), #0);
  outputlist.ins (indent + 'Fixed disks:');
  for i := 2 to 25 do begin
    S1[1] := Chr(i + 65);
    move(S1[1], Sx, Length(S1));
    S[Length(S1)] := #0;
    Bad := GetInfo;
    outputstring := indent + indent + 'Drive ' + s1 [1] + ':';
    if bad then begin
      if (i = 0) or (i = 1) then
        {normal := normal + s1 [1]}
        outputstring := outputstring + '  Local'
      else begin
        if IsOk (i + 1) then
          outputstring := outputstring + '  Local'
        else
          outputstring := outputstring + '  Network';
      end;
      if outputstring = '' then
        outputlist.ins ('None')
      else
        outputlist.ins (outputstring);
      verbose_write (skipspaces (outputstring) + ' ');
    end;
  end;
  verbose_writeln ('Freeing memory...');
  FreeMem(S, 80);
end;

procedure getdrives;

const
  drive_types : array [0..6] of string [6] = (
    'None', '360kb', '1.2mb', '720kb', '1.44mb', '2.88mb', '2.88mb' );

var
  result : byte;
  q : integer;
  serialnumber : string [19];
  firmware : string [7];
  modelname : string [39];
  outputstring : string;
begin
  outputlist.ins ('Disk drives:');
  port [$70] := $10;
  result := port [$71];
  outputstring := '';
  outputlist.ins (indent + 'Floppy drives:');
  verbose_writeln ('Floppy info from BIOS:  ' + dectohex8 (result));
  if result = $00 then begin
    outputlist.ins (indent + indent + 'None');
    exit;
  end;
  outputlist.ins (indent + indent + 'Drive A:  ' + drive_types [(result and $F0) shr 4]);
  outputlist.ins (indent + indent + 'Drive B:  ' + drive_types [result and $0F]);
  _GetDrives;
end;

{-----------------------------------------------------------------------------}
function getpartname (parttype : byte) : string;
begin
  case parttype of
    $00 : getpartname := 'None';
    $01 : getpartname := '12-bit FAT';
    $02 : getpartname := 'Xenix root file system';
    $03 : getpartname := 'Xenix /usr file system (obsolete)';
    $04 : getpartname := 'DOS 16-bit FAT (<32M)';
    $05 : getpartname := 'DOS 16-bit FAT';
    $06 : getpartname := 'DOS 16-bit FAT (>32M)';
    $07 : getpartname := 'OS/2 HPFS/WinNT NTFS';
    {07 : getpartname := 'Windows NT NTFS';
    $07 : getpartname := 'Advanced Unix';}
    $08 : getpartname := 'OS/2 v1.0-1.3';
    {08 : getpartname := 'AIX bootable';
    $08 : getpartname := 'Commodore DOS';
    $08 : getpartname := 'DELL multi-drive';}
    {09 : getpartname := 'AIX data';}
    $09 : getpartname := 'Coherent filesystem';
    $0A : getpartname := 'OS/2 Boot Manager';
    {0A : getpartname := 'OPUS';
    $0A : getpartname := 'Coherent swap';}
    $0B : getpartname := 'Windows95 with FAT32';
    $0C : getpartname := 'Windows95 with FAT32 (using LBA-mode)';
    $0E : getpartname := 'LBA VFAT';
    $0F : getpartname := 'LBA VFAT >32M';
    $10 : getpartname := 'OPUS';
    $11 : getpartname := 'OS/2 Boot Manager 12-bit FAT';
    $12 : getpartname := 'Compaq Diagnostics';
    {14 : getpartname := 'resulted from using Novell DOS 7.0 FDISK to delete Linux Native part)';}
    $14 : getpartname := 'OS/2 Boot Manager <32M 16-bit FAT';
    $16 : getpartname := 'OS/2 Boot Manager >32M 16-bit FAT';
    $17 : getpartname := 'OS/2 Boot Manager HPFS';
    $18 : getpartname := 'AST special Windows swap file ("Zero-Volt Suspend" partition)';
    $19 : getpartname := 'Willowtech Photon coS';
    $1B : getpartname := 'OS/2 Boot Manager VFAT';
    $1C : getpartname := 'OS/2 Boot Manager VFAT';
    $20 : getpartname := 'Willowsoft Overture File System (OFS1)';
    $21, $23, $26, $31, $33, $34, $36, $73, $74, $76, $a1, $a3,
    $a4 : getpartname := 'reserved';
    $24 : getpartname := 'NEC MS-DOS 3.x';
    $38 : getpartname := 'Theos';
    $3C : getpartname := 'PowerQuest PartitionMagic recovery partition';
    $40 : getpartname := 'VENIX 80286';
    $41 : getpartname := 'Personal RISC Boot';
    $42 : getpartname := 'SFS by Peter Gutmann';
    $50 : getpartname := 'OnTrack Disk Manager (RO)';
    $51 : getpartname := 'OnTrack Disk Manager (RW)';
    $51 : getpartname := 'NOVEL';
    $52 : getpartname := 'CP/M';
    {52 : getpartname := 'Microport System V/386';}
    $53 : getpartname := 'OnTrack Disk Manager (WO?)';
    $54 : getpartname := 'OnTrack Disk Manager (DDO)';
    $55 : getpartname := 'EZ Drive';
    $56 : getpartname := 'GoldenBow VFeature';
    $61 : getpartname := 'SpeedStor';
    $63 : getpartname := 'Unix System V';
    $63 : getpartname := 'Mach, MtXinu BSD 4.3 on Mach';
    $63 : getpartname := 'GNU HURD';
    $64 : getpartname := 'Novell NetWare 286';
    $65 : getpartname := 'Novell NetWare (3.11)';
    $67, $68, $69 : getpartname := 'Novell';
    $70 : getpartname := 'DiskSecure Multi-Boot';
    $75 : getpartname := 'PC/IX';
    $80 : getpartname := 'Minix v1.1 - 1.4a';
    $81 : getpartname := 'Linux 1.x/Minix v1.4b+';
    {81 : getpartname := 'Linux';}
    {81 : getpartname := 'Mitac Advanced Disk Manager';}
    $82 : getpartname := 'Linux swap';
    {82 : getpartname := 'Prime';}
    $83 : getpartname := 'Linux native (ext2fs/xiafs)';
    {84 : getpartname := 'OS/2-renumbered type 04h partition (related to hiding DOS C: drive)'; }
    $85 : getpartname := 'Linux extended';
    $86 : getpartname := 'FAT16 volume/stripe set (Windows NT)';
    {87 : getpartname := 'HPFS Fault-Tolerant mirrored';
    $87 : getpartname := 'NTFS volume/stripe set';}
    $87 : getpartname := 'WinNT HPFS or NTFS';
    $93 : getpartname := 'Amoeba file system';
    $94 : getpartname := 'Amoeba bad block table';
    $A0 : getpartname := 'Phoenix NoteBIOS Power Management "Save-to-Disk"';
    $A5 : getpartname := 'BSD Variant';
    $A6 : getpartname := 'OpenBSD';
    $A7 : getpartname := 'NeXTStep';
    $b1, $b3, $b4, $b6 : getpartname := 'reserved';
    $B7 : getpartname := 'BSDI file system (secondarily swap)';
    $B8 : getpartname := 'BSDI swap (secondarily file system)';
    $C1 : getpartname := 'DR DOS 6.0 LOGIN.EXE-secured 12-bit FAT';
    $C4 : getpartname := 'DR DOS 6.0 LOGIN.EXE-secured 16-bit FAT';
    $C6 : getpartname := 'DR DOS 6.0 LOGIN.EXE-secured Huge';
    {C6 : getpartname := 'corrupted FAT16 volume/stripe set (Windows NT)';}
    $C7 : getpartname := 'Syrinx Boot';
    $C7 : getpartname := 'corrupted NTFS volume/stripe set';
    $D8 : getpartname := 'CP/M-86';
    $DB : getpartname := 'CP/M, Concurrent CP/M, Concurrent DOS';
    {DB : getpartname := 'CTOS (Convergent Technologies OS)';}
    $E1 : getpartname := 'SpeedStor 12-bit FAT extended';
    $E3 : getpartname := 'DOS RO';
    $E3 : getpartname := 'Storage Dimensions';
    $E4 : getpartname := 'SpeedStor 16-bit FAT';
    $e5, $e6, $f3, $f6 : getpartname := 'reserved';
    $F1 : getpartname := 'Storage Dimensions';
    $F2 : getpartname := 'DOS 3.3+ secondary';
    $F4 : getpartname := 'SpeedStor';
    $F4 : getpartname := 'Storage Dimensions';
    $FE : getpartname := 'LANstep';
    $FE : getpartname := 'IBM PS/2 IML';
    $FF : getpartname := 'Xenix bad block table';
  end;
end;

procedure get_part_info;
type
  partition_block = record
    boot_indicator : byte;
    start_head : byte;
    start_sector : byte;
    start_track : byte;
    os_indicator : byte;
    end_head : byte;
    end_cylinder : byte;
    end_track : byte;
    preceding_sectors : longint;
    length_sectors : longint;
  end;

var
  disk_data : array [0..2048 - 1] of byte;
  regs : registers;
  partition_number : byte;
  partition_type : word;
  record_block : word;
  pb : ^partition_block;
  drive : byte;
  count : byte;   { <--- The number of drives in the system }
  rpart_size : real;
  rlength_sectors : real;

begin
  verbose_write ('Reading partition info...');
  if running_winnt then begin
    outputlist.ins ('Cannot read partition information due to the presense of Windows NT');
    verbose_writeln ('Not done: Windows NT');
    exit;
  end;
  if is_dosemu then begin
    outputlist.ins ('Cannot read partition information due to the presense of DOSEmu');
    verbose_writeln ('Not done: DOSEmu');
    exit;
  end;
  { first we count the drives }
  verbose_writeln ('');
  verbose_writeln ('Counting drives...');
  count := 0;
  for drive := $80 to $99 do begin
    regs.ax := $0201;
    regs.cx := $0001;
    regs.dx := drive;
    regs.es := seg (disk_data);
    regs.bx := ofs (disk_data);
    intr ($13, regs);
    if (regs.flags and fcarry) = 0 then begin
      verbose_writeln (indent + '$' + dectohex8 (regs.dx) + ', flags = $' + dectohex16 (regs.flags));
      inc (count);
    end else begin
      continue;
    end;
  end;
  { Done!  Give results... }
  verbose_writeln ('Physical drive count: ' + numstr (count));
  if count = 1 then
    outputlist.ins ('1 hard drive found')
  else
    outputlist.ins (numstr (count) + ' hard drives found');
  if count = 0 then exit;    { Why bother?  There are no HDs }
  { Now we go through again, this time actually doing something with the
    data }
  count := 0;
  for drive := $80 to $99 do begin
    regs.ax := $0201;
    regs.cx := $0001;
    regs.dx := drive;
    regs.es := seg (disk_data);
    regs.bx := ofs (disk_data);
    intr ($13, regs);
    if (regs.flags and fcarry) = 0 then begin
      { Yay!  A valid drive! }
      for partition_number := 0 to 3 do begin
        record_block := $01be + partition_number shl 4;
        partition_type := record_block + 4;
        if disk_data [partition_type] = $00 then begin
          continue;
        end else begin
          verbose_write ('Disk ' + numstr (drive - $80) + ' Partition ' +
                         numstr (partition_number + 1) + ' is type ' +
                         dectohex8 (disk_data [partition_type]));
          outputstring := indent + 'Disk ' + numstr (drive - $80) + ' Partition '
                          + numstr (partition_number + 1) + ':';
          pb := @(disk_data [record_block]);
          outputlist.ins (outputstring);
          rlength_sectors := pb^.length_sectors;
          rpart_size := rlength_sectors * 512;
          outputstring := indent + indent + 'Type 0x' + dectohex8 (disk_data [partition_type])
                           + ' (' + getpartname (disk_data [partition_type]) + ')';
          outputlist.ins (outputstring);
          str (rpart_size:1:0, tempstr);
          verbose_writeln (' Partition length is ' + numstr (pb^.length_sectors) + ' 512-byte sectors');
          outputstring := indent + indent + 'Partition is ' + strcommas (tempstr) + ' bytes';
          str (rpart_size / 1024 / 1024:1:1, tempstr);
          outputstring := outputstring + ' (' + strcommas (tempstr) + ' MBytes) long';
          outputlist.ins (outputstring);
        end;
      end;
    end else
      continue;
  end;
end;
{-----------------------------------------------------------------------------}

procedure getDisk;
type
  fat32_free_space_rec = record
    struct_size : word;
    struct_ver : word;
    sectors_per_cluster : longint;
    bytes_per_sector : longint;
    avail_clusters : longint;
    number_clusters : longint;
    avail_sectors : longint;
    total_phys_sectors : longint;
    avail_alloc_units : longint;
    total_alloc_units : longint;
    reserved : array [1..8] of char;
  end;

  procedure handle_drive_data (drive : byte; cd_drive_list : string);
  var
    regs : registers;
    total, avail : real;
    clustsize : word;
    currentDir : string [80];
    curDrive : char;
    temp : real;
    path : array [0..5] of char;
    fat32_free_space : fat32_free_space_rec;
  begin
    { under int 0x21/ax=0x36, you can get the free disk space on a drive.
      dl is the drive number.
      on return:
      if ax is 0xffff then the drive is invalid.
      otherwise:
        ax = sectors per cluster
        bx = number of free clusters
        cx = bytes per sector
        dx = total clusters on the drive
    }
    { we'll try the FAT32 functions first. }
{    strcopy (path, 'A:\');
    path [0] := char (drive + $40);
    writeln ('path = ', path); readkey;
    regs.ax := $7303;
    regs.ds := seg (path);
    regs.dx := ofs (path);
    regs.es := seg (fat32_free_space);
    regs.di := seg (fat32_free_space);
    intr ($21, regs);
    if (regs.flags and fcarry) <> 0 then begin
      writeln ('fat32 not supported');
      readkey;
    end;}

    regs.ah := $36;
    regs.dl := drive;
    intr ($21, regs);
    if regs.ax = $ffff then
      exit
    else begin
      avail := regs.ax;
      avail := avail * regs.bx * regs.cx;
      total := regs.ax;
      total := total * regs.cx * regs.dx;
    end;
    getdir (drive, currentDir);
    clustsize := word (regs.ax) * word (regs.cx);
    curDrive := upcase (currentDir [1]);
    verbose_write ('Drive ' + curdrive + ': ');
    outputlist.ins (indent + 'Drive ' + curDrive + ':');
    str ((avail / 1048576):1:2, tempstr);
    outputstring := indent + indent + strcommas (tempstr) + ' MBytes of ';
    verbose_write (tempstr + ' Mbytes used   ');
    str ((total / 1048576):1:2, tempstr);
    outputstring := outputstring + strcommas (tempstr) + ' MBytes total used';
    verbose_writeln (tempstr + ' Mbytes total');
    outputlist.ins (outputstring);
    temp := (total - avail) / total * 100.0;
    str (temp:1:2, tempstr);
    outputlist.ins (indent + indent + 'Utilization rate is ' + tempstr + '%');
    if pos (currentDir [1], cd_drive_list) = 0 then begin
      outputlist.ins (indent + indent + 'Drive has ' + numbercommas (clustsize) +
                      ' byte clusters');
      exit;
    end;
  end;
var
  i, _ax, _bx : word;
  result : byte;
  currentDir : pathstr;
  cddrives : string [26];
begin
  asm
    mov ax, $150b
    mov cx, 0
    int $2f
    mov _bx, bx
    mov _ax, ax
  end;
  cddrives := '';
  if _bx = $adad then begin
    for i := 0 to 25 do begin
      asm
        mov ax, $150b
        mov cx, i
        int $2f
        mov _bx, bx
        mov _ax, ax
      end;
      if _ax <> 0 then
        cddrives := cddrives + char (i + $41);
    end;
  end;
  get_part_info;
  verbose_writeln ('Gathering space usage information...');
  outputlist.ins ('Drive information:');
  for i := 3 to 26 do
    handle_drive_data (i, cddrives);
  verbose_writeln ('Done');
  { under int0x2f/ax=0x150b/cx=0, there is a check for the presence of CD-ROM
    extensions, such as mscdex or corelcdx.  the value in cx is the number of
    the drive to check ... serves to check which drive is a CD. 0 = a:,
    1 = b:, and so on.  on return, if bx != 0xadad, then there are no
    extensions loaded.
  }
  verbose_write ('Searching for CDROM drives...');
  if cddrives = '' then begin
    outputlist.ins (indent + 'There are no CD-ROM drives');
    verbose_writeln ('None');
  end else begin
    outputlist.ins (indent + 'CD-ROM drives are ' + cddrives);
    verbose_writeln ('Drive letters ' + cddrives);
  end;
end;

{$F+}
{$L fpu.obj}
function iscyrixfpu : boolean; external;
procedure get_fpu_type; external;
{$F-}

{ CONSTANT DEFINITIONS ---------------------------------------- }
const
  { GENERIC CONSTANTS ----------------------------------------- }
  ULONG_MAX = $7FFFFFFF;
  MAXCLOCKS = 150;                     { Maximum number of cycles per BSF
                                         instruction }
  EOA = 0;                             { End of Array variable. }
  CLONE_MASK = $8000;                  { Mask to be 'OR'ed with processor
                                         family type }
  { ACCURACY AFFECTING CONSTANTS ------------------------------ }
  ITERATIONS = 1500;                   { Number of times to repeat BSF
                                         instruction in samplings.
                                         Initially set to 4000.
                                         NOTE-- Set this value
                                         identically in timeproc.asm }

  SAMPLING_DELAY = 500;                { Number of ticks to allow to
                                         to elapse during sampling. Initially
                                         set to 60. }
  TOLERANCE = 2;                       { Number of MHz to allow samplings to
                                         deviate from average of samplings.
                                         Initially set to 2. }
  MAX_TRIES = 20;                      { Maximum number of samplings to allow
                                         before giving up and returning
                                         current average. Initially set to
                                         20. }
  SAMPLINGS = 10;                      { Number of BSF sequence samplings to
                                         make.  Initially set to 10. }
  TOL386 = 2;                          { Number of MHz above normalised value
                                         to normalise down to the current
                                         normalised speed. Initially set to
                                         2. }
  TOL486 = 4;                          { Number of MHz above normalised value
                                         to normalise down to the current
                                         normalised speed. Initially set
                                         to 4. }
  TOLP5 = 5;                           { Number of MHz above normalised value
                                         to normalise down to the current
                                         normalised speed. Initially set to
                                         5. }
  TOLP6 = 5;                           { Number of MHz above normalised value
                                         to normalise down to the current
                                         normalised speed. Initially set
                                         to 5. }
{ Number of cycles needed to execute a single BSF
  instruction. Note that processors below i386(tm)
  are not supported. }
  processor_cycles : array [0..20] of longint = (
    00, 00, 00, 115, 48, 43, 38, 38, 38, 38, 38, 38,
    38, 38, 38, 38, 38, 38, 38, 38, 38 );
{ Normalize raw clock frequency to one of these values. }
  i386Speeds : array [0..5] of word = ( 16, 20, 25, 33, 40, EOA );
  i486Speeds : array [0..10] of word = ( 16, 20, 25, 33, 50, 66, 75, 100, 120, 133, EOA );
  iP5Speeds : array [0..13] of word = ( 60, 66, 75, 90, 100, 120, 133, 150,
    166, 180, 200, 233, 266, EOA );
  iP6Speeds : array [0..18] of word = ( 60, 66, 75, 90, 100, 120, 133, 150,
    166, 180, 200, 233, 266, 300, 333, 350, 400, 450, EOA );

{$L timeproc.obj}
{$F+}
function _grabtime : integer; external;
function _Time_Processor_bsf : integer; external;
{$F-}

function NormFreq (processor : word; freq : longint) : longint;
var
  ptr : integer;
begin
  ptr := 0;
  if 3 = processor then begin
    while (i386Speeds [ptr] <> EOA) do begin
      if freq <= ((i386Speeds [ptr]) + TOL386) then begin
        NormFreq := i386Speeds [ptr];
        { Scan each speed in array until current calculated frequency
          is less than or equal to normalized value plus TOL386. }
        exit;
      end;
      inc (ptr);
    end;
    NormFreq := freq;                { If raw speed is higher than highest
                                       normalized speed plus TOL386, return
                                       raw frequency. }
    exit;
  end else if 4 = processor then begin
    while i486Speeds [ptr] <> EOA do begin
      if freq <= (integer (i486speeds [ptr] + TOL486)) then begin
        NormFreq := i486speeds [ptr];
        { Scan each speed in array until current calculated frequency
          is less than or equal to normalized value plus TOL486. }
        exit;
      end;
      inc (ptr);
    end;
    NormFreq := freq;                  { If raw speed is higher than highest
                                         normalized speed plus TOL486, return
                                         raw frequency. }
    exit;
  end else if 5 = processor then begin
    while ip5speeds [ptr] <> EOA do begin
      if freq <= ((ip5speeds[ptr] + TOLP5)) then begin
        NormFreq := ip5speeds [ptr];
        { Scan each speed in array until current calculated frequency
          is less than or equal to normalized value plus TOLP5. }
        exit;
      end;
      inc (ptr);
    end;
    NormFreq := freq;                 { If raw speed is higher than highest
                                        normalized speed plus TOLP5, return
                                        raw frequency. }
    exit;
  end else if 6 = processor then begin
    while ip6speeds [ptr] <> EOA do begin
      if freq <= ((ip6speeds[ptr] + TOLP6)) then begin
        NormFreq := ip6speeds [ptr];
        { Scan each speed in array until  current calculated frequency
          is less than or equal to normalized value plus TOLP6. }
        exit;
      end;
      inc (ptr);
    end;
    NormFreq := freq;                  { If raw speed is higher than highest
                                         normalized speed plus TOLP6, return
                                         raw frequency. }
    exit;
  end else
    NormFreq := freq;                  { return raw frequency }
end; { NormFreq }

function cpu_speed (processor : word) : longint;
var
  i : integer;                         { Temporary Counter Variable }
  current : longint;                   { Variable to store time elapsed
                                         during loop of of BSF instructions }
  lowest : longint;                    { Since algorithm finds the lowest
                                         value out of a set of sampling, this
                                         variable is set intially to the max
                                         unsigned long value).  This
                                         guarantees that the initialized
                                         value is not later used as the least
                                         time through the loop. }
  ticks : longint;                     { Microseconds elapsed during test }
  cycles : longint;                    { Clock cycles elapsed during test }
  u0, u1 : word;                       { 16-bit variables for time }
  v0, v1 : word;                       { stamp reads. These are
                                         later merged into stamp0, and
                                         stamp1 (32-bit variables) }
  stamp0, stamp1 : longint;            { Time Stamp Variable for
                                         beginning and end of
                                         PentiumPro(R) and above
                                         tests }
  t0, t1 : word;                       { Variables to store GrabTime Results
                                         for i386(tm), i486(tm), and
                                         Pentium(R) tests. }
  freq : longint;                      { Most current frequ. calculation }
  freq2 : longint;                     { 2nd most current frequ. calc. }
  freq3 : longint;                     { 3rd most current frequ. calc. }
  total : longint;                     { Sum of previous three frequency
                                         calculations }
  tries : integer;                     { Number of times a calculation has
                                         been made on this call to cpuspeed }

begin
  current := 0; lowest := ULONG_MAX;
  tries := 0; cycles := ITERATIONS * processor_cycles [processor];
  if (processor >= 6) then begin
    repeat
      inc (tries);                     { Increment number of times sampled on
                                         this call to cpuspeed }
      freq3 := freq2;                  { Shift frequencies back to }
      freq2 := freq;                   { make room for new frequency
                                         measurement }
      t0 := _GrabTime;                 { Get Start Time. This code directly
                                         peeks the Counter/Timer.  This tends
                                         to cause inaccuracies on MS-Windows*
                                         NT }
      asm
        db $0f, $31                    { Read Time Stamp }
        MOV CL, 16
        MOV u0, AX
        db $66; SHR AX,CL
        MOV u1, AX
      end;
      t1 := t0;
      while (t1 - t0 < SAMPLING_DELAY) do begin
      { Loop until 500 ticks have passed since last GrabTime read.
        This allows for elapsed time for sampling }
        t1 := _GrabTime;                 { Get End Time }
      end;
      asm
        db $0f, $31                    { Read Time Stamp }
        MOV CL, 16
        MOV v0, AX
        db $66; SHR AX,CL
        MOV v1, AX
      end;
      stamp0 := u1 * 65536 + u0;
      stamp1 := v1 * 65536 + v0;
      { Move two 16-bit values into one 32-bit value for the time
        stamp read at both the beginning and end of the test }
      cycles := stamp1 - stamp0;
      { Number of internal clock cycles is difference between two
        time stamp reads }
      ticks := t1 - t0;
      { Number of external ticks is difference between two GrabTime reads }
      ticks := ticks * 100000;
      { Convert ticks to hundred thousandths of a tick }
      ticks := ticks div 119318;
      { Convert hundred thousandths of ticks to microseconds (us) }
      if (ticks mod 119318) >= (119318 / 2) then
        inc (ticks);                   { Round up if necessary }
      freq := cycles div ticks;        { Cycles / us = MHz }
      if (cycles mod ticks) > (ticks / 2) then
        inc (freq);                    { Round up if necessary }
      total := (freq + freq2 + freq3);
      { Total last three frequency calculations }
    until not ((tries < 3) or (tries < MAX_TRIES) and
              ((abs (3 * integer (freq) - integer (total)) > 3 * TOLERANCE) or
               (abs (3 * integer (freq2) - integer (total)) > 3 * TOLERANCE) or
               (abs (3 * integer (freq3) - integer (total)) > 3 * TOLERANCE)));
    { Compare last three calculations to average of last three
      calculations. }
    if (total div 3) <> ((total + 1) div 3) then
      inc (total);                      { Round up if necessary }
    freq := total div 3;                { Average last three calculations }
  end else begin
    { If processor does not support ring 0 time stamp reading, but is
      at least a 386 or above, utilize method of timing a loop of BSF
      instructions which take a known number of cycles to run on i386(tm),
      i486(tm), and Pentium(R) processors. }
    for i := 0 to SAMPLINGS - 1 do begin
    { Sample Ten times. Can be increased or decreased depending on accuracy
      vs. time requirements }
      current := _Time_Processor_bsf;
      if current < lowest then         { Take lowest elapsed }
        lowest := current;             { time to account for some samplings
                                         being interrupted by other
                                         operations }
      ticks := lowest;
      ticks := ticks * 100000;         { Convert ticks to hundred thousandths
                                         of a tick }
      ticks := ticks div 119318;       { Convert hundred thousandths of ticks
                                         to microseconds (us) }
      if (ticks mod 119318) >= (119318 div 2) then
        inc (ticks);                   { Round up if necessary }
      freq := cycles div ticks;        { Cycles / us  = MHz }
      if (cycles mod ticks) > (ticks div 2) then
        inc (freq);                    { Round up if necessary }
    end;
  end;
  verbose_writeln ('Raw CPU speed is ' + numstr (freq));
  cpu_speed := normfreq (processor, freq);
end;

{$F+}
{$L p5-fpu.obj}
function testndp : word; external;
function p5_fpu_test : boolean;
begin
  p5_fpu_test := testndp = 64;
end;
{$F-}

procedure getProcessor;
label pentium, not_Nx586, is_Nx586, end_getcpuid, finish, no_cpuid, done,
      done2;
var
  cpuidok : boolean;
  cpudata, eax, ebx, ecx, edx : longint;
  dir0, dir1, stepping, model : byte;
  vendor_id : string [12];
  outputstr : string;
  cpu_freq, data : word;
begin
  outputstring := 'CPU is ';
  vendor_id := '';
  check_fpu := 1;
  verbose_write ('Testing 8088...');
  if is_8088 then begin
    cputype := 0;
    goto end_getcpuid;
  end;
  verbose_writeln ('Done');
  verbose_write ('Testing 80286...');
  if is_80286 then begin
    cputype := 2;
    goto end_getcpuid;
  end;
  verbose_writeln ('Done');
  verbose_write ('Testing 80386...');
  if is_80386 then begin
    verbose_writeln ('Done');
    verbose_write ('Testing NexGen...');
    if is_nexgen then begin
      verbose_writeln ('Done');
      verbose_writeln ('CPU is NexGen');
      vendor_id := 'NexGenDriven';
      cputype := 5;
      goto end_getcpuid;
    end;
    cputype := 3;
    goto end_getcpuid;
  end;
  verbose_writeln ('Done');
  verbose_write ('Testing 80486...');
  cpuidok := false;
  if is_80486 then begin
    verbose_writeln ('Done');
    cputype := 4;
    verbose_write ('CPUID is not supported.  Checking for Cyrix...');
    if (iscyrix) then begin
      verbose_write ('Done');
      verbose_writeln ('CPU is Cyrix');
{      cpu_type (dir0, dir1);
      display_which_cyrix (dir0, dir1);}
      outputlist.ins (outputstring + 'Cyrix Cx486 or higher');
      exit;
    end;
  end;
pentium:
  verbose_writeln ('Done');
  verbose_write ('Using CPUID...');
  vendor_id := '';
  cpuidok := true;
  check_fpu := 0;
  { the following code uses the CPUID function to determine the cpu }
  model := 0;
  do_cpuid (1, eax, ebx, ecx, edx);
  data := eax;
  do_cpuid (0, eax, ebx, ecx, edx);
  vendor_id := vendor_id + char (ebx and $000000FF);
  vendor_id := vendor_id + char (ebx and $0000FF00 shr 8);
  vendor_id := vendor_id + char (ebx and $00FF0000 shr 16);
  vendor_id := vendor_id + char (ebx and $FF000000 shr 24);
  vendor_id := vendor_id + char (edx and $000000FF);
  vendor_id := vendor_id + char (edx and $0000FF00 shr 8);
  vendor_id := vendor_id + char (edx and $00FF0000 shr 16);
  vendor_id := vendor_id + char (edx and $FF000000 shr 24);
  vendor_id := vendor_id + char (ecx and $000000FF);
  vendor_id := vendor_id + char (ecx and $0000FF00 shr 8);
  vendor_id := vendor_id + char (ecx and $00FF0000 shr 16);
  vendor_id := vendor_id + char (ecx and $FF000000 shr 24);
  verbose_writeln ('Done');
  verbose_writeln ('Vendor is ' + vendor_id);
  verbose_writeln ('CPU is 0x' + dectohex32 (data));
  verbose_writeln ('              fms');
  cputype := (data and $0F00) shr 8;
  model := (data and $00F0) shr 4;
  stepping := data and $000F;
end_getcpuid:
  verbose_writeln ('CPU is type ' + numstr (cputype));
  if (cputype >= 3) and (not (is_dosemu) or (is_winnt)) then begin
    cpu_freq := cpu_speed (cputype);
    verbose_writeln ('CPU speed calculated at ' + numstr (cpu_freq) + ' MHz');
  end else
    cpu_freq := 0;
  if (cputype < 4) then begin
    case cputype of
      0 : outputlist.ins (outputstring + '8086/8088 or clone');
      2 : begin
        outputlist.ins (outputstring + '80286 or clone');
        check_fpu := 0;
      end;
      3 : begin
        outputstr := '80386SX';
        inline ($0f/$20/$c0 );         { MOV EAX,CR0 }
        inline ($66/$8b/$c8 );         { MOV ECX,EAX }
        inline ($66/$83/$F0/$10);      { MOV EAX,10H }
        inline ($0f/$22/$c0 );         { MOV CR0,EAX }
        inline ($0f/$20/$c0 );         { MOV EAX,CR0 }
        inline ($0f/$22/$c1 );         { MOV CR0,ECX }
        inline ($66/$33/$c1 );         { XOR EAX,ECX }
        asm
           JZ done                     { if the cpu was a 386sx then say so . . .}
        end;
        outputstr := '80386DX';
done:
        outputstring := outputstring + outputstr + ' or clone';
        if cpu_freq > 0 then
          outputstring := outputstring + ' operating at ' + numstr (cpu_freq) + ' MHz';
        outputlist.ins (outputstring);
        check_fpu := 1;
        exit;
      end;
    end;
  end else if cputype >= 4 then begin
    if cpuidok = false then begin
      outputstr := '80487SX';
      inline ($0f/$20/$c0 );         { MOV EAX,CR0 }
      inline ($66/$8b/$c8 );         { MOV ECX,EAX }
      inline ($66/$83/$F0/$10);      { MOV EAX,10H }
      inline ($0f/$22/$c0 );         { MOV CR0,EAX }
      inline ($0f/$20/$c0 );         { MOV EAX,CR0 }
      inline ($0f/$22/$c1 );         { MOV CR0,ECX }
      inline ($66/$33/$c1 );         { XOR EAX,ECX }
      asm
         JZ done2                    { if the cpu was a 487sx then say so . . .}
      end;
      outputstr := '80486DX';
done2:
      outputstring := outputstring + outputstr + ' or clone';
      if cpu_freq > 0 then
        outputstring := outputstring + ' operating at ' + numstr (cpu_freq) + ' MHz';
      outputlist.ins (outputstring);
      check_fpu := 1;
      exit;
    end;
    if (vendor_id = 'GenuineIntel') then begin
      outputstring := outputstring + 'Intel ';
      case cputype of
        4 : begin
          outputstring := outputstring + '80486';
          case model of
            0 : outputstring := outputstring + 'DX';
            1 : begin
              outputstring := outputstring + 'DX-50';
              {notiming := true;}
            end;
            2 : outputstring := outputstring + 'SX';
            3 : outputstring := outputstring + 'DX2';
            4 : outputstring := outputstring + 'SL';
            5 : outputstring := outputstring + 'SX2';
            7 : outputstring := outputstring + 'DX2WB';
            8 : outputstring := outputstring + 'DX4';
            9 : outputstring := outputstring + 'DX4WB';
          end;
        end;
        5 : begin
          outputstring := outputstring + 'Pentium';
          case model of
            3 : outputstring := outputstring + ' OverDrive PODP5V83';
            4 : outputstring := outputstring + ' MMX';
            5 : outputstring := outputstring + ' OverDrive for 80486DX4';
            6 : outputstring := outputstring + ' OverDrive for 5V Pentium';
            8 : outputstring := outputstring + ', Mobile Edition';
          end;
        end;
        6 : begin
          case model of
            0, 1 : outputstring := outputstring + 'Pentium Pro';
            3 : outputstring := outputstring + 'Pentium II';
            4 : outputstring := outputstring + 'P54C socket OverDrive';
            5 : outputstring := outputstring + 'Pentium II (Celeron?)';
            6 : outputstring := outputstring + 'Celeron A';
          end;
        end else
          outputstring := outputstring + '80' + numstr (cputype) + '86-class';
      end;
      if cpu_freq > 0 then
        outputstring := outputstring + ' operating at ' + numstr (cpu_freq) + ' MHz';
      outputlist.ins (outputstring);
      goto finish;
    end else if (vendor_id = 'UMC UMC UMC ') then begin
      outputstring := (outputstring + 'UMC486');
      case model of
        1 : outputstring := (outputstring + ' U5D');
        2 : outputstring := (outputstring + ' U5S');
        else
          outputstring := outputstring + ' model ' + numstr (model);
      end;
      outputlist.ins (outputstring);
      goto finish;
    end else if (vendor_id = 'AuthenticAMD') then begin
      outputstring := (outputstring + 'American Micro Devices ');
      case cputype of
        4 : begin
          case model of
            0 : outputstring := outputstring + 'Am486DX';
            2 : outputstring := outputstring + 'Am486SX';
            3 : outputstring := outputstring + 'Am486DX2';
            4 : outputstring := outputstring + 'Am486SL';
            5 : outputstring := outputstring + 'Am486SX2';
            7 : outputstring := outputstring + 'Am486DX2WB';
            8 : outputstring := outputstring + 'Am486DX4';
            9 : outputstring := outputstring + 'Am486DX4WB';
            14 : outputstring := outputstring + 'Am5x86';
            15 : outputstring := outputstring + 'Am5x86';
          end;
        end;
        5 : begin
          case model of
            0..1 : outputstring := outputstring + 'K5';
            2 : outputstring := outputstring + 'K5-PR166';
            3 : outputstring := outputstring + 'K5-PR200';
            6..7 : outputstring := outputstring + 'K6';
            8 : outputstring := outputstring + 'K6-2';
            9 : outputstring := outputstring + 'K6-3';
          end;
        end;
        else
          outputstring := outputstring + '80' + numstr (cputype) + '86-class';
      end;
      if cpu_freq > 0 then
        outputstring := outputstring + ' operating at ' + numstr (cpu_freq) + ' MHz';
      outputlist.ins (outputstring);
      goto finish;
    end else if (vendor_id = 'CyrixInstead') then begin
      outputstring := outputstring + 'Cyrix ';
      case cputype of
        4 : outputstring := outputstring + 'Cx486';
        5 : outputstring := outputstring + '6x86';
        6 : outputstring := outputstring + '6x86L';
        else
          outputstring := outputstring + '80' + numstr (cputype) + '86 class';
      end;
      outputlist.ins (outputstring);
      goto finish;
    end else if vendor_id = 'NexGenDriven' then
      outputstring := outputstring + 'NexGen Nx586';
      outputlist.ins (outputstring);
      goto finish;
    end else if vendor_id = 'CentaurHauls' then begin
      outputstring := outputstring + 'Centaur C6';
      outputlist.ins (outputstring);
      goto finish;
    end else begin
      verbose_writeln ('Unable to ID CPU');
      outputlist.ins ('Unidentifiable CPU detected');
      outputlist.ins (indent + 'Data Returned:');
      outputlist.ins (indent + 'vendor identification string = ''' + vendor_id + '''');
      outputlist.ins (indent + '   model id = ' + numstr (model));
      outputlist.ins (indent + '  family id = ' + numstr (cputype));
      outputlist.ins (indent + 'stepping id = ' + numstr (stepping));
      outputlist.ins (indent + 'EAX = 0x' + dectohex32 (eax));
      outputlist.ins (indent + 'EBX = 0x' + dectohex32 (ebx));
      outputlist.ins (indent + 'EDX = 0x' + dectohex32 (edx));
      outputlist.ins (indent + 'ECX = 0x' + dectohex32 (ecx));
      check_fpu := 1;
      exit;
    end;
finish:
  { early Pentiums had a bug in the FPU code - they produced the wrong
    results.  We should bother only for the 60 and 66 MHz units }
  if (cputype = 5) and (vendor_id = 'GenuineIntel') and (cpu_freq <= 66) and
     (cpu_freq > 0) then begin
    verbose_write ('Checking for Pentium FPU bug...');
    if p5_fpu_test = false then
      outputlist.ins (indent + 'This FPU has the Pentium math bug. :-(')
    else
      outputlist.ins (indent + 'This FPU does not have the Pentium math bug. :-)');
    verbose_writeln ('Done');
  end;
end;

procedure getMath;
begin
  verbose_write ('Testing NDP...');
  fpu_type := 0;
  get_fpu_type;
  verbose_writeln ('Done');
  if fpu_type = 0 then begin
    verbose_writeln ('No NDP');
    outputlist.ins ('No math coprocessor');
  end else if iscyrixfpu then begin
    outputlist.ins ('Cyrix NDP');
    verbose_writeln ('Cyrix FPU detected');
  end else begin
    verbose_writeln ('NDP detected');
    case (fpu_type) of
      2 : outputlist.ins ('Math coprocessor is Intel287');
      3 : outputlist.ins ('Math coprocessor is 80387');
      else outputlist.ins ('Math coprocessor detected');
    end
  end;
end;

function xmsfound : boolean;
var
  result : byte;
begin
   asm
     mov ax, $4300
     int $2f
     mov result, al
   end;
   xmsfound := result = $80;
end;

function initXMS : boolean;
var _es, _bx : word;
begin
  if not (xmsfound) then begin
    present := false;
    initxms := false;
    exit;
  end;
  present := true;
  asm
    mov ax, $4310;
    int $2F;
    mov _es, es
    mov _bx, bx
  end;
  xmsdriver := ptr (_es, _bx);
  initxms := true;
end;

function xmsMemAvail : word;
var
  memory : word;
label label1, label2;
begin
  xmsError := $00;
  if not (present) then begin
    xmsMemAvail := 0;
    exit;
  end;
  asm
    mov ah, $08
    call xmsDriver
    or ax, ax
    jne label1
    mov xmsError, bl
    jmp label2;
label1:
    mov memory, dx;
  end;
label2:
  xmsMemAvail := memory;
end;

procedure getXMSVersion (var major : byte; var minor : byte);
var
  highbyte, lowbyte : byte;
label label1, label2;
begin
  xmsError := $00;
  if not (present) then exit;
  asm
    mov ah, $00
    call xmsDriver
    or ax, ax;
    jne label1;
    mov xmsError, bl
    jmp label2;
label1:
    mov highByte, ah
    mov lowByte, al
  end;
label2:
   major := highByte;
   minor := lowByte;
end;

Function GetAvailEMBHandles: Byte;

Var
  Temp: Byte;

Begin
  XMSError:=0;
  If Not(Present) Then Exit;
  Asm
    Mov  AH,0Eh
    Call [xmsdriver]
    Or   AX,AX
    Jne  @@1
    Mov  XMSError,BL
    Jmp  @@2
@@1:
    Mov  Temp,BL
@@2:
  End;
  GetAvailEMBHandles:=Temp;
End;

{amount of memory your program is using in bytes}
function program_memory : longint;
begin
  program_memory := longint(Seg(HeapEnd^)-PrefixSeg) * 16;
end;
{note : For TP5.0 or TP5.5 use the following:
       program_memory := longint(seg(freeptr) + $1000 - prefixseg) * 16;}
{       end;}

{amount of system memory in bytes}
function system_memory : longint;
var
  regs : registers;
begin
  intr ($12, regs);
  system_memory := longint (regs.ax) * 1024;
end;

procedure getMemory;
var
  ivec : pointer;
  low, high, emsVer : byte;
  avail, pages_free, total_pages : word;
  extended_memory : longint;
  result : boolean;
  xmsMajor, xmsMinor : byte;
  free, total, used, percent_used : real;
  regs : registers;
label ok, print_message, get_mem;
begin
  verbose_write ('Using BIOS to get memory info...');
  outputlist.ins ('System Memory:');
  port [$70] := $15;
  low := port [$71];
  port [$70] := $16;
  high := port [$71];
  verbose_write ('Conventional done; ');
  outputlist.ins (indent + numstr ((high shl 8) + low) + 'k base');

  regs.bx := 0;
  regs.ax := $e801;
  intr ($15, regs);
  verbose_writeln ('Extended done');
  if (regs.flags and fcarry) <> 0 then begin
    port [$70] := $17;
    low := port [$71];
    port [$70] := $18;
    high := port [$71];
    extended_memory := (high shl 8) + low;
    verbose_writeln ('  Used CMOS');
    verbose_writeln ('  Word is ' + dectohex16 (extended_memory));
  end else begin
    verbose_writeln ('  Used INT $15');
    verbose_writeln ('  Data is ' + numstr (longint (regs.dx) * 64 + longint (regs.cx)));
    extended_memory := regs.dx * 64 + regs.cx;
  end;
  outputlist.ins (indent + strcommas (numstr (extended_memory)) + 'k extended');

  verbose_write ('Getting DOS memory...');
  used := program_memory / 1024;
  total := system_memory / 1024;
  percent_used := (used / total) * 100;
  str (used:1:2, tempstr);
  outputstring := indent + tempstr + 'k of ';
  str (total:1:2, tempstr);
  outputstring := outputstring + tempstr + 'k (';
  str (percent_used:1:2, tempstr);
  outputstring := outputstring + tempstr + '%) used';
  outputlist.ins (outputstring);
  verbose_writeln ('Done');
  outputstring := '';
{ first, check the xms }
  result := initxms;
  verbose_write ('Getting XMS info...');
  outputlist.ins (indent + 'XMS:');
  if (result) then begin
    avail := xmsMemAvail;
    getXMSVersion (xmsMajor, xmsMinor);
    outputlist.ins (indent + indent + 'Version ' + versionstring (xmsmajor, xmsminor));
    outputlist.ins (indent + indent + strcommas (numstr (avail)) + 'k available');
    outputlist.ins (indent + indent + strcommas (numstr (GetAvailEMBHandles)) + ' handles available');
  end else
    outputlist.ins (indent + indent + 'None');
  verbose_writeln ('Done');
{ then, check the ems }
  verbose_write ('Getting EMS info...');
  outputlist.ins (indent + 'EMS:');
  getintvec ($67, ivec);
  ivec := pchar (ptr (seg (ivec^), 10));
  if (strlcomp (ivec, 'EMMXXXX0', 8) = 0) then begin
    asm
      mov ah, $46
      int $67
      cmp ah, $00
      je ok
      mov emsVer, 0
      jmp print_message
ok:
      mov emsVer, al
      mov ah, $42
      int $67
      cmp ah, $00
      je get_mem
      mov pages_free, 0
      mov total_pages, 0
      jmp print_message
get_mem:
      mov total_pages, dx
      mov pages_free, bx
    end;
print_message:
    outputlist.ins (indent + indent + 'Version ' + versionstring (emsver mod 10, emsver div 10));
    free := pages_free * 16;
    total := total_pages * 16;
    percent_used := (free / total) * 100;
    str (free:1:0, tempstr);
    outputstring := indent + indent + strcommas (tempstr) + 'k of ';
    str (total:1:0, tempstr);
    outputstring := outputstring + strcommas (tempstr) + 'k (';
    str (percent_used:1:2, tempstr);
    outputstring := outputstring + tempstr + '%) available';
    outputlist.ins (outputstring);
  end else
    outputlist.ins (indent + indent + 'None');
  verbose_writeln ('Done');
end;

procedure getDateTime;
const
   months : array [1..12] of pchar = (
     'January',
     'February',
     'March',
     'April',
     'May',
     'June',
     'July',
     'August',
     'September',
     'October',
     'November',
     'December' );
var
  hour, minute, sec, sec100 : word;
  year, month, day, dayofweek : word;
  locator : string [2];
begin
  verbose_write ('Getting date and time...');
  gettime (hour, minute, sec, sec100);
  getdate (year, month, day, dayofweek);
  if (hour > 12) then begin
    hour := hour - 12;
    locator := 'PM'
  end else
    locator := 'AM';
  str (hour:2, tempstr);
  outputstring :=  'The time is ' + tempstr + ':';
  str (minute:2, tempstr);
  outputstring := outputstring + tempstr + ':';
  str (sec:2, tempstr);
  outputstring := outputstring + tempstr + '.';
  str (sec100:2, tempstr);
  outputstring := outputstring + tempstr + ' ' + locator + ' on ' + strpas (months [month]) + ' ';
  str (day, tempstr);
  outputstring := outputstring + tempstr + ', ';
  str (year, tempstr);
  outputstring := outputstring + tempstr;
  outputlist.ins (outputstring);
  verbose_writeln ('Done');
end;

procedure getOSversion;

var
  result : word;
  pcmos_ver : word;
  nature, sys_ver_id, major, minor, oemCode : byte;
  patchlevel : word;
  date_string : string [8];
  e6vector : pointer;
  dosemu_inst : word;
  flags : word;

begin
  verbose_write ('Getting OS information...');
  { First, check for Windows NT.  If we are running under Windows NT, we
    we don't need to do anything else. }
  if running_winnt then begin
    outputlist.ins ('Windows NT');
    exit;
  end;
  { check for DR-DOS, Novell DOS 7, OpenDOS.  DR-DOS was a direct
    competitor to MS-DOS right up until it was purchased by Novell, and turned
    into Novell DOS 7.  Then Caldera bought it from Novell, and released it
    as OpenDOS 7.01.
  }
  asm
    pushf
    pop flags
  end;
  if (flags and fCarry) = 0 then
    flags := flags and fCarry;
  asm
    push flags
    popf
    mov ax, $4452
    int $21
    pushf
    pop flags
    mov nature, ah
    mov sys_ver_id, al
  end;
  if (flags and fCarry) = 0 then begin
    if (nature = $10) then
      verbose_writeln ('A variant of DR-DOS has been found.');
    case nature of
      $10 : begin
        case sys_ver_id of
          $41 : outputstring := 'DOS Plus 1.2';
          $60 : outputstring := 'DOS Plus 2.x';
          $63 : outputstring := 'DR-DOS 3.41';
          $64 : outputstring := 'DR-DOS 3.42';
          $65 : outputstring := 'DR-DOS 5.00';
          $71, $67 : outputstring := 'DR-DOS 6.00';
          $70 : outputstring := 'PalmDOS';
          $72 : outputstring := 'Novell DOS 7.0';
          $73 : outputstring := 'OpenDOS';
        end;
      end;
      $14 : begin
        case sys_ver_id of
          $32 : outputstring := 'Concurrent PC-DOS 3.2';
          $41 : outputstring := 'Concurrent DOS 4.1';
          $50 : outputstring := 'Concurrent DPS/XM 5.0 or Concurrent DPS/386 1.1';
          $60 : outputstring := 'Concurrent DPS/XM 6.0 or Concurrent DPS/386 2.0';
          $62 : outputstring := 'Concurrent DPS/XM 6.2 or Concurrent DPS/386 3.0';
          $66 : outputstring := 'Multiuser DR-DOS 5.1';
          $67 : outputstring := 'Concurrent DOS 5.1';
        end;
      end;
    end;
    outputlist.ins (outputstring);
    exit;
  end;
  { then check for everything else }
  asm
    mov ah, $30
    mov al, $00
    int $21
    mov major, al
    mov minor, ah
    mov oemcode, bh
  end;
  if major = 10 then begin
    outputlist.ins ('OS/2 1.x compatibility mode');
    exit;
  end else if (major = 20) and (minor <> 30) then begin
    outputlist.ins ('OS/2 2.x compatibility mode');
    exit;
  end else if (major = 20) and (minor = 30) then begin
    outputlist.ins ('OS/2 Warp 3.0 virtual machine');
    exit;
  end else if (major = 20) and (minor = 40) then begin
    outputlist.ins ('OS/2 Warp 4.0 virtual machine');
    exit;
  end;
  if major = 4 then begin
    asm
      mov al, $00
      mov ah, $87
      int $21
      mov major, al
    end;
    if major <> 0 then
      outputstring := 'European MS-DOS 4.0';
  end else if (major = 6) and (minor = 0) and (oemCode = $00) then
    outputstring := 'PC-DOS 6.1 or later'
  else begin
    case oemCode of
      $00 : outputstring := 'PC-DOS ';
      $01 : outputstring := 'Compaq DOS ';
      $33, $FF : outputstring := 'MS-DOS ';
      $33 : outputstring := 'Novell DOS ';
      $5E : outputstring := 'RxDOS ';
      $66 : outputstring := 'PTS-DOS ';
      $EE : outputstring := 'DR-DOS ';
      $EF : outputstring := 'Novell DOS ';
      $FD : outputstring := 'FreeDOS ';
      else begin
        outputstring := 'Unknown DOS OEM code of 0x' + dectohex8 (oemCode) + '; ';
        outputstring := outputstring + 'DOS version is ' + versionString (major, minor);
        outputlist.ins (outputstring);
        exit;
      end;
    end;
    outputstring := outputstring + versionString (major, minor);
  end;
  verbose_writeln ('Done');
  verbose_write ('Checking MS-Windows...');
  asm
    mov ax, $1600
    int $2F
    mov major, al
    mov minor, ah
  end;
  verbose_writeln ('Done');
  case major of
    $00 : outputstring := outputstring + '; Standard Mode Windows 3.x or plain MS-DOS';
    $01, $FF : outputstring := outputstring + '; Windows/386 2.x';
    else begin
      if (major = 4) and (minor = 0) then
        outputstring := outputstring + '; Windows 95'
      else if (major = 4) and (minor = 10) then
        outputstring := outputstring + '; Windows 98'
      else
        outputstring := outputstring + '; Windows ' + versionString (major, minor);
      verbose_writeln ('Windows version reported as ' + versionstring (major, minor));
    end;
  end;
  verbose_write ('Checking for DOSEmu...');
  outputlist.ins (outputstring);
{ check for DOSEmu }
  getintvec ($e6, e6vector);
  date_string [0] := char (8);
  date_string [1] := char (mem [$f000:$fff5]);
  date_string [2] := char (mem [$f000:$fff6]);
  date_string [3] := char (mem [$f000:$fff7]);
  date_string [4] := char (mem [$f000:$fff8]);
  date_string [5] := char (mem [$f000:$fff9]);
  date_string [6] := char (mem [$f000:$fffa]);
  date_string [7] := char (mem [$f000:$fffb]);
  date_string [8] := char (mem [$f000:$fffc]);
  if (date_string = '02/25/93') and (seg (e6vector^) = $f000) then begin
    asm
      mov ax, $0000
      int $e6
      mov dosemu_inst, ax
      mov major, bh
      mov minor, bl
      mov patchlevel, cx
    end;
    if dosemu_inst = $aa55 then begin
      outputstring := 'Running DOSEmu ' + versionString (major, minor) + ' patchlevel ' + numstr (patchlevel);
      outputlist.ins (outputstring);
    end
  end;
  verbose_writeln ('Done');
end;

procedure getDPMI;
var
  regs : registers;
begin
  verbose_write ('Getting DPMI information...');
  regs.ax := $1687;
  intr ($2F, regs);
  if regs.ax = $0000 then begin
    outputlist.ins ('DPMI detected');
    outputlist.ins (indent + '80' + numstr (regs.cl) + '86 CPU reported by DPMI server');
    outputlist.ins (indent + 'Server supports the version ' + versionString (regs.dh, regs.dl) + ' spec');
    outputlist.ins (indent + numstr (regs.si) + ' paragraphs for extender private data');
    outputlist.ins (indent + 'Mode switch entry point at 0x' + dectohex16 (regs.es) + ':0x' + dectohex16 (regs.di));
    if (regs.bx and $01) <> 0 then
      outputlist.ins (indent + '32-bit programs supported')
    else
      outputlist.ins (indent + '32-bit programs not supported');
  end;
{    outputlist.ins ('DPMI not detected');}
  verbose_writeln ('Done');
end;

const
  uarts : array [0..4] of pchar = (
    'no UART',
    '8250A',
    '16450 or 8250B',
    '16550',
    '16550A' );

function detect_UART (baseaddr : word) : word;
var
  x : integer;
begin
  { this function returns 0 if no UART is installed. }
  { 1: 8250, 2: 16450 or 8250 with scratch reg., 3: 16550, 4: 16550A }
  { first step: see if the LCR is there }
  port [baseaddr + 3] := $1b;
  if port [baseaddr + 3] <> $1b then begin
    detect_UART := 0;
    exit;
  end;
  port [baseaddr + 3] := $3;
  if port [baseaddr + 3] <> $3 then begin
    detect_UART := 0;
    exit;
  end;
  { next thing to do is look for the scratch register }
  port [baseaddr + 7] := $55;
  if port [baseaddr + 7] <> $55 then begin
    detect_UART := 1;
    exit;
  end;
  port [baseaddr + 7] := $AA;
  if port [baseaddr + 7] <> $AA then begin
    detect_UART := 1;
    exit;
  end;
  { then check if there's a FIFO }
  port [baseaddr + 2] := 1;
  x := port [baseaddr + 2];
  { some old-fashioned software relies on this! }
  port [baseaddr + 2] := $00;
  if (x and $80) = 0 then begin
    detect_UART := 2;
    exit;
  end;
  if (x and $40) = 0 then begin
    detect_UART := 3;
    exit;
  end;
  detect_UART := 4;
end;

procedure getPorts;
label do_printer;
const
  ports : array [1..4] of word = (
    $03f8,
    $02f8,
    $03e8,
    $02e8 );

var
  port, offs : word;
  i : byte;
begin
  verbose_write ('Checking UARTs...');
  if running_winnt = true then begin
    verbose_write ('Not done due to Windows NT');
    outputlist.ins ('This computer is running Microsoft''s Windows NT.  This program cannot probe');
    outputlist.ins ('   the communication ports - this has proven to cause problems.  Skipping to');
    outputlist.ins ('   the next section.');
    goto do_printer;
  end;
  i := 0;
  for offs := 1 to 4 do
    if detect_UART (ports [offs]) <> 0 then
      i := i + 1;
  outputlist.ins (numstr (i) + ' communications port(s) detected');
  if i > 0 then
    for offs := 1 to 4 do begin
      outputstring := (indent + 'COM' + numstr (offs) + ' at 0x' + dectohex16 (ports [offs]));
      outputstring := outputstring + ' has ' + strpas (uarts [detect_UART (ports [offs])]);
      outputlist.ins (outputstring);
    end;
  verbose_writeln ('Done');
do_printer:
  verbose_write ('Checking printer port information...');
  offs := $08;
  i := 1;
  repeat
    inc (offs, 2);
    inc (i);
  until (offs > $0e) or (memw [$40:offs] = 0);
  outputlist.ins (numstr (i - 1) + ' printer port(s) detected');
  offs := $08;
  i := 1;
  outputstring := '    ';
  repeat
    outputlist.ins (indent + 'LPT' + numstr (i) + ' at 0x' + dectohex16 (memw [$40:offs]));
    inc (offs, 2);
    inc (i);
  until (offs > $0e) or (memw [$40:offs] = 0);
  {outputlist.ins (outputstring);}
  verbose_writeln ('Done');
end;

procedure display (the_list : list);
const
  ENTER = #$0d;
  text_bg_color = 0; { black }
  text_fg_color = 7; { light gray }
  header_bg_color = 7;
  header_fg_color = 0;
  header_number = 0;
  header_color = (header_bg_color shl 4) + header_fg_color;
  text_color = (text_bg_color shl 4) + text_fg_color;

var
  cur : word;
  count, i : longint;
  inputfile : text;
  limit : byte;
  scrwidth : word;

  procedure display_help;
  const
    presskeymsg : pchar = '-- Press any key to continue --';
  begin
    clear_screen;
    writeln ('   Key    Description');
    writeln ('+');
    writeln ('F1, H':6, indent, 'Show this help screen');
    writeln ('PgDn':6, indent, 'Moves down one screenfull');
    writeln ('PgUp':6, indent, 'Moves up one screenfull');
    writeln ('Up':6, indent, 'Moves up one line');
    writeln ('Down':6, indent, 'Moves down one line');
    writeln ('B':6, indent, 'Moves to the beginning of the information');
    writeln ('E':6, indent, 'Moves to the end of the information');
    write_xy (41 - (strlen (presskeymsg) div 2), 24, strpas (presskeymsg), text_color);
    readekey;
    if keyepressed then
      readekey;
    clear_screen;
  end;

  procedure printfile;
  var
    i, count : word;
  begin
    clear_screen;
    setcursor (cur);
    count := the_list.count;
    for i := 0 to count do begin
      writeln (lst, the_list.getitem (i));
      write (output, 'Writing line #', i:10, ' to printer', #$0D);
    end;
    setcursor (NO_CURSOR);
  end;

  procedure showfile;
  var
    i, startIndex : longint;
    keycode : char;
    y, startcol : byte;
    redraw, done : boolean;
    outputstr : string [80];
  begin
    startcol := 1;
    done := false;
    cur := getcursor;
    setcursor ($2000);
    startIndex := 1;
    redraw := true;
    clear_screen;
    repeat
      if redraw then begin
        cleararea (0, 0, 79, 0, (header_bg_color shl 4) + header_fg_color);
        y := 1;
        write_xy (0, 0, 'System Information  Press F1 for help', header_color);
        the_list.reset;
        if startindex <> 1 then
          for i := 1 to startindex do
            the_list.next;
        for i := 1 to limit + 1 do begin
          outputstr := copy (the_list.get, startcol, startcol + scrwidth);
          outputstr := outputstr + spaces (80 - length (outputstr));
          write_xy (0, y, outputstr, (text_bg_color shl 4) + text_fg_color);
          inc (y);
          the_list.next;
        end;
        redraw := false;
      end;
      keycode := readkey;
      if keycode = #0 then begin
        if keypressed then begin
          keycode := readkey;
          case keycode of
            endkey : begin
              if count > limit then begin
                if (startindex + limit - 1) <> count then begin
                  startindex := count - limit + 1;
                  redraw := true;
                end;
                the_list.reset;
                for i := 1 to startindex - 1 do
                  the_list.next;
                redraw := true;
              end;
            end;
            home : begin
              if startindex <> 1 then begin
                redraw := true;
                the_list.reset;
                startindex := 1;
              end;
            end;
            f1 : begin
              display_help;
              redraw := true;
            end;
            pgup : begin
              if startindex = 1 then
                redraw := false
              else begin
                redraw := true;
                if startindex >= 1 then begin
                  startindex := startindex - limit;
                  if startindex < 1 then
                    startindex := 1;
                end;
                the_list.reset;
                if startindex > 1 then
                  for i := 1 to startindex do
                    the_list.next;
              end;
            end;
            pgdn : begin
              if count > limit then begin
                if startindex + limit - 1 <> count then begin
                  redraw := true;
                  startindex := startindex + limit;
                  if (startindex + limit) > count then
                    startindex := count - limit + 1;
                  the_list.reset;
                  for i := 1 to startindex - 1 do
                    the_list.next;
                end;
              end;
            end;
            uparrow : begin
              if startindex = 1 then
                redraw := false
              else begin
                if startIndex >= 1 then begin
                  startindex := startIndex - 1;
                  the_list.prev;
                  if startindex < 1 then begin
                    the_list.reset;
                    startindex := 1;
                    redraw := false;
                  end;
                end;
                redraw := true;
              end;
            end;
            downarrow : begin
              if (startIndex + limit) <= count then begin
                startindex := startindex + 1;
                the_list.next;
                redraw := true;
              end;
            end;
          end; { case }
        end;
      end else begin
        case keycode of
          enter, esc, 'q', 'Q' : done := true;
          'p', 'P' : printfile;
          'h', 'H' : begin
            display_help;
            redraw := true;
          end;
          'e', 'E' : begin
            if count > limit then begin
              if (startindex + limit - 1) <> count then begin
                startindex := count - limit + 1;
                redraw := true;
              end;
              the_list.reset;
              for i := 1 to startindex - 1 do
                the_list.next;
              redraw := true;
            end;
          end;
          'b', 'B' : begin
            if startindex <> 1 then begin
              redraw := true;
              the_list.reset;
              startindex := 1;
            end;
          end;
          'g', 'G' : begin
            setcursor (cur);
            clear_screen;
            write ('Go to what line:  ');
            readln (startindex);
            if startindex <= 1 then
              the_list.reset
            else if startindex + limit > count then
              startindex := count - limit + 1;
            redraw := true;
            setcursor (NO_CURSOR);
            the_list.reset;
            for i := 1 to startindex do
              the_list.next;
          end;
        end; { case }
      end;
    until done;
    setcursor (cur);
  end;

var
  code : integer;
  character : char;

begin
  limit := mem [$40:$84] - 1;
  scrwidth := memw [$40:$4a];
  count := the_list.count;
  textcolor (text_fg_color);
  textbackground (text_bg_color);
  showfile;
  textbackground (black);
  clear_screen;
end;

const
  bus_eisa = 0;
  bus_isa = 1;
  bus_mca = 2;
  bus_isa_pci = 3;

function getbustype : byte;
var
  works : boolean;
  data_seg, data_ofs : word;
  test : string [4];
begin
  getbustype := bus_isa;
  works := false;
  if not works then begin                { EISA }
    test := 'EISA';
    test [2] := chr (mem [$f000:$ffd9]);
    test [1] := chr (mem [$f000:$ffda]);
    test [4] := chr (mem [$f000:$ffdb]);
    test [3] := chr (mem [$f000:$ffdc]);
    if test = 'EISA'  then begin
      works := true;
      getbustype := bus_eisa;
    end;
  end;
  if not works then begin               { MCA }
    asm
      mov ah,0c0h
      int 15h
      cmp ah,0
      jnz @nope
      mov works,true
      mov data_seg,es
      mov data_ofs,bx
    @nope:
    end;
    if works then
      if (mem [data_seg:data_ofs + 5] and 2) = 2 then
        getbustype := bus_mca
      else
        works := false;
  end;

  if not works then begin                { PCI }
    asm
      mov ax,$b101
      int $1a
      cmp ah,00
      jne @nope
      mov works,true
    @nope:
    end;
    if works then getbustype := bus_isa_pci;
  end;
  if not works then getbustype := bus_isa
end;

type
  video_mode_info_rec = record
    vidmode : word;
    modename : pchar;
  end;

procedure use_bios;
var
  card : cardtype;
begin
  card := videocard;
  outputstring := strpas (CardTypeStrings [integer (VideoCard)]);
  verbose_write ('Video type ' + numstr (integer (VideoCard)) + ' ');
  outputlist.ins (indent + 'BIOS reports a ' + outputstring + ' card');
end;

function power (base, exponent : real) : real;
var
  value : real;
  i : real;
begin
  value := base;
  i := 1;
  repeat
    value := value * base;
    i := i + 1;
  until i = exponent;
  power := value;
end;

procedure getHardware;

  procedure probe_mca_bus;
  var
    slot_number : byte;
    slot_count : byte;
    device_number : word;
  begin
    verbose_write ('Counting MCA devices...');
    slot_count := mca_count_devices;
    verbose_writeln ('Done;  ' + numstr (slot_count) + ' devices');
    verbose_write ('Probing MCA bus...');
    case slot_count of
      0 : outputlist.ins (indent + 'There are no MCA devices');
      1 : outputlist.ins (indent + 'There is 1 MCA device');
      else outputlist.ins (indent + 'There are ' + numstr (slot_count) + ' MCA devices');
    end;
    for slot_number := 0 to 7 do begin
      device_number := mca_probe_slot (slot_number);
      outputstring := indent + indent + 'Slot ' + numstr (slot_number + 1) + ':' + indent;
      if device_number <> $ffff then
        outputlist.ins (outputstring + mca_resolve_desc (device_number))
      else
        outputlist.ins (outputstring + 'No device');
    end;
    verbose_writeln ('Done');
  end;

var
  infoseg, infoofs : word;
  vidinfo : vesainfo;
  status, mode : word;
  bustype, i, result : byte;
  pci_info : pci_info_block;
  devinfo : pci_device_info;
  bustype_str : string [20];
  device_type : word;
  vesa_mode : vesa_mode_info;

label vesa;
begin
  verbose_write ('Getting expansion bus type...');
  bustype := getbustype;
  case bustype of
    bus_eisa : bustype_str := 'EISA';
    bus_isa : bustype_str := 'ISA';
    bus_isa_pci : bustype_str := 'ISA/PCI';
    bus_mca : bustype_str := 'Microchannel';
  end;
  verbose_writeln ('Done');
  outputlist.ins (bustype_str + ' Subsystem detected');
  if bustype = bus_mca then begin
    verbose_write ('Probing MCA bus...');
    probe_mca_bus;
    verbose_write ('Done');
  end;
  if bustype <> bus_isa_pci then
    goto vesa;                        { Don't bother with the PCI bus -
                                        there isn't one!  Note that the
                                        PCI probing code will break on
                                        CPUs < i386; it uses 32-bit code }
  if pci_detect (pci_info) then begin { -- Scan the PCI Bus -- }
    verbose_write ('Probing PCI bus...');
    outputstring := indent + 'PCI Bus version is ' + dectohex8 (pci_info.major) + '.' + dectohex8 (pci_info.minor);
    outputlist.ins (outputstring);
    if pci_info.last_pci_bus = 0 then
      outputlist.ins (indent + 'There is 1 PCI bus')
    else
      outputlist.ins (indent + 'There are ' + numstr (pci_info.last_pci_bus) + ' PCI busses');
    pci_scan_bus (pci_info);
    if pci_count_devices = 0 then begin
      outputlist.ins (indent + 'There are no PCI devices');
      goto vesa;
    end else if pci_count_devices = 1 then
      outputlist.ins (indent + 'There is one PCI device')
    else
      outputlist.ins (indent + 'There are ' + numstr (pci_count_devices) + ' PCI devices');
    for i := 1 to pci_count_devices do begin
      pci_get_device_info (i, devinfo);
      with devinfo do begin
        outputstring := indent + 'Bus ' + numstr (bus) + ' Device ' + numstr (device)
                        + ' Function ' + numstr (func);
        outputlist.ins (outputstring);
        with dev_data do begin
          outputstring := indent + indent + 'Vendor: 0x';
          outputstring := outputstring + dectohex16 (vendor) + indent;
          outputstring := outputstring + pci_resolve_vendor (vendor);
          outputlist.ins (outputstring);
          outputstring := indent + indent + 'Device: 0x';
          outputstring := outputstring + dectohex16 (device) + indent;
          outputstring := outputstring + pci_resolve_device (vendor, device);
          outputlist.ins (outputstring);
          outputstring := indent + indent + '  Header type:' + indent;
          case header_type and $7f of
            $00 : outputstring := outputstring + 'non-bridge';
            $01 : outputstring := outputstring + 'PCI-to-PCI bridge';
            $02 : outputstring := outputstring + 'PCI-to-CardBus bridge';
          end;
          outputlist.ins (outputstring);
          outputstring := indent + indent + '  Device type:' + indent;
          if (header_type and $80) <> 0 then
            outputstring := outputstring + 'multi-function'
          else
            outputstring := outputstring + 'single-function';
          outputlist.ins (outputstring);
          if (irq_line <> 0) and (irq_line <> $ff) then
             outputlist.ins (indent + indent + indent+ 'IRQ Line:' + indent + numstr (irq_line));
          device_type := (class shl 8) + subclass;
          outputstring := indent + indent + ' Device class:' + indent + '0x';
          outputstring := outputstring + dectohex16 (device_type) + ' (' + pci_resolve_class (device_type) + ')';
          outputlist.ins (outputstring);
        end;
      end;
    end;
    verbose_writeln ('Done;  ' + numstr (pci_count_devices) + ' devices');
  end;
vesa:
  outputlist.ins ('Video:');
  verbose_write ('Using BIOS to detect video type...');
  use_bios;
  verbose_writeln ('Done');
  if vesa_detect (vidinfo) then begin
    { --- VESA Version --- }
    verbose_writeln ('Using VESA interface to detect graphics...');
    outputstring := indent + 'BIOS supports VESA ';
    outputstring := outputstring + numstr ((vidinfo.version and $FF00) shr 8);
    outputstring := outputstring + '.' + numstr ((vidinfo.version and $00FF) shr 8);
    verbose_writeln (indent + 'version 0x' + dectohex16 (vidinfo.version) + ' reported');
    outputlist.ins (outputstring);
    { --- Video Memory --- }
    outputlist.ins (indent + numstr (vidinfo.memory * 64) + 'K video memory found');
    { --- OEM Name --- }
    if strcomp (pchar (vidinfo.oemname), '76129552') = 0 then begin
      outputlist.ins (indent + 'OEM is ATI');
    end else begin
      outputstring := strpas (pointer (vidinfo.oemname));
      outputlist.ins (indent + 'OEM is ' + outputstring);
    end;
    { --- Supported Modes --- }
    outputlist.ins (indent + 'Supported video modes:');
    i := 0;
    infoseg := seg (vesa_mode);
    infoofs := ofs (vesa_mode);
    repeat
      mode := word (pointer (vidinfo.modes_supported + i)^);
      if vesa_get_mode_info (mode, vesa_mode) then begin
        with vesa_mode do begin
          outputstring := indent + indent + 'Mode 0x' + dectohex16 (mode);
          outputstring := outputstring + ':  ' + numstr (width) + ' x ';
          outputstring := outputstring + numstr (height) + ' x ';
          str (power (2, bits_per_pixel):1:0, tempstr);
          outputstring := outputstring + strcommas (tempstr) + ' colors; ';
          case memory_model of
            $00 : tempstr := 'text';
            $01 : tempstr := 'CGA';
            $02 : tempstr := 'HGC';
            $03 : tempstr := '16-color (EGA)';
            $04 : tempstr := 'packed pixel';
           { $05 : tempstr := 'sequ 256" (non-chain 4) graphics';}
            $05 : tempstr := 'sequ 256"';
            $06 : tempstr := 'HiColor';
          end;
        end;
        outputlist.ins (outputstring + 'Type: ' + tempstr);
      end;
      i := i + 2;
    until mode = $FFFF;
    verbose_writeln ('Done');
  end;
end;

procedure getConfiguration;
type
  tromtable = record
    size : word;
    model : byte;
    submodel : byte;
    revision : byte;
    features : array [1..5] of byte;
  end;
  bios_str = array [1..8] of char;

var
  regs : registers;
  romtable : ^tromtable;
  bios_date : ^bios_str;
  computer_type : string;
  model, submodel, revision : byte;

begin
  verbose_write ('Reading BIOS model information...');
  regs.ah := $c0;
  intr ($15, regs);
  romtable := ptr (regs.es, regs.bx);
  model := romtable^.model;
  submodel := romtable^.submodel;
  revision := romtable^.revision;
  bios_date := ptr ($F000, $FFF5);
  computer_type := 'Unknown system model';
  case model of
    $ff : begin
      case submodel of
        $00 : computer_type := 'Tandy 1000SL';
        $01 : computer_type := 'Tandy 1000TL';
        $40 : computer_type := 'Olivetti M15';
        else begin
          if bios_date^ = '04/24/81' then
            computer_type := 'Original PC'
          else if bios_date^ = '10/19/81' then
            computer_type := 'Original bugfixed PC'
          else if bios_date^ = '10/27/82' then
            computer_type := 'Original PC with 640K, HD and EGA';
        end;
      end;
    end;
    $fe : begin
      case submodel of
        $43 : computer_type := 'Olivetti M240';
        $A6 : computer_type := 'Quadram Quad386';
        else begin
          if bios_date^ = '08/16/82' then
            computer_type := 'PC XT'
          else if bios_date^ = '11/08/82' then
            computer_type := 'PC XT or Portable';
        end;
      end;
    end;
    $fd : begin
      if bios_date^ = '06/01/83' then
        computer_type := 'PCjr';
    end;
    $fc : begin
      case submodel of
        $00 : case revision of
          $00 : computer_type := 'PC3720/AT';
          $01 : if bios_date^ = '06/10/85' then
            computer_type := 'AT model 239'
          else
            computer_type := '7531/2 Industrial AT';
        end; { case revision $00 }
        $01 : if (revision = $04) and (bios_date^ = '02/25/93') then
          computer_type := 'DOSEmu'
        else
          computer_type := 'AT Clone';
        $02 : begin
          if bios_date^ = '08/11/88' then
            computer_type := 'SoftWindows'
        end;
        $04 : begin
          if (bios_date^ = '02/13/87') and (revision = $00) then
            computer_type := 'PS/2 Model 50 (''286 @ 10 MHz)'
          else if (bios_date^ = '05/09/87') and (revision = $01) then
            computer_type := 'PS/2 Model 50 (''286 @ 10 MHz)'
          else if (revision = $02) and (bios_date^ <> '01/28/88') then
            computer_type := 'PS/2 Model 50 (''286 @ 10 MHz)'
          else
            computer_type := 'PS/2 Model 50Z (''286 @ 10 MHz)';
        end;
        $05 : computer_type := 'PS/2 Model 60 (''286 @ 10 MHz)';
        $06 : begin
          case revision of
            $00 : computer_type := 'IBM 7552-140 "Gearbox"';
            $01 : computer_type := 'IBM 7552-540 "Gearbox"';
          end;
        end;
        $09 : computer_type := 'PS/2 Model 25 or Model 30';
        else begin
          if bios_date^ = '01/10/84' then
            computer_type := 'AT model 068 or model 099'
        end;
      end;
    end;
    $fa : begin
      case submodel of
        $00 : computer_type := 'PS/2 Model 30 (8 MHz ''86)';
        $01 : computer_type := 'PS/2 Model 25/25L (8 MHz ''86)';
        $30 : computer_type := 'IBM Restaurant Terminal';
        $FE : computer_type := 'IBM PCradio 9075';
      end;
    end;
    $f8 : begin
      case submodel of
        $00 : begin
          if revision = $00 then begin
            if bios_date^ = '03/30/87' then
              computer_type := 'PS/2 Model 80 (16 MHz ''386)'
            else
              computer_type := 'PS/2 Model 75 486 (33 Mhz ''486)'
          end;
        end;
        $01 : computer_type := 'PS/2 Model 80 (20 MHz ''386)';
        $02 : computer_type := 'PS/2 Model 55-5571';
        $04 : computer_type := 'PS/2 Model 70 (20 MHz ''386DX)';
        $05 : computer_type := 'IBM PC 7568';
        $06 : computer_type := 'PS/2 Model 55-5571';
        $07 : begin
          case revision of
            $00 : computer_type := 'IBM PC 7561/2';
            $01 : computer_type := 'PS/2 Model 55-5551';
            $02 : computer_type := 'IBM PC 7561/2';
            $03 : computer_type := 'PS/2 Model 55-5551';
          end;
        end;
        $09 : computer_type := 'PS/2 Model 70 (16 MHz ''386DX)';
        $0B : computer_type := 'PS/2 Model P70';
        $0C : computer_type := 'PS/2 Model 55SX (16 MHz ''386SX)';
        $0D : begin
          if bios_date^ = '12/01/89' then
            computer_type := 'PS/2 Model 70 (25 MHz ''486)'
          else
            computer_type := 'PS/2 Model 70 (25 MHz ''386)';
        end;
        $0E : computer_type := 'PS/1 ''486SX';
        $0F : computer_type := 'PS/1 ''486DX';
        $10 : computer_type := 'PS/2 Model 55-5551';
        $11 : computer_type := 'PS/2 Model 90 XP (25 MHz ''486)';
        $12 : computer_type := 'PS/2 Model 95 XP';
        $13 : computer_type := 'PS/2 Model 90 XP (33 MHz ''486)';
        $14 : computer_type := 'PS/2 Model 90-AK9 (25 MHz ''486) or 95 XP';
        $15 : computer_type := 'PS/2 Model 90 XP';
        $16 : computer_type := 'PS/2 Model 90-AKD/95XP486 (33 MHz ''486)';
        $17 : computer_type := 'PS/2 Model 90 XP';
        $19 : computer_type := 'PS/2 Model 35/35LS/35SX/40/40SX (20 MHz ''386 SX)';
        $1A : computer_type := 'PS/2 Model 95 XP';
        $1B : begin
          computer_type := 'PS/2 Model 70-486';
          if bios_date^ = '09/29/89' then
            computer_type := computer_type + ' (25 MHz ''386DX)'
          else
            computer_type := computer_type + ' (25 MHz ''486)';
        end;
        $1C : computer_type := 'PS/2 Model 65-121/65SX (16 MHz ''386SX)';
        $1E : computer_type := 'PS/2 Model 55LS (16 MHz ''386SX)';
        $23 : computer_type := 'PS/2 Model L40 SX (20 MHz ''386SX)';
        $25 : begin
          case revision of
            $00 : computer_type := 'PS/2 Model 57 SLC';
            $06 : computer_type := 'PS/2 Model M57 (20 MHz ''386SLC)';
            $09 : computer_type := 'PS/2 Model 56 SLC (20 MHz ''386SLC)';
          end;
        end;
        $26 : begin
          case revision of
            $00 : computer_type := 'PS/2 Model 57 SX';
            $01 : computer_type := 'PS/2 Model 57 (20 MHz ''386SX)';
            $02 : computer_type := 'PS/2 Model 57 SX (20 MHz ''386SX, SCSI)';
          end;
        end;
        $28 : computer_type := 'PS/2 Model 95 XP';
        $29 : computer_type := 'PS/2 Model 90 XP';
        $2A : computer_type := 'PS/2 Model 95 XP (50 MHz ''486DX)';
        $2B : computer_type := 'PS/2 Model 90/90XP486 (50 MHz ''486DX)';
        $2C : begin
          case revision of
            $02 : computer_type := 'PS/2 Model 95 XP';
            $01 : computer_type := 'PS/2 Model 95 (20 MHz ''486SX)';
          end;
        end;
        $2D : computer_type := 'PS/2 Model 90 XP (20 MHz ''486SX)';
        $2E : begin
          computer_type := 'PS/2 Model 95 XP/XP486';
          if revision = $01 then
            computer_type := 'PS/2 Model 95 (20 MHz ''486SX + ''487SX)';
        end;
        $2F : computer_type := 'PS/2 Model 90 XP (20 MHz ''486SX + ''487SX)';
        $30 : computer_type := 'PS/1 Model 2121 (16 MHz ''386SX)';
        $33 : computer_type := 'PS/2 Model 30-''386';
        $34 : computer_type := 'PS/2 Model 25-''386';
        $36 : computer_type := 'PS/2 Model 95 XP';
        $37 : computer_type := 'PS/2 Model 90 XP';
        $38 : computer_type := 'PS/2 Model 57';
        $39, $40 : computer_type := 'PS/2 Model 95 XP';
        $3F : computer_type := 'PS/2 Model 90 XP';
        $41 : computer_type := 'PS/2 Model 77';
        $45 : computer_type := 'PS/2 Model 90 XP (Pentium)';
        $46 : computer_type := 'PS/2 Model 95 XP (Pentium)';
        $47 : computer_type := 'PS/2 Model 90/95 E (Pentium)';
        $48 : computer_type := 'PS/2 Model 85';
        $49 : computer_type := 'PS/ValuePoint 325T';
        $4A : computer_type := 'PS/ValuePoint 425SX';
        $4B : computer_type := 'PS/ValuePoint 433DX';
        $4E : computer_type := 'PS/2 Model 295';
        $50 : computer_type := 'PS/2 Model P70';
        $52 : computer_type := 'PS/2 Model P75 (33 MHz ''486)';
        $56 : computer_type := 'PS/2 Model CL57 SX';
        $57, $59, $5B : computer_type := 'PS/2 Model 90 XP';
        $58, $5A, $5C : computer_type := 'PS/2 Model 95 XP';
        $5D : computer_type := 'PS/2 Model N51 SLC';
        $5E : computer_type := 'IBM ThinkPad 700';
        $80 : computer_type := 'PS/2 Model 80 (20 MHz ''386)';
        $81 : computer_type := 'PS/2 Model 55-5502';
        $87 : computer_type := 'PS/2 Model N33SX';
        $88 : computer_type := 'PS/2 Model 55-5530T';
        $97 : computer_type := 'PS/2 Model 55 Note N23SX';
        $99 : computer_type := 'PS/2 Model N51 SX';
      end;
    end;
  end;
  if model = $e1 then computer_type := 'PS/2 Model 55-5530';
  outputlist.ins ('System information:');
  outputstring := indent + 'BIOS Model is ' + dectohex8 (model)
                                            + dectohex8 (submodel)
                                            + dectohex8 (revision);
  outputlist.ins (outputstring + ' (' + computer_type + ')');
  outputlist.ins (indent + 'BIOS Date is ' + bios_date^);
  verbose_writeln ('Done');
end;

procedure help;
begin
  writeln ('compinfo - 1994-1998 by Phil Brutsche');
  writeln ('Syntax:');
  writeln ('  compinfo -? - This help');
  writeln ('  compinfo -p - Sends the output to lpt1');
  writeln ('  compinfo -f:filename - Sends the output to filename');
  writeln ('  compinfo -v - Verbose output.  Test information will be output to the file');
  writeln ('    results.inf');
  writeln ('  compinfo -i - Interactive mode');
end;

var
  localfile : boolean;
  i, count : longint;
  interactive_mode : boolean;

procedure interactive (outstring : string);
var
  ch : char;
begin
  if interactive_mode then begin
    writeln ('Interactive mode: ', outstring);
    write ('Continue?  (y/n): ');
    ch := readkey;
    writeln (ch);
    if upcase (ch) = 'N' then
      halt;
  end;
end;

begin
  {video_mode := lastmode;}
  writeln ('Main program has control');
  write ('Setting up variables...');
  interactive_mode := false;
  outputlist.init;
  check_fpu := 1;
  out := @output;
  localfile := false;
  verbose := false;
  writeln ('Done');
  writeln ('Parsing command line parameters:');
  for i := 1 to paramcount do begin
    param := paramstr (i);
    if (upcase (param [2]) = 'V') and (param [1] in ['-', '/']) then begin
      verbose := true;
      assign (outputfile, 'results.inf');
      rewrite (outputfile);
      writeln ('  Verbose');
    end else if (strupr (copy (param, 1, 2)) = '-I') or (strupr (copy (param, 1, 2)) = '/I') then begin
      interactive_mode := true;
      writeln ('  Interactive');
    end else if (param [2] = '?') and ((param [1] = '-') or (param [1] = '/')) then begin
      help;
      halt;
    end else if (strupr (copy (param, 1, 2)) = '-P') or (strupr (copy (param, 1, 2)) = '/P') then begin
      out := @lst;
      writeln ('  Printer');
    end else if (strupr (copy (param, 1, 2)) = '-D') or (strupr (copy (param, 1, 2)) = '-D') then begin
      writeln ('Compinfo release date:  ', compinfo_release_date);
      halt;
    end else if (strupr (copy (param, 1, 2)) = '-F') or (strupr (copy (param, 1, 2)) = '/F') then begin
      delete (param, 1, 3);
      if param = '' then begin
         writeln ('No filename specified');
         exit;
      end;
      assign (outfile, param);
      rewrite (outfile);
      out := @outfile;
      localfile := true;
      writeln ('  File output');
    end else begin
      writeln ('Invalid parameter');
      exit;
    end;
  end;
  writeln ('Done');
  writeln ('Checking for Windows NT...');
  if is_winnt then begin
    verbose_writeln ('Warning! WinNT!');
    writeln ('This program is running under Windows NT.  Various tests have proven to not');
    writeln ('work very well.  A warning will be displayed at those tests.');
    writeln ('Please press the space bar to quit, or any other key to continue.');
    running_winnt := true;
    if readkey = ' ' then
      halt;
  end else
    running_winnt := false;
  writeln ('Beginning tests...');
  getDrives;
  interactive ('Drive info done');
  getDisk;
  interactive ('Disk capacity info done');
  getMemory;
  interactive ('Memory info done');
  getProcessor;
  interactive ('Processor info done');
  if check_fpu = 1 then begin
    getMath;
    interactive ('NPU info done');
  end;
  getHardware;
  interactive ('Hardware info done');
  getConfiguration;
  interactive ('Configuration info done');
  getPorts;
  interactive ('Port info done');
  getDateTime;
  interactive ('Date & time info done');
  getOSVersion;
  interactive ('OS info done');
  getDPMI;
  interactive ('DPMI info done');
  outputlist.ins ('');
  {textmode (video_mode);}
  if localfile then begin
    outputlist.reset;
    count := outputlist.count;
    for i := 1 to count do begin
      writeln (out^, outputlist.get);
      outputlist.next;
    end;
    close (outfile);
  end else
    display (outputlist);
  outputlist.free;
  verbose_write ('Program terminated normally');
  if verbose then close (outputfile);
end.
