/*
 * Copyright 2021-2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.tooling.core.htmlreport;

import com.microsoft.playwright.Page;
import com.microsoft.playwright.Page.GetByRoleOptions;
import com.microsoft.playwright.Page.ScreenshotOptions;
import com.microsoft.playwright.junit.UsePlaywright;
import com.microsoft.playwright.options.AriaRole;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.MediaType;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestReporter;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.api.io.TempDir;

import java.nio.file.Path;
import java.util.List;
import java.util.regex.Pattern;

import static com.microsoft.playwright.assertions.PlaywrightAssertions.assertThat;
import static java.util.Objects.requireNonNull;
import static org.assertj.core.api.Assertions.assertThat;

@UsePlaywright
@ExtendWith(PlaywrightTracePublisher.class)
class DefaultHtmlReportWriterBrowserTests {

	@TempDir
	static Path tempDir;
	static Path sampleHtmlReport;

	@BeforeAll
	static void createHtmlReport() throws Exception {
		var sampleXmlReport = Path.of(requireNonNull(System.getProperty("sampleXmlReport")));
		assertThat(sampleXmlReport).exists().hasExtension("xml");

		sampleHtmlReport = tempDir.resolve("open-test-report.html");
		new DefaultHtmlReportWriter().writeHtmlReport(List.of(sampleXmlReport), sampleHtmlReport);
	}

	@BeforeEach
	void openHtmlReport(Page page) {
		page.setViewportSize(1920, 1080);
		page.navigate(sampleHtmlReport.toUri().toString());
		assertThat(page).hasTitle(Pattern.compile("Open Test Report"));
	}

	@Test
	void sampleHtmlReportIsRenderedCorrectly(Page page, TestReporter reporter) {
		reporter.publishFile("screenshot-failure.png", MediaType.IMAGE_PNG,
			file -> page.screenshot(new ScreenshotOptions().setPath(file)));

		assertThat(
			page.getByRole(AriaRole.HEADING, new GetByRoleOptions().setName("failed(TestReporter)"))).isVisible();
		assertThat(page.getByRole(AriaRole.STATUS)).containsText("FAILED");

		page.getByRole(AriaRole.SWITCH).click();
		page.getByRole(AriaRole.LINK, new GetByRoleOptions().setName("KitchenSinkTests").setExact(true)).click();

		reporter.publishFile("screenshot-success.png", MediaType.IMAGE_PNG,
			file -> page.screenshot(new ScreenshotOptions().setPath(file)));

		assertThat(page.getByRole(AriaRole.HEADING, new GetByRoleOptions().setName("KitchenSinkTest"))).isVisible();
		assertThat(page.getByRole(AriaRole.STATUS)).containsText("SUCCESSFUL");
	}

	@Test
	void canOpenAttachments(Page page) {
		page.navigate(sampleHtmlReport.toUri().toString());
		assertThat(page.getByText("Hellö, Wörld!")).isVisible();

		page.getByText(Pattern.compile("[/\\\\]test.txt$"), new Page.GetByTextOptions().setExact(false)).click();
		assertThat(page.getByText("Hellö, Wörld!")).isVisible();
	}
}
