use Test2::V0;
use Bitcoin::Crypto qw(btc_block btc_transaction);
use Bitcoin::Crypto::Util qw(to_format);

subtest 'Block Construction' => sub {
	my $custom_prev_hash = "\x00" x 32;
	my $custom_merkle_root = "\x11" x 32;
	my $custom_timestamp = 1234567890;
	my $custom_bits = 0x1d00ffff;
	my $custom_nonce = 12345;
	my $custom_height = 100;

	ok lives {
		my $block = btc_block->new(height => 0);
	}, "Block with height 0 created successfully";

	ok lives {
		my $block = btc_block->new(
			version => 2,
			prev_block_hash => $custom_prev_hash,
			merkle_root => $custom_merkle_root,
			timestamp => $custom_timestamp,
			bits => $custom_bits,
			nonce => $custom_nonce,
			height => $custom_height,
		);
	}, "Block with custom parameters created successfully";

	ok lives {
		my $block = btc_block->new(
			version => 2,
			height => 22
		);
		is $block->height, 22, 'height ok';
	}, 'Block with custom set height does not try to recalculate it from coinbase';
};

subtest 'Error Handling' => sub {
	my $err = dies {
		btc_block->from_serialized([hex => "deadbeef"]);
	};

	is $err->message, 'serialized block data too short for header',
		'Correct error message for invalid serialized data';

	$err = dies {
		my $block = btc_block->new(height => 10);
		$block->merkle_root;    # No transactions
	};

	is $err->message, 'cannot calculate merkle root for empty block', 'Correct error message for short block data';
};

subtest 'Mainnet Block Data Parsing' => sub {
	my @mainnet_blocks = (
		{
			hex =>
				"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",
			version => 0x20010000,
			height => 919991,
			prev_block_hash => "00000000000000000000a4720da2f181f73869066d9b09b2112de7c9fd471dbc",
			merkle_root => "ffbd3023f5dacf0f1c372b827de64659dbedddb56c5dfca11ff35da220944a36",
			timestamp => 1760990901,
			bits => 386001697,
			nonce => 3087424331,
			hash => "00000000000000000000c5dd7a3c996b97869cf6596320265eb23419b1493e38",
			tx_count => 4,
			weight => 4979,
			size => 1595,
		},
		{
			hex =>
				"04c02122383e49b11934b25e26206359f69c86976b993c7addc500000000000000000000935d2e162759b6f4a4f457fadf8454c43bdbe80e070b8d6aedbcf40046138694b896f66821eb01173d627ff503020000000001010000000000000000000000000000000000000000000000000000000000000000ffffffff2903b8090e04b896f6682f53424943727970746f2e636f6d20506f6f6c2f046cffd07400000000000000ffffffff027264a012000000001600145740f0702f345bf530296b8309453a511d40d7c30000000000000000266a24aa21a9edeeab68725eef9deca81b0f5e4bd4514a2b674a15e473954d4690a0d2574a025f012000000000000000000000000000000000000000000000000000000000000000000000000001000000000101b5480c52e57c1f106570240fef826959ce0ad5da588a8454ccad58d7446bfdca0100000000ffffffff0154cc06000000000016001428012b45374b571af08b5e60717333a77130f7cc02483045022100e04db6fd6b9b04342c67b55600e07de7ef07438fac7f565387291331677c80c902203370e5bc8aa2d9b35829f8a774898467267d7766d05ec9d95b61a4404376153a01210287dfc8809d70b92392c28c67154cd0d21e01ea6bcd33673d23874f4687c2a0c10000000002000000000104153895b13a7efca77f4b28a7069df5ae0af2324550ce2c14989db74cf359d5230100000000ffffffff1268ba22b48eb0c5a6400685cc674f5f7f3685739dbc341e39a6b78ce14db69c00000000001000000048dd3c49a7e5f25f3a7057cbbe5bd913e1def6534f3401a4ba38feffa16383f501000000001000000048dd3c49a7e5f25f3a7057cbbe5bd913e1def6534f3401a4ba38feffa16383f500000000001000000001f9010000000000002251207434164bd41e2185651f084b6a79e11ce57abe69093b7f939bb1c8786e5d233b0140f8887111e02908c74d7f8a9fcf8fdc55798074f1de795222bf01a5707ba72e4ab91690a48b75d510f9725f5c997b0801692f978fee1d18b195ae8aa4ccebb05a02002821025fc1cb47027fc2963b62e20b1f8f7f33fd9bb0eeb1c022931e8bbf84eec32641ac736460b26802002821038cf57c5d6e5c72af12d817fd9fd72a5e665621c2a9b10104b3552e6c488d384cac736460b2680200282103f986418c145d0b6571fbe665af18066dd704ae0ad138ffbaf8bddaca9f4a4b9dac736460b26800000000",
			version => 0x2221c004,
			height => 919992,
			prev_block_hash => "00000000000000000000c5dd7a3c996b97869cf6596320265eb23419b1493e38",
			merkle_root => "9486134600f4bced6a8d0b070ee8db3bc45484dffa57f4a4f4b65927162e5d93",
			timestamp => 1760990904,
			bits => 386001697,
			nonce => 4118766141,
			hash => "00000000000000000001086665340a8d6c9a068bb707696c6a087937c52b80fb",
			tx_count => 3,
			weight => 2543,
			size => 893,
		},
		{
			hex =>
				"00000920a3e94e9a49b83735b4f7601a70f8110402e7d6ded2920000000000000000000076b61417a5c91d8b7c7a779b1ade89e2e27b19c4ac3948e3091b8354368c6a2b0946ef68b4dd011765f454e401020000000001010000000000000000000000000000000000000000000000000000000000000000ffffffff29036b060e040946ef68537069646572506f6f6c2f383831382f5d8afc170310fcc19215000000000000ffffffff04205fa012000000001976a914717a4c9074577a05af94271c32b249d298a22d9888ac0000000000000000266a24aa21a9ede2f61c3f71d1defd3fa999dfa36953755c690689799962b48bebd836974e8cf900000000000000002f6a2d434f52450164db24a662e20bbdf72d1cc6e973dbb2d12897d596a6689031f48a857d344e1a42fdb272bb15d6210000000000000000126a10455853415401120f080304111f1200130120000000000000000000000000000000000000000000000000000000000000000000000000",
			version => 0x20090000,
			height => 919147,
			prev_block_hash => "0000000000000000000092d2ded6e7020411f8701a60f7b43537b8499a4ee9a3",
			merkle_root => "2b6a8c3654831b09e34839acc4197be2e289de1a9b777a7c8b1dc9a51714b676",
			timestamp => 1760511497,
			bits => 385998260,
			nonce => 3830772837,
			hash => "0000000000000000000110ed8af0660260e17528112c1382ef85d2e2a91a05df",
			tx_count => 1,
			weight => 1384,
			size => 373,
		},
		{
			hex =>
				"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",
			version => 0x24a32000,
			height => 918698,
			prev_block_hash => '00000000000000000001220dbaccf66a070658e6c7fd27b00ebd0e2c49e49cf4',
			merkle_root => 'eeea80c47c2fbdb58d29b3935c5e36c89b2cad57ab958676e88859c9dc6f1e3e',
			timestamp => 1760253491,
			bits => 385998260,
			nonce => 4047338056,
			hash => '000000000000000000000361336ce2d4a8cad44efc8744dc239f587be7ae9791',
			tx_count => 29,
			weight => 28722,
			size => 10656,
		}
	);

	for my $block_data (@mainnet_blocks) {
		my $block = btc_block->from_serialized([hex => $block_data->{hex}]);

		is($block->version, $block_data->{version}, 'Correct block version');
		is($block->height, $block_data->{height}, 'Block height matches expected value');
		is(scalar @{$block->transactions}, $block_data->{tx_count}, 'Block contains transactions');
		is(
			to_format [hex => $block->prev_block_hash],
			$block_data->{prev_block_hash},
			'Correct previous block hash'
		);

		is(to_format [hex => $block->merkle_root], $block_data->{merkle_root}, 'Correct merkle root');

		is($block->timestamp, $block_data->{timestamp}, 'Correct timestamp');
		is($block->bits, $block_data->{bits}, 'Correct bits');
		is($block->nonce, $block_data->{nonce}, 'Correct nonce');
		is(to_format [hex => $block->get_hash], $block_data->{hash}, 'Block hash matches expected value');
		is($block->weight, $block_data->{weight}, 'Correct block weight');
		is($block->size, $block_data->{size}, 'Correct block size');

		my $reserialized = $block->to_serialized;
		my $hex_reserialized = to_format [hex => $reserialized];
		is($hex_reserialized, $block_data->{hex}, 'Round-trip serialization matches original data');
	}
};

subtest 'should allow building block step by step' => sub {
	my $sole_block_tx =
		'010000000001010000000000000000000000000000000000000000000000000000000000000000ffffffff4e0304230d115768697465506f6f6c64012902f1607f29fabe6d6dfc49c9371003be5de29f5b812b250189efdf351915896223eb1fc1e8710706e0100000000000000000008c770700000000000000ffffffff04205fa0120000000017a914fd3938eda8ef5c62bdda522d8321d6ee6456a2c1870000000000000000266a24aa21a9ede2f61c3f71d1defd3fa999dfa36953755c690689799962b48bebd836974e8cf900000000000000002f6a2d434f5245015b9b30813264eaab2b70817a36c94733812e591c0b5996b2e8a83bdb76300105f7d4c2a79799604500000000000000002b6a2952534b424c4f434b3a1455ba850d3f676e54ef3a5e2cb9ef3738f707e7eee4d22b467ce024006616490120000000000000000000000000000000000000000000000000000000000000000000000000';
	my $block = btc_block->new;

	$block->set_timestamp(1726101953);
	$block->set_version(0x242ee000);
	$block->set_bits(0x1703098c);
	$block->set_nonce(0xb8ec7c27);
	$block->set_prev_block_hash([hex => '0000000000000000000138b6987659c57c81e21b0bf5eed47d2192e3c5bcffe5']);
	$block->add_transaction(btc_transaction->from_serialized([hex => $sole_block_tx]));

	is to_format [hex => $block->get_hash], '0000000000000000000021cbca8d7e3c778648070da31e394929ba0fc65c001c',
		'block hash ok';
};

done_testing();

