
#define TEST_NAME "core_keccak1600"
#include "cmptest.h"

static void
print_hex(const char *label, const unsigned char *data, size_t len)
{
    size_t i;
    printf("%s", label);
    for (i = 0; i < len; i++) {
        printf("%02x", data[i]);
    }
    printf("\n");
}

static int
compare_states(const char *label, const unsigned char *actual, const unsigned char *expected,
               size_t len)
{
    size_t i, j;
    for (i = 0; i < len; i++) {
        if (actual[i] != expected[i]) {
            printf("FAIL: %s mismatch at byte %u\n", label, (unsigned int) i);
            printf("  Expected: ");
            for (j = 0; j < len; j++) {
                printf("%02x", expected[j]);
            }
            printf("\n  Got:      ");
            for (j = 0; j < len; j++) {
                printf("%02x", actual[j]);
            }
            printf("\n");
            return -1;
        }
    }
    printf("PASS: %s\n", label);
    return 0;
}

int
main(void)
{
    crypto_core_keccak1600_state state;
    unsigned char                extracted[64];
    unsigned char                test_data[64];
    unsigned char                sentinel[64];
    crypto_core_keccak1600_state state_12, state_24;
    unsigned char                last_byte;
    size_t                       i;
    int                          test_failures = 0;
    int                          differs;

    /* Test vectors for Keccak-f[1600] (24 rounds) */
    /* Test vector 1: All-zero input for Keccak-f[1600] */
    static const unsigned char keccak_f_1600_zero_input[200]    = { 0 };
    static const unsigned char keccak_f_1600_zero_expected[200] = {
        0xe7, 0xdd, 0xe1, 0x40, 0x79, 0x8f, 0x25, 0xf1, 0x8a, 0x47, 0xc0, 0x33, 0xf9, 0xcc, 0xd5,
        0x84, 0xee, 0xa9, 0x5a, 0xa6, 0x1e, 0x26, 0x98, 0xd5, 0x4d, 0x49, 0x80, 0x6f, 0x30, 0x47,
        0x15, 0xbd, 0x57, 0xd0, 0x53, 0x62, 0x05, 0x4e, 0x28, 0x8b, 0xd4, 0x6f, 0x8e, 0x7f, 0x2d,
        0xa4, 0x97, 0xff, 0xc4, 0x47, 0x46, 0xa4, 0xa0, 0xe5, 0xfe, 0x90, 0x76, 0x2e, 0x19, 0xd6,
        0x0c, 0xda, 0x5b, 0x8c, 0x9c, 0x05, 0x19, 0x1b, 0xf7, 0xa6, 0x30, 0xad, 0x64, 0xfc, 0x8f,
        0xd0, 0xb7, 0x5a, 0x93, 0x30, 0x35, 0xd6, 0x17, 0x23, 0x3f, 0xa9, 0x5a, 0xeb, 0x03, 0x21,
        0x71, 0x0d, 0x26, 0xe6, 0xa6, 0xa9, 0x5f, 0x55, 0xcf, 0xdb, 0x16, 0x7c, 0xa5, 0x81, 0x26,
        0xc8, 0x47, 0x03, 0xcd, 0x31, 0xb8, 0x43, 0x9f, 0x56, 0xa5, 0x11, 0x1a, 0x2f, 0xf2, 0x01,
        0x61, 0xae, 0xd9, 0x21, 0x5a, 0x63, 0xe5, 0x05, 0xf2, 0x70, 0xc9, 0x8c, 0xf2, 0xfe, 0xbe,
        0x64, 0x11, 0x66, 0xc4, 0x7b, 0x95, 0x70, 0x36, 0x61, 0xcb, 0x0e, 0xd0, 0x4f, 0x55, 0x5a,
        0x7c, 0xb8, 0xc8, 0x32, 0xcf, 0x1c, 0x8a, 0xe8, 0x3e, 0x8c, 0x14, 0x26, 0x3a, 0xae, 0x22,
        0x79, 0x0c, 0x94, 0xe4, 0x09, 0xc5, 0xa2, 0x24, 0xf9, 0x41, 0x18, 0xc2, 0x65, 0x04, 0xe7,
        0x26, 0x35, 0xf5, 0x16, 0x3b, 0xa1, 0x30, 0x7f, 0xe9, 0x44, 0xf6, 0x75, 0x49, 0xa2, 0xec,
        0x5c, 0x7b, 0xff, 0xf1, 0xea
    };

    /* Test vector 2: Pattern input for Keccak-f[1600] */
    static const unsigned char keccak_f_1600_pattern_input[200] = {
        0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
        0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
        0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
        0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
        0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
        0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
        0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
        0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
        0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
        0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
        0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
        0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
        0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3, 0xa3,
        0xa3, 0xa3, 0xa3, 0xa3, 0xa3
    };
    static const unsigned char keccak_f_1600_pattern_expected[200] = {
        0xcc, 0x44, 0x83, 0xfe, 0xb5, 0x5e, 0x43, 0xf4, 0x6d, 0x88, 0x81, 0xbd, 0x35, 0x8e, 0xbf,
        0x75, 0x9a, 0x06, 0xe7, 0xcd, 0x81, 0xf5, 0xca, 0x48, 0xbf, 0xb7, 0x1b, 0x19, 0xc8, 0x02,
        0x61, 0x45, 0x00, 0x0a, 0x17, 0x39, 0xb1, 0x18, 0x89, 0xc7, 0x3d, 0x53, 0xa9, 0x78, 0x17,
        0xd2, 0x82, 0x4e, 0x52, 0xe6, 0x76, 0xbd, 0xe5, 0xce, 0xee, 0x9a, 0x86, 0x6e, 0x8b, 0x4c,
        0xca, 0x8c, 0xf2, 0x99, 0xd8, 0x18, 0x53, 0xa0, 0x06, 0x15, 0x02, 0xff, 0x1a, 0x7b, 0x11,
        0x82, 0x7c, 0x96, 0x7d, 0x8c, 0xf2, 0xc6, 0x21, 0xa6, 0x24, 0xda, 0x96, 0x75, 0xd2, 0xab,
        0xc9, 0x30, 0x89, 0x22, 0x18, 0x14, 0x7e, 0xa9, 0x07, 0xa6, 0xf0, 0x88, 0x7a, 0x86, 0x6c,
        0x7b, 0x79, 0x89, 0xe0, 0x6c, 0xcf, 0x82, 0x66, 0x62, 0x63, 0x79, 0x98, 0x37, 0x44, 0xce,
        0x1d, 0xe1, 0xb1, 0xb5, 0x12, 0xca, 0x63, 0x6f, 0x25, 0x47, 0x14, 0x1d, 0xef, 0x48, 0x47,
        0xac, 0x64, 0x6a, 0x76, 0xbc, 0x25, 0xbc, 0xed, 0x98, 0xb7, 0x34, 0xfd, 0xd6, 0x65, 0x15,
        0x4d, 0x85, 0xe8, 0x1e, 0x65, 0x08, 0x09, 0x28, 0x19, 0x39, 0x04, 0xf7, 0x6f, 0xec, 0x96,
        0x27, 0xa2, 0x23, 0x3b, 0x2f, 0x75, 0x02, 0x09, 0x25, 0x0d, 0x46, 0xb9, 0x77, 0x65, 0xd0,
        0x19, 0xa5, 0x5a, 0x97, 0xc8, 0xf9, 0x81, 0x4b, 0xfd, 0xb5, 0x38, 0xb9, 0xbc, 0x68, 0x55,
        0xac, 0x35, 0x1b, 0xf1, 0xe4
    };

    /* Test vectors for Keccak-p[1600,12] (12 rounds, TurboSHAKE) */
    /* Test vector 3: All-zero input for Keccak-p[1600,12] */
    static const unsigned char keccak_p_12_zero_expected[200] = {
        0x17, 0x86, 0xa7, 0xb9, 0x38, 0x54, 0x5e, 0x8e, 0x1e, 0xd0, 0x59, 0xf2, 0x50, 0x6a, 0xcd,
        0xd9, 0x35, 0x1f, 0xa9, 0x52, 0xc6, 0xe7, 0xb8, 0x87, 0xc5, 0xe0, 0xe4, 0xcd, 0x67, 0xe0,
        0x93, 0x10, 0x45, 0x5a, 0xd9, 0xf2, 0x90, 0xab, 0x33, 0xb0, 0x45, 0x1a, 0xdd, 0xa8, 0x72,
        0x2f, 0xa7, 0xe0, 0x9c, 0x2f, 0x67, 0x14, 0xaa, 0x80, 0x37, 0xc5, 0x1d, 0x07, 0x51, 0x00,
        0xf5, 0x47, 0xdd, 0x3e, 0xcc, 0x8a, 0x17, 0x0c, 0x31, 0x1d, 0xa3, 0xb3, 0xa0, 0xaa, 0x57,
        0x92, 0xa5, 0x86, 0xb5, 0x79, 0x9b, 0xf9, 0xb1, 0xb3, 0x3d, 0x7c, 0x4a, 0xbc, 0x93, 0x67,
        0x8a, 0xe6, 0x63, 0x40, 0x87, 0x68, 0x66, 0x25, 0x0e, 0x2e, 0x33, 0x03, 0x6c, 0x5c, 0xda,
        0x30, 0xf0, 0xb9, 0x02, 0x12, 0xaa, 0x9c, 0x9f, 0x7a, 0xcf, 0x2b, 0x78, 0x9a, 0x3b, 0x5f,
        0x23, 0x79, 0xae, 0x61, 0xe0, 0xc1, 0x36, 0xe5, 0xec, 0x87, 0x3c, 0xb7, 0x18, 0xb6, 0xe9,
        0x6d, 0xc2, 0x8a, 0x91, 0x70, 0xf1, 0xd1, 0xbe, 0x2a, 0xb7, 0x24, 0xed, 0xda, 0x53, 0xbd,
        0xab, 0x6a, 0x5a, 0xe1, 0x2e, 0x2c, 0x6a, 0x41, 0xc1, 0xbf, 0xaf, 0x52, 0x09, 0xb9, 0x36,
        0xe0, 0xcf, 0xc6, 0xd7, 0x60, 0x70, 0xdc, 0x17, 0x36, 0x50, 0x45, 0xe4, 0x7a, 0x9f, 0xc2,
        0xb2, 0x11, 0x56, 0x62, 0x7a, 0x64, 0x30, 0x2c, 0xdb, 0x71, 0x36, 0xd4, 0x1c, 0xa0, 0x2c,
        0x22, 0x76, 0x0d, 0xfd, 0xcf
    };

    printf("=== Keccak-1600 Core Function Tests ===\n\n");

    /* Test 1: Init function */
    printf("Test 1: crypto_core_keccak1600_init\n");
    memset(state.opaque, 0xFF, 200);
    crypto_core_keccak1600_init(&state);
    for (i = 0; i < 200; i++) {
        if (state.opaque[i] != 0) {
            printf("  FAIL: State not zeroed at byte %u\n", (unsigned int) i);
            test_failures++;
            break;
        }
    }
    if (i == 200) {
        printf("  PASS: State initialized to zeros\n\n");
    }

    /* Test 2: XOR and extract functions */
    printf("Test 2: crypto_core_keccak1600_xor_bytes and extract_bytes\n");
    crypto_core_keccak1600_init(&state);
    for (i = 0; i < sizeof test_data; i++) {
        test_data[i] = (unsigned char) i;
    }

    /* Edge case: zero-length XOR/extract should be a no-op (test at offset 0 and at end) */
    {
        static const size_t offsets[] = { 0, 200 };
        size_t              t;
        for (t = 0; t < sizeof offsets / sizeof offsets[0]; t++) {
            crypto_core_keccak1600_init(&state);
            memset(extracted, 0xA5, sizeof extracted);
            memset(sentinel, 0xA5, sizeof sentinel);
            crypto_core_keccak1600_xor_bytes(&state, test_data, offsets[t], 0);
            crypto_core_keccak1600_extract_bytes(&state, extracted, offsets[t], 0);
            for (i = 0; i < 200; i++) {
                if (state.opaque[i] != 0) {
                    printf("  FAIL: XOR length 0 at offset %u modified state\n\n",
                           (unsigned) offsets[t]);
                    test_failures++;
                    break;
                }
            }
            if (memcmp(extracted, sentinel, sizeof extracted) != 0) {
                printf("  FAIL: extract length 0 at offset %u modified output\n\n",
                       (unsigned) offsets[t]);
                test_failures++;
            }
        }
    }

    /* Edge case: XOR/extract last byte */
    crypto_core_keccak1600_init(&state);
    memset(extracted, 0, sizeof extracted);
    last_byte = 0x5A;
    crypto_core_keccak1600_xor_bytes(&state, &last_byte, 199, 1);
    crypto_core_keccak1600_extract_bytes(&state, extracted, 199, 1);
    if (extracted[0] != last_byte) {
        printf("  FAIL: XOR/extract last byte mismatch\n\n");
        test_failures++;
    }

    crypto_core_keccak1600_init(&state);
    crypto_core_keccak1600_xor_bytes(&state, test_data, 0, sizeof test_data);
    crypto_core_keccak1600_extract_bytes(&state, extracted, 0, sizeof test_data);
    if (memcmp(extracted, test_data, sizeof test_data) == 0) {
        printf("  PASS: XOR and extract work correctly\n\n");
    } else {
        printf("  FAIL: XOR/extract mismatch\n\n");
        test_failures++;
    }

    /* Test 3: Keccak-f[1600] with all-zero input (24 rounds) */
    printf("Test 3: Keccak-f[1600] (24 rounds) - Zero input\n");
    memcpy(state.opaque, keccak_f_1600_zero_input, 200);
    crypto_core_keccak1600_permute_24(&state);
    test_failures +=
        compare_states("  Keccak-f[1600] zero", state.opaque, keccak_f_1600_zero_expected, 200);
    printf("\n");

    /* Test 4: Keccak-f[1600] with pattern input (24 rounds) */
    printf("Test 4: Keccak-f[1600] (24 rounds) - Pattern 0xa3 input\n");
    memcpy(state.opaque, keccak_f_1600_pattern_input, 200);
    crypto_core_keccak1600_permute_24(&state);
    test_failures +=
        compare_states("  Keccak-f[1600] pattern", state.opaque, keccak_f_1600_pattern_expected, 200);
    printf("\n");

    /* Test 5: Keccak-p[1600,12] with all-zero input (12 rounds) */
    printf("Test 5: Keccak-p[1600,12] (12 rounds) - Zero input\n");
    crypto_core_keccak1600_init(&state);
    crypto_core_keccak1600_permute_12(&state);
    test_failures +=
        compare_states("  Keccak-p[1600,12] zero", state.opaque, keccak_p_12_zero_expected, 200);
    printf("\n");

    /* Test 6: Verify 12 and 24 rounds produce different outputs */
    printf("Test 6: Verify 12-round and 24-round differ\n");
    crypto_core_keccak1600_init(&state_12);
    crypto_core_keccak1600_init(&state_24);
    crypto_core_keccak1600_permute_12(&state_12);
    crypto_core_keccak1600_permute_24(&state_24);

    differs = 0;
    for (i = 0; i < 200; i++) {
        if (state_12.opaque[i] != state_24.opaque[i]) {
            differs = 1;
            break;
        }
    }
    if (differs) {
        printf("  PASS: 12-round and 24-round produce different outputs\n");
        printf("    First difference at byte %u: 12-round=0x%02x, 24-round=0x%02x\n\n",
               (unsigned int) i, state_12.opaque[i], state_24.opaque[i]);
    } else {
        printf("  FAIL: 12-round and 24-round produce identical outputs\n\n");
        test_failures++;
    }

    /* Test 7: Multiple permutations */
    printf("Test 7: Double permutation consistency\n");
    crypto_core_keccak1600_init(&state);
    crypto_core_keccak1600_permute_24(&state);
    memcpy(state_24.opaque, state.opaque, 200);
    crypto_core_keccak1600_permute_24(&state);
    printf("  After 24+24 rounds: ");
    print_hex("", state.opaque, 32);

    crypto_core_keccak1600_init(&state);
    crypto_core_keccak1600_permute_12(&state);
    memcpy(state_12.opaque, state.opaque, 200);
    crypto_core_keccak1600_permute_12(&state);
    printf("  After 12+12 rounds: ");
    print_hex("", state.opaque, 32);
    printf("\n");

    /* Final summary */
    printf("=== Test Summary ===\n");
    if (test_failures == 0) {
        printf("All tests PASSED!\n");
        return 0;
    } else {
        printf("FAILED: %d test(s) failed\n", test_failures);
        return 1;
    }
}
