package Ekahau::Response::Error;
use base 'Ekahau::Response'; our $VERSION=Ekahau::Response::VERSION;
use base 'Exporter';

# Written by Scott Gifford <gifford@umich.edu>
# Copyright (C) 2004 The Regents of the University of Michigan.
# See the file LICENSE included with the distribution for license
# information.

use strict;
use warnings;

=head1 NAME

Ekahau::Response::Error - An error response from Ekahau

=head1 SYNOPSIS

Contains information about an error returned by the Ekahau Positioning
Engine.

=head1 DESCRIPTION

=head2 Constructor

Generally you will not want to construct these objects yourself; they
are created by L<Ekahau::Response|Ekahau::Response>, and use its constructor.

=head2 Methods

=cut


use constant EKAHAU_ERR_WARNING => 1;
use constant EKAHAU_ERR_ERROR   => 2;
use constant EKAHAU_ERR_FATAL   => 3;

# Error codes
# Generated by:
# perl -MEkahau::Response::Error -e 'while(my($k,$v) = each(%Ekahau::Response::Error::codenames)) { print "use constant EKAHAU_ERR_$v->[0] => $k;\n"; }'
use constant EKAHAU_ERR_DEVICE_NOT_TRACKED => -600;
use constant EKAHAU_ERR_HARDWARE_FAILURE => -6;
use constant EKAHAU_ERR_ORDER_PRODUCT_TIMEOUT => -304;
use constant EKAHAU_ERR_AUTHENTICATION_FAILED => 2;
use constant EKAHAU_ERR_CONSUMER_TASK_NOT_STARTED => -302;
use constant EKAHAU_ERR_MALFORMED_REQUEST => 1;
use constant EKAHAU_ERR_DATABASE_VERSION_CLASH => -602;
use constant EKAHAU_ERR_UNKNOWN_PRODUCT => -301;
use constant EKAHAU_ERR_ORDER_REPLY_TIMEOUT => -303;
use constant EKAHAU_ERR_UDP_SEND_ERROR => -3;
use constant EKAHAU_ERR_AUTH_TIMEOUT => 6;
use constant EKAHAU_ERR_CLIENT_DEVICE_REMOVED => -5;
use constant EKAHAU_ERR_UNSUPPORTED_PROTOCOL => 3;
use constant EKAHAU_ERR_PRODUCTION_FAILED => -305;
use constant EKAHAU_ERR_CLIENT_CANNOT_MEASURE_DATA => -4;
use constant EKAHAU_ERR_CONTEXT_NOT_FOUND => -1001;
use constant EKAHAU_ERR_LICENSE_VIOLATION => 4;
use constant EKAHAU_ERR_PRODUCER_NOT_RESPONDING => -307;
use constant EKAHAU_ERR_DEVICE_LIMIT_EXCEEDED => -2;
use constant EKAHAU_ERR_DEVICE_NOT_FOUND => -601;
use constant EKAHAU_ERR_AGENT_LOOKUP_FAILED => -300;
use constant EKAHAU_ERR_EMPTY_MODEL => -306;
use constant EKAHAU_ERR_ACCESS_DENIED => 5;
use constant EKAHAU_ERR_CONSUMER_NOT_RESPONDING => -308;
use constant EKAHAU_ERR_DEVICE_NOT_RESPONDING => -1;

# Many of these codes generated by
# perl -MEkahau::Response::Error -e 'while(my($k,$v) = each(%Ekahau::Response::Error::codenames)) { print "$v->[0] "; }'

our %EXPORT_TAGS = (codes => [qw(

EKAHAU_ERR_WARNING EKAHAU_ERR_ERROR EKAHAU_ERR_FATAL

DEVICE_NOT_TRACKED ERR_HARDWARE_FAILURE ORDER_PRODUCT_TIMEOUT
EKAHAU_ERR_AUTHENTICATION_FAILED CONSUMER_TASK_NOT_STARTED
EKAHAU_ERR_MALFORMED_REQUEST DATABASE_VERSION_CLASH UNKNOWN_PRODUCT
ORDER_REPLY_TIMEOUT UDP_SEND_ERROR EKAHAU_ERR_AUTH_TIMEOUT
CLIENT_DEVICE_REMOVED EKAHAU_ERR_UNSUPPORTED_PROTOCOL
PRODUCTION_FAILED CLIENT_CANNOT_MEASURE_DATA
EKAHAU_ERR_CONTEXT_NOT_FOUND EKAHAU_ERR_LICENSE_VIOLATION
PRODUCER_NOT_RESPONDING DEVICE_LIMIT_EXCEEDED DEVICE_NOT_FOUND
AGENT_LOOKUP_FAILED EMPTY_MODEL EKAHAU_ERR_ACCESS_DENIED
CONSUMER_NOT_RESPONDING DEVICE_NOT_RESPONDING

)]);

our @EXPORT_OK = (@{$EXPORT_TAGS{codes}});

use constant CODENAME_NAME => 0;
use constant CODENAME_DESCRIPTION => 1;
use constant CODENAME_EXPLANATION => 2;

use constant DEFAULT_ERROR_NAME => 'UNKNOWN_ERROR';
use constant DEFAULT_ERROR_DESCRIPTION => 'Unknown Error';
use constant DEFAULT_ERROR_EXPLANATION => 'No explanation available for this error.';
use constant DEFAULT_ERROR_INFO => [DEFAULT_ERROR_NAME,DEFAULT_ERROR_DESCRIPTION,DEFAULT_ERROR_EXPLANATION];

# This is the canonical location for this information.
our %codenames = (
		  -600 => ['DEVICE_NOT_TRACKED','Device Not Tracked','The device is not being tracked'],
		  -601 => ['DEVICE_NOT_FOUND','Device Not Found','The device has not been detected by the Positioning Engine'],
		  -602 => ['DATABASE_VERSION_CLASH','Database Version Clash','Internal error, the system could not update the model database.  Requires restarting the software'],
		  -303 => ['ORDER_REPLY_TIMEOUT','Order Reply Timeout','The device which you are trying to track might not be responding.  Check that the device is still connected to the network and properly running Ekahau Client.  Usually you get a EKAHAU_ERR_DEVICE_NOT_RESPONDING error first.'],
		  -304 => ['ORDER_PRODUCT_TIMEOUT','Order Product Timeout','The device which you are trying to track might not be responding.  Check that the device is still connected to the network and properly running Ekahau Client.  Usually you get -1 error first.'],
		  -306 => ['EMPTY_MODEL','Empty Model','Create a positioning model with Ekahau Manager and save it to the Positioning Engine'],
		  -307 => ['PRODUCER_NOT_RESPONDING','Producer Not Responding','The device which you are trying to track might not be responding.  Check that the device is still connected to the network and properly running Ekahau Client.  Usually you get -1 error first.'],
		  -308 => ['CONSUMER_NOT_RESPONDING','Consumer Not Responding','The device which you are trying to track might not be responding.  Check that the device is still connected to the network and properly running Ekahau Client.  Usually you get -1 error first.'],
		  -300 => ['AGENT_LOOKUP_FAILED','Agent Lookup Failed','If this occurs, contact Ekahau support.'],
		  -301 => ['UNKNOWN_PRODUCT','Unknown Product','If this occurs, contact Ekahau support.'],
		  -302 => ['CONSUMER_TASK_NOT_STARTED','Consumer Task Not Started','If this occurs, contact Ekahau support.'],
		  -305 => ['PRODUCTION_FAILED','Production Failed','If this occurs, contact Ekahau support.'],
		  -1   => ['DEVICE_NOT_RESPONDING','Device Not Responding','No data (any UDP packets) has been received from the device'],
		  -2   => ['DEVICE_LIMIT_EXCEEDED','Device Limit Exceeded','Device limit (how many devices your license allows you to track) is exceeded.'],
		  -3   => ['UDP_SEND_ERROR','UDP Send Error','Could not send UDP packet'],
		  -4   => ['CLIENT_CANNOT_MEASURE_DATA','Client Cannot Measure Data','Ekahau Client can not get signal information from the network adapter.'],
		  -5   => ['CLIENT_DEVICE_REMOVED','Client Device Removed','Network adapter has been removed from the client device.'],
		  -6   => ['HARDWARE_FAILURE','Hardware Failure','Network adapter has an internal error.'],
 	         -1001 => ['CONTEXT_NOT_FOUND','Location Context Not Found','No location context exists for the requested context ID'],
		  1    => ['MALFORMED_REQUEST','Malformed Request','The Ekahau engine expected HELLO and TALK commands but received something else.'],
		  2    => ['AUTHENTICATION_FAILED','Authentication Failed','Password or license is wrong.'],
		  3    => ['UNSUPPORTED_PROTOCOL','Unsupported protocol','The requested protocol was not found.'],
		  4    => ['LICENSE_VIOLATION','License Violation','Too many open sockets for your license.'],
		  5    => ['ACCESS_DENIED','Access Denied','Authentication was OK, but the connect was refused either because your IP address is not allowed to connect, or because the server is too busy.'],
		  6    => ['AUTH_TIMEOUT','Authentication timeout','HELLO and TALK commands were not sent quickly enough'],
		  );

# Internal method
sub init
{
    my $self = shift;

    warn "Created Ekahau::Response::Error object\n"
	if ($ENV{VERBOSE});

    if ($self->{cmd} eq 'CONTEXT_NOT_FOUND')
    {
	# This isn't really an error, but we treat it as one.
	$self->{params}{errorCode} = EKAHAU_ERR_CONTEXT_NOT_FOUND;
	$self->{params}{errorLevel} = EKAHAU_ERR_ERROR;
    }
    elsif ($self->{cmd} eq 'FAILURE')
    {
	# Authentication Failure
	$self->{params}{errorCode} = $self->{args}[0];
	$self->{params}{errorLevel} = EKAHAU_ERR_FATAL;
    }
}

=head3 error ( )

Always returns true, to indicate this is an error object.  This method
can be used on an arbitrary L<Ekahau::Response|Ekahau::Response> object to find out if
it represents an error or not.

=cut

sub error
{
    1;
}

=head3 error_level ( )

Returns an integer representing the severity of the error.  Errors are
ordered from least to most severe, so you can say things like C<if
($err->error_level >= EKAHAU_ERR_ERROR).  Constants are used to refer
to the various levels, which you can refer to as
C<Ekahau::Response::Error::EKAHAU_ERR_CODE>.  You can also import
these symbols into your program using the C<:codes> tag, which allows
them to be used without the C<Ekahau::Response::Error> prefix.  They
can be imported like this:

   use Ekahau::Response::Error qw(:codes)

These constants represent the different error levels:

=over 4

=item EKAHAU_ERR_WARNING

A small problem that should not affect tracking or other operations on
the server.

=item EKAHAU_ERR_ERROR

An error that may affect tracking, but may still be recoverable.

=item EHAKAU_ERR_FATAL

A fatal error requires restarting Ekahau.

=back

=cut

sub error_level
{
    my $self = shift;
    $self->{params}{errorLevel};
}

=head3 error_code ( )

Returns the code for this error.  See L<Errors|/Errors> for a list of errors.

=cut

sub error_code
{
    my $self = shift;
    $self->{params}{errorCode};
}

=head3 error_msg ( )

Returns the exact error response from the Ekahau engine.

=cut 

sub error_msg
{
    my $self = shift;
    $self->{cmd};
}

=head3 error_info ( )

Returns a triple containing the error name, description, and
explanation for this error.

=cut

sub error_info
{
    my $self = shift;
    return $codenames{($self->error_code)} || DEFAULT_ERROR_INFO;
}

=head3 error_name ( )

Returns the name of this error.

=cut

sub error_name
{
    my $self = shift;
    return $self->error_info->[CODENAME_NAME];
}

=head3 error_description

Returns a brief description of the error

=cut 

sub error_description
{
    my $self = shift;
    return $self->error_info->[CODENAME_DESCRIPTION];
}

=head3 error_explanation

Returns a longer explanation for this error.

=cut

sub error_explanation
{
    my $self = shift;
    return $self->error_info->[CODENAME_EXPLANATION];
}

# Generated by
# perl -MEkahau::Response::Error -e 'while(my($k,$v) = each(%Ekahau::Response::Error::codenames)) { print "=head3 EKAHAU_ERR_$v->[0]\n\n$v->[1].  $v->[2]\n\n"; }'

=head2 Errors

=head3 EKAHAU_ERR_DEVICE_NOT_TRACKED

Device Not Tracked.  The device is not being tracked

=head3 EKAHAU_ERR_HARDWARE_FAILURE

Hardware Failure.  Network adapter has an internal error.

=head3 EKAHAU_ERR_ORDER_PRODUCT_TIMEOUT

Order Product Timeout.  The device which you are trying to track might not be responding.  Check that the device is still connected to the network and properly running Ekahau Client.  Usually you get -1 error first.

=head3 EKAHAU_ERR_AUTHENTICATION_FAILED

Authentication Failed.  Password or license is wrong.

=head3 EKAHAU_ERR_CONSUMER_TASK_NOT_STARTED

Consumer Task Not Started.  If this occurs, contact Ekahau support.

=head3 EKAHAU_ERR_MALFORMED_REQUEST

Malformed Request.  The Ekahau engine expected HELLO and TALK commands but received something else.

=head3 EKAHAU_ERR_DATABASE_VERSION_CLASH

Database Version Clash.  Internal error, the system could not update the model database.  Requires restarting the software

=head3 EKAHAU_ERR_UNKNOWN_PRODUCT

Unknown Product.  If this occurs, contact Ekahau support.

=head3 EKAHAU_ERR_ORDER_REPLY_TIMEOUT

Order Reply Timeout.  The device which you are trying to track might not be responding.  Check that the device is still connected to the network and properly running Ekahau Client.  Usually you get a EKAHAU_ERR_DEVICE_NOT_RESPONDING error first.

=head3 EKAHAU_ERR_UDP_SEND_ERROR

UDP Send Error.  Could not send UDP packet

=head3 EKAHAU_ERR_AUTH_TIMEOUT

Authentication timeout.  HELLO and TALK commands were not sent quickly enough

=head3 EKAHAU_ERR_CLIENT_DEVICE_REMOVED

Client Device Removed.  Network adapter has been removed from the client device.

=head3 EKAHAU_ERR_UNSUPPORTED_PROTOCOL

Unsupported protocol.  The requested protocol was not found.

=head3 EKAHAU_ERR_PRODUCTION_FAILED

Production Failed.  If this occurs, contact Ekahau support.

=head3 EKAHAU_ERR_CLIENT_CANNOT_MEASURE_DATA

Client Cannot Measure Data.  Ekahau Client can not get signal information from the network adapter.

=head3 EKAHAU_ERR_CONTEXT_NOT_FOUND

Location Context Not Found.  No location context exists for the requested context ID

=head3 EKAHAU_ERR_LICENSE_VIOLATION

License Violation.  Too many open sockets for your license.

=head3 EKAHAU_ERR_PRODUCER_NOT_RESPONDING

Producer Not Responding.  The device which you are trying to track
might not be responding. Check that the device is still connected to
the network and properly running Ekahau Client.  Usually you get -1
error first.

=head3 EKAHAU_ERR_DEVICE_LIMIT_EXCEEDED

Device Limit Exceeded.  Device limit (how many devices your license
allows you to track) is exceeded.

=head3 EKAHAU_ERR_DEVICE_NOT_FOUND

Device Not Found.  The device has not been detected by the Positioning Engine

=head3 EKAHAU_ERR_AGENT_LOOKUP_FAILED

Agent Lookup Failed.  If this occurs, contact Ekahau support.

=head3 EKAHAU_ERR_EMPTY_MODEL

Empty Model.  Create a positioning model with Ekahau Manager and save
it to the Positioning Engine

=head3 EKAHAU_ERR_ACCESS_DENIED

Access Denied.  Authentication was OK, but the connect was refused
either because your IP address is not allowed to connect, or because
the server is too busy.

=head3 EKAHAU_ERR_CONSUMER_NOT_RESPONDING

Consumer Not Responding.  The device which you are trying to track
might not be responding.  Check that the device is still connected to
the network and properly running Ekahau Client.  Usually you get -1
error first.

=head3 EKAHAU_ERR_DEVICE_NOT_RESPONDING

Device Not Responding.  No data (any UDP packets) has been received from the device

=head1 AUTHOR

Scott Gifford E<lt>gifford@umich.eduE<gt>, E<lt>sgifford@suspectclass.comE<gt>

Copyright (C) 2005 The Regents of the University of Michigan.

See the file LICENSE included with the distribution for license
information.


=head1 SEE ALSO

L<Ekahau::Response|Ekahau::Response>, L<Ekahau::Base|Ekahau::Base>.

=cut

1;
